// x86 byte order required. To compile under MacOS, skip byte reordering.

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "main.h"

int main( int argc, char* argv[])
{
	printf("ICC Profile Reader 1.00\n1999 by Joern Loviscach\n");

	if (2 != argc)
	{
		printf("Call this function with the file path of\n");
		printf("an ICC color management profile to examine.\n");
		printf("Use \"| more\" or \"> data.txt\".\n");

		return ERROR;
	}

	FILE* file;
	if (0 == (file = fopen(argv[1], "r+b")))
	{
		printf("Error when opening file.\n");
		return ERROR;
	}

	unsigned long result;

	if(readFourBytes(file, &result))
	{
		printf("Error when reading profile size.\n");
		return ERROR;
	}
	printf("Profile size: %d\n", result);

	unsigned char* profile = (unsigned char*)malloc(result);
	
	if(0 == profile)
	{
		printf("Memory allocation error.\n");
		return ERROR;
	}

	if(result-4 != fread(profile, 1, result-4, file))
	{
		printf("Error when reading profile.\n");
		return ERROR;
	}

	fclose(file);

	examineHeader(profile);
	
	examineTags(profile);

	free(profile);
	printf("Run successfully.\n");
	return OK;
}

void examineHeader(unsigned char* profile)
{
	unsigned long result;

	printf("CMM Type signature: ");
	printString(getFourBytes(profile, 0));

	printf("Profile version number: %d.%d.%d\n",
		   (profile[4]&0x0F)+10*((profile[4]&0xF0)>>4),
		   (profile[5]>>4)&0x0F, (profile[5])&0x0F);

	printf("Profile/Device Class: ");
	printString(getFourBytes(profile, 8));
	printf("(scnr = input, mntr = display, output = prtr)\n");

	printf("Color space: ");
	printString(getFourBytes(profile, 12));

	printf("Profile Connection Space: ");
	printString(getFourBytes(profile, 16));

	printf("Date of first creation: %d-%02d-%02d, %02d:%02d:%02d\n",
		   getTwoBytes(profile, 20),
		   getTwoBytes(profile, 22),
		   getTwoBytes(profile, 24),
		   getTwoBytes(profile, 26),
		   getTwoBytes(profile, 28),
		   getTwoBytes(profile, 30) );

	printf("acsp signature: ");
	printString(getFourBytes(profile, 32));

	printf("Primary plattform: ");
	printString(getFourBytes(profile, 36));

	result = getFourBytes(profile, 40);
	printf("Embedded: %d\n", result&1);
	printf("Cannot be used independently from embedded: %d\n", (result>>1)&1);

	printf("Device manufacturer: ");
	printString(getFourBytes(profile, 44));

	printf("Device model: ");
	printString(getFourBytes(profile, 48));

	result = getFourBytes(profile, 52);
	printf("Transparency: %d\n", result&1);
	printf("Matte: %d\n", (result>>1)&1);

	printf("Rendering intent: %d\n", getTwoBytes(profile, 60)&0xFF);
	printf("(0 = percpt., 1 = rel.color., 2 = sat., 3 = abs.color.)\n");

	printf("Illuminant of connection space:\n");
	printf("X = %e, Y = %e, Z = %e\n",
		   getSignedFixed(profile, 64),
   		   getSignedFixed(profile, 68),
		   getSignedFixed(profile, 72)   );

	printf("Profile creator: %X\n", getFourBytes(profile, 76));
}

void examineTags(unsigned char* profile)
{
	unsigned long tagCount = getFourBytes(profile, 124);
	printf("Tag count: %d\n", tagCount);

	unsigned long i;
	for (i=0; i<tagCount; i++)
	{
		examineTag(profile, i);
	}
}

void examineTag(unsigned char* profile, unsigned long i)
{
	unsigned long tag = getFourBytes(profile, 128+12*i);
	printf("Tag: ");
	printString(tag);

	unsigned long pointer = getFourBytes(profile, 128+12*i+4);

	unsigned long size = getFourBytes(profile, 128+12*i+8);
	printf("Size: %d\n", size);

	if(    (tag == 'A2B0')
		|| (tag == 'A2B1')
		|| (tag == 'A2B2')
		|| (tag == 'B2A0')
		|| (tag == 'B2A1')
		|| (tag == 'B2A2')
		|| (tag == 'gamt')
		|| (tag == 'pre0')
		|| (tag == 'pre1')
		|| (tag == 'pre2') )
	{
		printf("Precision: %c (1 = 8 bit, 2 = 16 bit)\n", profile[pointer-4+3]);
		printf("Input channels: %d\n", profile[pointer-4+8]);
		printf("Output channels: %d\n", profile[pointer-4+9]);
		printf("Grid points: %d\n", profile[pointer-4+10]);
		printf("Input table entries: %d\n", profile[pointer-4+48]+256*profile[pointer-4+49]);
		printf("Output table entries: %d\n", profile[pointer-4+48]+256*profile[pointer-4+49]);
	}

	if(    (tag == 'bXYZ')
		|| (tag == 'gXYZ')
		|| (tag == 'rXYZ')
		|| (tag == 'lumi')
		|| (tag == 'bkpt')
		|| (tag == 'wtpt')
		|| (tag == '1amt') )
	{
		printf("X = %e, Y = %e, Z = %e\n",
			   getSignedFixed(profile, pointer-4+8),
   			   getSignedFixed(profile, pointer-4+12),
			   getSignedFixed(profile, pointer-4+16)   );
	}

	if(    (tag == 'bTRC')
		|| (tag == 'gTRC')
		|| (tag == 'rTRC') )
	{
		printf("Grid points: %d\n", getFourBytes(profile, pointer-4+8));
	}

	if(    (tag == 'targ')
		|| (tag == 'cprt') )
	{
		printf((char*)(profile+pointer-4+8));
		printf("\n");
	}

	if(    (tag == 'dmnd')
		|| (tag == 'dmdd')
		|| (tag == 'desc')
		|| (tag == 'scrd')
		|| (tag == 'vued') )
	{
		printf((char*)(profile+pointer-4+12));
		printf("\n");
	}

	printf("\n");
}

int readFourBytes(FILE* file, unsigned long* result)
{
	if(1 != fread(result, 4, 1, file))
	{
		printf("Error when reading.\n");
		return ERROR;
	}
	*result =   (((*result)&0xFF000000L)>>24)
			  | (((*result)&0x00FF0000L)>>8)
			  | (((*result)&0x0000FF00L)<<8)
			  | (((*result)&0x000000FFL)<<24);

	return OK;
}

unsigned long getFourBytes(unsigned char* profile, unsigned long index)
{
	return   (((unsigned long)profile[index])<<24)
		   | (((unsigned long)profile[index+1])<<16)
		   | (((unsigned long)profile[index+2])<<8)
		   | (((unsigned long)profile[index+3]));
}

unsigned short getTwoBytes(unsigned char* profile, unsigned long index)
{
	return   (((unsigned short)profile[index])<<8)
		   | (((unsigned short)profile[index+1]));
}

double getSignedFixed(unsigned char* profile, unsigned long index)
{
	return   (   (((unsigned long)profile[index])<<24)
			   | (((unsigned long)profile[index+1])<<16)
		  	   | (((unsigned long)profile[index+2])<<8)
			   | (((unsigned long)profile[index+3])) ) / (double)0x10000L;
}

void printString(unsigned long input)
{
	char string[5];
	string[0] = (char)((input & 0xFF000000L) >> 24);
	string[1] = (char)((input & 0x00FF0000L) >> 16);
	string[2] = (char)((input & 0x0000FF00L) >> 8);
	string[3] = (char)(input & 0x000000FFL);
	string[4] = (char)0;
	int i;
	for(i=0; i<4; i++)
	{
		if( (string[i]<32) || (string[i]>126) )
		{
			string[i] = 32;
		}
	}
	printf(string);
	printf("\n");
}
