unit DXPlayAS;
{ Delphi-Komponenten fr DirectPlay, baut auf den von Erik Unger
  umgesetzten Interface-Deklarationen fr DirectX auf.

  Pro Anwendung: *ein* DPlayer-Objekt (globale Variable DPlayer)
}

interface
uses Windows, SysUtils, Classes, ActiveX, DPLobby, DPlay;
type
  TDPlayer = class;
  TDPMessage = TNotifyEvent;
  TDPModifyConnInfo = function(const ConnInfo:
                            PDPLConnection): Boolean of Object;

  TDPlayer = class(TComponent)
  private
    FID: TDPID;  // Kennziffer
    EvHandle: THandle;
    FOnModifyConnInfo: TDPModifyConnInfo;
    FOnUserMsg, FOnSystemMsg: TDPMessage;
    FConnected: Boolean;
  protected
    procedure Notification(AComponent: TComponent;
        Operation: TOperation); override;
  protected  // von ReceiverThread gesetzt
    FMsgIDFrom, FMsgIDTo: Integer;
    FMsgData: PDPMsg_Generic;
    FMsgDataSize: Integer;
    FMsgPriority: Word;  // 0 = niedrigste Prioritt (Standard)
    FMsgTimeOut: Integer;  // 0 = kein Timeout (Standard)
    ReceiveThread: TThread;
    procedure OnDPMessage;  // synchron
  public
    PlayerName: String;
    DPlayObject: IDirectPlay4;
    DPlayLobbyObject: IDirectPlayLobby3;
    SessionDesc: TDPSessionDesc2;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    // immer zuerst probieren!
    function ConnectUsingLobby: HResult;
    function CreatePlayer(PlayerName: String;
           PData: Pointer; DataSize, Flags: Integer): HResult;
    function Send(Target: TDPID; Flags: DWord;
           Msg: Pointer; MsgSize: Integer): Integer;
  published  // Nachrichenempfang
    property ID: TDPID read FID;
    property MsgIDFrom: Integer read FMsgIDFrom;
    property MsgIDTo: Integer read FMsgIDTo;
    property MsgData: PDPMsg_Generic read FMsgData;
    property MsgDataSize: Integer read FMsgDataSize;
    property MsgPriority: Word read FMsgPriority write FMsgPriority;
    property MsgTimeOut: Integer read FMsgTimeOut write FMsgTimeOut;
    property OnUserMsg: TDPMessage read FOnUserMsg write FOnUserMsg;
    property OnSystemMsg: TDPMessage 
                     read FOnSystemMsg write FOnSystemMsg;
  published
    property Connected: Boolean read FConnected;
    property OnModifyConnInfo: TDPModifyConnInfo
                     read FOnModifyConnInfo write FOnModifyConnInfo;
  end;

// Exception mit ErrorString(Res), wenn nicht DP_OK
function CheckRes(Res: HResult; Msg: String): Boolean;

var DPlayer: TDPlayer;  // der (einzige) Spieler des Programms

implementation
// --------- Empfngerthread ----------------
type
  TDPReceiveThread = class(TThread)
  protected
    Player: TDPlayer;
    procedure Execute; override;
  public
    constructor Create(_Player: TDPlayer);
  end;

constructor TDPReceiveThread.Create(_Player: TDPlayer);
begin
  inherited Create(True);
  FreeOnTerminate := True;
  Player := _Player;
  Resume;
end;

procedure TDPReceiveThread.Execute;
var Res: HResult;
begin
  while not Terminated do
  begin
    if WaitForSingleObject(Player.EvHandle,INFINITE)
      <> WAIT_OBJECT_0 then
    begin
      Terminate; Break;
    end;
    with Player do
      repeat //eine Signalisierung kann fr mehrere Nachrichten stehen
        FMsgIDFrom := 0; FMsgIDTo := 0;
        repeat  // eine Nachricht lesen, ggf. Puffer vergrern
          Res := DPlayObject.Receive(FMsgIDFrom, FMsgIDTo,
            DPRECEIVE_ALL, FMsgData, FMsgDataSize);
          if Res = DPERR_BUFFERTOOSMALL then
          begin
            if FMsgData <> nil then FreeMem(FMsgData);
            GetMem(FMsgData,FMsgDataSize);
          end;
        until Res <> DPERR_BUFFERTOOSMALL;
        // Auswertung
        if SUCCEEDED(Res) and (MsgDataSize >= SizeOf(TDPMsg_Generic))
          then Synchronize(OnDPMessage);
      until FAILED(Res);  // DPERR_NOMESSAGES, d.h. Queue ausgelesen
  end;
end;

// -------------------------------------------------------------

function CheckRes(Res: HResult; Msg: String): Boolean;
begin
  Result := SUCCEEDED(Res);
  if not Result then
   raise Exception.Create(Msg+': '+ErrorString(Res));
end;

constructor TDPlayer.Create(AOwner: TComponent);
begin
  inherited;
  EvHandle := CreateEvent(nil,False,False,nil);
  CoInitialize(nil);
  ReceiveThread := TDPReceiveThread.Create(Self);
end;

procedure TDPlayer.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if (AComponent is TDPlayer) and (AComponent <> Self) and
    (Operation = opInsert) then raise Exception.Create(
      'Nur eine TDPlayer-Komponente pro Anwendung!');
end;

destructor TDPlayer.Destroy;
begin
  CloseHandle(EVHandle);  // beendet den Thread
  if FMsgData <> nil then FreeMem(FMsgData);
  CoUninitialize;
  inherited;
end;

// Diese Routine stammt mehr oder weniger aus dem Beispielprogramm
// CHAT des DirectX-SDKs und zeigt, wie fix es ber eine Lobby gehen kann
function TDPlayer.ConnectUsingLobby: HResult;
var CInfoSize: DWord; CInfo: PDPLConnection; PDesc: PDPSessionDesc2;
begin
  if Connected then raise Exception.Create('TDPlayer: already connected');
  Result := CoCreateInstance(CLSID_DirectPlayLobby, nil,
    CLSCTX_INPROC_SERVER, IID_IDirectPlayLobby3, DPlayLobbyObject);
  if FAILED(Result) then Exit;
  // Verbindungsinformationen ber die Lobby verfgbar?
  Result := DPlayLobbyObject.GetConnectionSettings(0,nil,CInfoSize);
  if Result <> DPERR_BUFFERTOOSMALL then Exit;
  // wesentlich mehr als SizeOf(TDPLConnection), da hngen
  // offensichtlich undokumentierte Felder hintendran
  CInfo := AllocMem(CInfoSize);
  try
    Result := DPlayLobbyObject.GetConnectionSettings(0,CInfo,CInfoSize);
    if FAILED(Result) then Exit;  // kann DPERR_NOTLOBBIED sein
    // Callback fr Modifikationen der Verbindungsinformationen
    if Assigned(FOnModifyConnInfo) then
    begin
      if not OnModifyConnInfo(CInfo) then Result := DPERR_NOTLOBBIED
       else Result := DPlayLobbyObject.SetConnectionSettings(0,0,CInfo^);
      if FAILED(Result) then Exit;  // auch fr DPERR_NOTLOBBIED
    end;
    // Verbindung herstellen, DirectPlay-Objekt anlegen
    Result := DPlayLobbyObject.ConnectEx(0,IID_IDirectPlay4,DPlayObject,nil);
    if FAILED(Result) then Exit;
    // Spieler anlegen
    Result := DPlayObject.CreatePlayer(FID,CInfo.lpPlayerName,EvHandle,nil,0,0);
    if FAILED(Result) then Exit;
    PlayerName := CInfo.lpPlayerName^.lpszShortName;
    // Sitzungsbeschreibung kopieren. Die Daten sollten
    // eigentlich schon in CInfo stehen, bei Join fehlt aber der
    // Sitzungsname (wohl ein Bug)
    // SessionDesc := CInfo.lpSessionDesc^;
    Result := DPlayObject.GetSessionDesc(nil,CInfoSize);
    if Result <> DPERR_BUFFERTOOSMALL then Exit;
    GetMem(PDesc,CInfoSize);
    Result := DPlayObject.GetSessionDesc(PDesc,CInfoSize);
    SessionDesc := PDesc^;  // nur den dokumentierten Teil kopieren
    FreeMem(PDesc);
    FConnected := True;
  finally
    FreeMem(CInfo);
  end;
end;

function TDPlayer.CreatePlayer(PlayerName: String; PData: Pointer; DataSize, Flags: Integer): HResult;
var PName: TDPName; SessionDescSize: DWord; PSessionDesc: PDPSessionDesc2;
begin
  if Connected then raise Exception.Create('TDPlayer: already connected');
  FillChar(PName,SizeOf(PName),0);
  PName.dwSize := SizeOf(PName); PName.lpszShortName := PChar(PlayerName);
  Result := DPlayObject.CreatePlayer(FID, @PName, EvHandle, PData, DataSize, Flags);
  if SUCCEEDED(Result) then
  begin  // Sitzungsbeschreibung und Spielername aktualisieren
    SessionDescSize := 0;
    // Hier sind auch wieder undokumentierte Felder hintendran
    DPlayObject.GetSessionDesc(nil,SessionDescSize);
    GetMem(PSessionDesc,SessionDescSize);
    DPlayObject.GetSessionDesc(PSessionDesc,SessionDescSize);
    // ... nur den dokumentierten Teil kopieren
    SessionDesc := PSessionDesc^;
    FreeMem(PSessionDesc);
    Self.PlayerName := PlayerName;
    FConnected := True;
  end;
end;

function TDPlayer.Send(Target: TDPID; Flags: DWord; Msg: Pointer; MsgSize: Integer): Integer;
var Res: HResult;
begin
  Res := DPlayObject.SendEx(ID,Target,Flags,Msg,MsgSize,
    MsgPriority, MsgTimeOut, Pointer($11223344), @Result);
  if Res <> DPERR_PENDING then CheckRes(Res,'SendEx');
end;

procedure TDPlayer.OnDPMessage;  // per Synchronize aufgerufen
begin
  if (MsgIDFrom = DPID_SYSMSG) then
  begin
    if Assigned(FOnSystemMsg) then OnSystemMsg(Self);
  end else if Assigned(FOnUserMsg) then OnUserMsg(Self);
end;

end.

