#!/bin/sh
#################################################################
#
#	@(#) ddnsupd.sh (c) 2007-2015 by H. Zuleger HZNET
#
#	Try to dynamic update public IP addresses in the DNS
#	Use TSIG or SIG(0) authentication
#
#	This script will not work in environments w/o a plain
#	routed (propably NATed) internet access with unblocked
#	port 53 tcp/udp.
#
#	Option -d is to turn on verbose messaging of nsupdate
#	Option -v prints some additional messages on how the
#	script works.
#	Option -k is for the keyfile, -h for the hostname, 
#	-i for the interface and -t for the ttl time.
#
#	It is recommended to put all the site/host specific configs
#	in the configuration section at the beginning of this script
#
#	Run this script as part of a NetworkManager dispatcher with
#	argument add in case of an ifup event and with argument del
#	in case of a pre-down event.
#	See NetworkManager(8) for details.
#
#################################################################
PATH=/usr/local/bin:/usr/bin:/bin:/sbin


###### default parameter settings
dev="eth0"
keydir="$HOME/keys"	# better use something like /etc/ddnsupd or /var/ddnsupd

#hostname="horst.dn.example.de"		# better to set the hostname here
					# to avoid a hostname(1) run every time
#SERVER="ns1.example.de"	# Determining the master ns out of the domain
				# is a time consuming process; Better to set it
				# here manually

TTL=604800	# 1w
TTL=7200	# 2 hours
###### default parameter settings (end)

PROGNAME=`basename $0`

debug=""	# set to "-d" for debug output
verbose=""	# set to 1 for verbose output


usage()
{
	test -n "$1" && echo "$PROGNAME: $1" 1>&2
	echo "usage: $PROGNAME [-d] [-v] [-h hostname] [-k keyfile] [-i dev] [-t ttl] add|del|show|showall" 1>&2
	exit 1
}


LANG="C"
export LANG

ipv4=""
ipv6=""

while test $# -gt 0
do
	case $1 in
	-h)	shift
		hostname="$1"
		;;
	-k)	shift
		keyfile="$1"
		;;
	-d)	debug=-d
		;;
	-v)	verbose=1
		;;
	-t)	shift
		ttl="$1"
		;;
	-i)	shift
		dev="$1"
		;;
	-*)	usage
		;;
	*)	break
		;;
	esac
	shift
done
cmd="$1"
case "$cmd" in
add|show)
	;;
del*)
	;;
"")	usage "no command given";
	;;
*)	usage "illegal command $cmd";
	;;
esac

# try to get the hostname if not already set
test -z "$hostname" && hostname=`hostname`
if test -z "$hostname"
then
	usage "Couldn't determine hostname" 1>&2
fi

# split of hostname in host and domain part
set -- `echo $hostname | sed 's/\./ /'`
host=$1
domain=$2

# try to get the authoritative name server if not already set
if test -z "$SERVER"
then
	set -- `dig +short SOA $domain`
	SERVER="$1"
	if test -z "$SERVER"
	then
		usage "Unable to get autoritative master for domain $domain" 1>&2
	fi
fi

test "$verbose" && echo "Hostname = \"$hostname\""
test "$verbose" && echo "Host = \"$host\""
test "$verbose" && echo "Domain = \"$domain\""
test "$verbose" && echo "Server = \"$SERVER\""
test "$verbose" && echo "TTL = \"$TTL\""

# just show what is actually in the zone
if test "$cmd" = "show"
then
	dig  +norec +noall +answer $host.$domain a @$SERVER
	dig  +norec +noall +answer $host.$domain aaaa @$SERVER
	exit
fi

# for the rest, we need a key to authenticate us
if test -z "$keyfile"
then
	keyfile="$keydir/$hostname.secret"		# tsig key file ?
	if test ! -f $keyfile
	then
		keyfile="$keydir/K$hostname.+*.private"	# or SIG0 key file
	fi
fi
if test ! -f $keyfile
then
	usage "No keyfile found";
fi
keyfileopt="-k $keyfile"

test "$verbose" && echo "KeyFile = \"$keyfile\""

# this is to remove old entries manually
if test $cmd = "del"
then
	{
		echo "server $SERVER"
		echo "zone $domain"

		echo "update delete $host.$domain IN A"
		echo "update delete $host.$domain IN AAAA"
		echo "send"

	} | nsupdate $debug $keyfileopt
	exit
elif test $cmd != "add"
then
	usage "illegal command $cmd"
fi

# now take a look at the current ip address on interface $dev
if test -z "$dev"
then
	echo "Missing interface identifier" 1>&2
	exit 1
fi

ipv4=`ip addr show dev $dev |
sed  -n "/inet /s/^[ 	]*inet \([0-9][0-9.]*\).*/\1/p"`

ipv6list=`ip -6 addr show dev $dev scope global | grep inet6 | grep -v temporary |
sed  -n "/inet6 /s/^[ 	]*inet6 \([23][0-9a-f:]*\)\/.*/\1/p"`

if test -z "$ipv6list"	# wait for SLAAC
then
	echo "sending RS message at `date`"
	/usr/local/bin/rdisc6 -1 -r 5 -q $dev

	ipv6list=`ip -6 addr show dev $dev scope global | grep inet6 | grep -v temporary |
	sed  -n "/inet6 /s/^[ 	]*inet6 \([23][0-9a-f:]*\)\/.*/\1/p"`
	echo "done at `date`"
	ip -6 addr show dev $dev
fi

test "$verbose" && echo "Raw IPv4: $ipv4" 1>&2
test "$verbose" && echo "Raw IPv6: $ipv6list" 1>&2

checkip4=`echo "0$ipv4" | tr -d "0-9."`
if test -n "$checkip4"
then
	usage "error: parsing of ip address on interface $dev failed: $ipv4"
fi

# remove private and other non global addresses
case $ipv4 in
	100.6[456789].*|100.[789][0123456789].*|100.1[0123456789].*|100.12[01234567].*)	# CGNAT (RFC6598) 100.64.0.0/10
		ipv4=""
	;;
	196.254.*)	# link local
		ipv4=""
	;;

	10.*|172.1[6789].*|172.2[0123456789].*|172.30.*|172.31*|192.168.*)	# rfc 1918
		ipv4=""
	;;
	127.*10.*|192.0.0.*|192.0.2.*|198.51.100.*|203.0.113.*)	# localhost & special pupose or document addresses
		ipv4=""
	;;
esac
# same for ipv6
ipv6l=""
for ipv6 in "$ipv6list"
do
	case $ipv6 in
		2001:0[dD][bB]8:*|2001:[dD][bB]8:*)
		;;
		2???:*)
			ipv6l="$ipv6l $ipv6"
		;;
	esac
done
ipv6l=`echo "$ipv6l" | sed "s/^ *//"`

test "$verbose" && echo "Public IPv4: \"$ipv4\""  1>&2
test "$verbose" && echo "Public IPv6: \"$ipv6l\"" 1>&2

if test -z "$ipv4" -a -z "$ipv6l"
then
	test $verbose && echo "No public ip address found" 1>&2 
	exit 1
fi

# now test if the ip addresses are already in the DNS or not
dnsipv4=`dig +short A $host.$domain @$SERVER`
test "$ipv4" = "$dnsipv4" && noipv4=1

# !!!FIX IT:  The following will not work with multiple ipv6 addresses !!!
dnsipv6=`dig +short AAAA $host.$domain @$SERVER`
test "$ipv6l" = "$dnsipv6" && noipv6l=1

test "$verbose" && echo "Published IPv4: \"$dnsipv4\""  1>&2
test "$verbose" && echo "Published IPv6: \"$dnsipv6\"" 1>&2

if test "$noipv4" -a "$noipv6l"
then
	test $verbose && echo "No update needed " 1>&2 
	exit 1
fi

{
	echo "server $SERVER"
	echo "zone $domain"
	test -n "$key" && echo "key $algo:$kname $key"

	# at first delete old entrys
	echo "update delete $host.$domain IN A"
	echo "update delete $host.$domain IN AAAA"
	echo "send"

	# now add the new ones
	test -n "$ipv4" && echo "update add $host.$domain $TTL IN A $ipv4"

	for ipv6 in $ipv6l
	do
		test -n "$ipv6" && echo "update add $host.$domain $TTL IN AAAA $ipv6"
	done

	test "$verbose" && echo "show"

	echo "send"
	#	echo "answer"

} | nsupdate $debug $keyfileopt
