package com.linkesoft.midiintervals;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.pm.PackageManager;
import android.media.midi.MidiDevice;
import android.media.midi.MidiDeviceInfo;
import android.media.midi.MidiInputPort;
import android.media.midi.MidiManager;
import android.media.midi.MidiManager.DeviceCallback;
import android.media.midi.MidiOutputPort;
import android.media.midi.MidiReceiver;
import android.os.Build;
import android.os.Handler;
import android.util.Log;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


/**
 * Zentrale Klasse zum Öffnen von Midi-Geräten und -Ports und zum Senden/Empfangen von MIDI-Nachrichten
 *
 * @see <a href="http://developer.android.com/reference/android/media/midi/package-summary.html">Android MIDI Packacke Docs</a>
 */
@TargetApi(Build.VERSION_CODES.M)
public class Midi  {

    public interface MidiNoteListener {
        void onMidiNoteReceived(byte note);
    }

    private final MidiManager midiManager;

    private final List<MidiDevice> devices; // geöffnete MIDI-Geräte
    private final List<MidiInputPort> inputPorts; // sende MIDI-Befehle an diese Ports
    private final List<MidiOutputPort> outputPorts; // empfange MIDI-Befehle von diesen Ports

    private final MidiFramer midiFramer; // Hilfsklasse zum Umsortieren von SysEx-Befehlen


    public static boolean isMidiSupported(Context ctx) {
        return ctx.getPackageManager().hasSystemFeature(PackageManager.FEATURE_MIDI);
    }


    public Midi(final Context ctx,final MidiNoteListener midiNoteListener) {
        inputPorts = new ArrayList<>();
        outputPorts = new ArrayList<>();
        devices=new ArrayList<>();
        midiFramer=new MidiFramer(new MidiReceiver() {
            @Override
            public void onSend(byte[] msg, int offset, int count, long timestamp) throws IOException {
                // copy bytes
                byte[] bytes = Arrays.copyOfRange(msg, offset, offset + count);
                if(bytes.length ==3 && bytes[0] == MidiConstants.STATUS_NOTE_ON && bytes[2]!=0)
                {
                    final byte note=bytes[1];
                    Log.v(getClass().getSimpleName(),"Received note "+note);
                    if(midiNoteListener!=null) {
                        // UI thread
                        new Handler(ctx.getMainLooper()).post(new Runnable() {
                            @Override
                            public void run() {
                                midiNoteListener.onMidiNoteReceived(note);
                            }
                        });
                    }
                }
            }
        });

        midiManager = (MidiManager) ctx.getSystemService(Context.MIDI_SERVICE);
        midiManager.registerDeviceCallback(deviceCallback, null);
        openPorts();
    }

    private DeviceCallback deviceCallback = new DeviceCallback() {
        @Override
        public void onDeviceAdded(MidiDeviceInfo device) {
            super.onDeviceAdded(device);
            openPorts();
        }

        @Override
        public void onDeviceRemoved(MidiDeviceInfo device) {
            super.onDeviceRemoved(device);
            openPorts();
        }
    };

    public void close() {
        midiManager.unregisterDeviceCallback(deviceCallback);
        closePorts();
        closeDevices();
    }

// öffne alle Input- und Output-Ports alle Geräte
   private void openPorts() {
        // schließe evtl. geöffnete Geräte/Ports
        closePorts();
        closeDevices();
        // und suche neu
        for (final MidiDeviceInfo deviceInfo : midiManager.getDevices()) {
            Log.v(getClass().getSimpleName(), "opening MIDI device " + deviceInfo);
            if (deviceInfo.getInputPortCount() == 0 && deviceInfo.getOutputPortCount() == 0)
                continue; // keine Input- oder Output-Ports
            midiManager.openDevice(deviceInfo, new MidiManager.OnDeviceOpenedListener() {
                @Override
                public void onDeviceOpened(MidiDevice device) {
                    String deviceName = deviceInfo.getProperties().getString(MidiDeviceInfo.PROPERTY_NAME);
                    if (device == null) {
                        Log.e(getClass().getSimpleName(), "could not open device " + deviceInfo);
                    } else {
                        devices.add(device);
                        Log.v(getClass().getSimpleName(), "opened device " + deviceName);
                        for (final MidiDeviceInfo.PortInfo portInfo : device.getInfo().getPorts()) {

                            if (portInfo.getType() == MidiDeviceInfo.PortInfo.TYPE_OUTPUT) {
                                Log.v(getClass().getSimpleName(), "MIDI output port " + portInfo.getName());
                                MidiOutputPort outputPort = device.openOutputPort(portInfo.getPortNumber());
                                if (outputPort != null) {
                                    outputPorts.add(outputPort);
                                    outputPort.connect(midiFramer);

                                }
                            } // output port

                            if (portInfo.getType() == MidiDeviceInfo.PortInfo.TYPE_INPUT) {
                                Log.v(getClass().getSimpleName(), "MIDI input port " + portInfo.getName());
                                MidiInputPort inputPort = device.openInputPort(portInfo.getPortNumber());
                                if (inputPort != null) {
                                    inputPorts.add(inputPort);
                                }
                            }

                        } // ports
                    }
                } // onDeviceOpened
            }, null); // OnDeviceOpenedListener
        } // midiDevices
    }

    // schließe geöffnete Ports
    private void closePorts() {
        for (MidiInputPort inputPort : inputPorts) {
            try {
                inputPort.close();
            } catch (IOException e) {
                Log.e(getClass().getSimpleName(), "Could not close port " + inputPort);
            }
        }
        inputPorts.clear();
        for (MidiOutputPort outputPort : outputPorts) {
            try {
                outputPort.disconnect(midiFramer);
                outputPort.close();
            } catch (IOException e) {
                Log.e(getClass().getSimpleName(), "Could not close port " + outputPort);
            }
        }
        outputPorts.clear();
    }

    // schließe geöffnete Geräte
    private void closeDevices() {
        for(MidiDevice device:devices) {
            try {
                device.close();
            } catch (IOException e) {
                Log.e(getClass().getSimpleName(),"Could not close device "+device);
            }
        }
		devices.clear();
    }

    public static String nameForNote(byte note)
    {
        final String noteNames[]={"C","C#","D","Eb","E","F","F#","G","G#","A","Bb","B"};
        note-=MidiConstants.NOTE_MIDDLE_C;
        while(note<0)
            note+=12;
        while(note>=noteNames.length)
            note-=12;
        return  noteNames[note];
    }

    // sende Noten nacheinander und zum Schluss ein All Notes Off
    public void sendNotes(byte[] notes) {
        try {
            for (MidiInputPort inputPort : inputPorts) {
                long timestamp = System.nanoTime();
                final long dt = MidiConstants.NANOS_PER_SECOND / 3; // 0.3 sec
                for (byte note : notes) {
                    Log.i(getClass().getSimpleName(), "Sending note " + note);
                    // Note On: 0x90 <note> 0x7F
                    byte[] bytes = new byte[]{MidiConstants.STATUS_NOTE_ON, note, 0x7F}; // letztes Byte ist Lautstärke
                    inputPort.send(bytes, 0, bytes.length, timestamp);
                    timestamp += dt;
                } // notes

                // send all notes off 0xB0 0x7B 0
                timestamp += MidiConstants.NANOS_PER_SECOND*2; // 2 sec
                byte[] bytes = new byte[]{MidiConstants.STATUS_CONTROL_CHANGE, MidiConstants.STATUS_ALL_NOTES_OFF, 0};
                inputPort.send(bytes, 0, bytes.length, timestamp);


            } // inputPorts
        } catch (IOException e) {
            Log.e(getClass().getSimpleName(), "Error sending Midi");
        }
    }

    // Hilfsklassen aus dem Android-Beispielprojekt
    // http://developer.android.com/samples/MidiScope/index.html

    /**
     * MIDI related constants and static methods.
     * These values are defined in the MIDI Standard 1.0
     * available from the MIDI Manufacturers Association.
     */
    public static class MidiConstants {
        final static long NANOS_PER_SECOND = 1000000000L;

        public static final byte STATUS_COMMAND_MASK = (byte) 0xF0;
        public static final byte STATUS_CHANNEL_MASK = (byte) 0x0F;


        public static final byte NOTE_MIDDLE_C = (byte)60;
        // Channel voice messages.
        public static final byte STATUS_NOTE_OFF = (byte) 0x80;
        public static final byte STATUS_NOTE_ON = (byte) 0x90;
        public static final byte STATUS_ALL_NOTES_OFF = 0x7B;
        public static final byte STATUS_POLYPHONIC_AFTERTOUCH = (byte) 0xA0;
        public static final byte STATUS_CONTROL_CHANGE = (byte) 0xB0;
        public static final byte STATUS_PROGRAM_CHANGE = (byte) 0xC0;
        public static final byte STATUS_CHANNEL_PRESSURE = (byte) 0xD0;
        public static final byte STATUS_PITCH_BEND = (byte) 0xE0;

        // System Common Messages.
        public static final byte STATUS_SYSTEM_EXCLUSIVE = (byte) 0xF0;
        public static final byte STATUS_MIDI_TIME_CODE = (byte) 0xF1;
        public static final byte STATUS_SONG_POSITION = (byte) 0xF2;
        public static final byte STATUS_SONG_SELECT = (byte) 0xF3;
        public static final byte STATUS_TUNE_REQUEST = (byte) 0xF6;
        public static final byte STATUS_END_SYSEX = (byte) 0xF7;

        // System Real-Time Messages
        public static final byte STATUS_TIMING_CLOCK = (byte) 0xF8;
        public static final byte STATUS_START = (byte) 0xFA;
        public static final byte STATUS_CONTINUE = (byte) 0xFB;
        public static final byte STATUS_STOP = (byte) 0xFC;
        public static final byte STATUS_ACTIVE_SENSING = (byte) 0xFE;
        public static final byte STATUS_RESET = (byte) 0xFF;

        /** Number of bytes in a message nc from 8c to Ec */
        public final static int CHANNEL_BYTE_LENGTHS[] = { 3, 3, 3, 3, 2, 2, 3 };

        /** Number of bytes in a message Fn from F0 to FF */
        public final static int SYSTEM_BYTE_LENGTHS[] = { 1, 2, 3, 2, 1, 1, 1, 1, 1,
                1, 1, 1, 1, 1, 1, 1 };

        /**
         * MIDI messages, except for SysEx, are 1,2 or 3 bytes long.
         * You can tell how long a MIDI message is from the first status byte.
         * Do not call this for SysEx, which has variable length.
         * @param statusByte status byte
         * @return number of bytes in a complete message, zero if data byte passed
         */
        public static int getBytesPerMessage(byte statusByte) {
            // Java bytes are signed so we need to mask off the high bits
            // to get a value between 0 and 255.
            int statusInt = statusByte & 0xFF;
            if (statusInt >= 0xF0) {
                // System messages use low nibble for size.
                return SYSTEM_BYTE_LENGTHS[statusInt & 0x0F];
            } else if(statusInt >= 0x80) {
                // Channel voice messages use high nibble for size.
                return CHANNEL_BYTE_LENGTHS[(statusInt >> 4) - 8];
            } else {
                return 0; // data byte
            }
        }

        /**
         * @param msg
         * @param offset
         * @param count
         * @return true if the entire message is ActiveSensing commands
         */
        public static boolean isAllActiveSensing(byte[] msg, int offset,
                                                 int count) {
            // Count bytes that are not active sensing.
            int goodBytes = 0;
            for (int i = 0; i < count; i++) {
                byte b = msg[offset + i];
                if (b != MidiConstants.STATUS_ACTIVE_SENSING) {
                    goodBytes++;
                }
            }
            return (goodBytes == 0);
        }

    }

    /**
     * Convert stream of arbitrary MIDI bytes into discrete messages.
     *
     * Parses the incoming bytes and then posts individual messages to the receiver
     * specified in the constructor. Short messages of 1-3 bytes will be complete.
     * System Exclusive messages may be posted in pieces.
     *
     * Resolves Running Status and interleaved System Real-Time messages.
     */
    public static class MidiFramer extends MidiReceiver {
        private MidiReceiver mReceiver;
        private byte[] mBuffer = new byte[3];
        private int mCount;
        private byte mRunningStatus;
        private int mNeeded;
        private boolean mInSysEx;

        public MidiFramer(MidiReceiver receiver) {
            mReceiver = receiver;
        }

        /*
         * @see android.midi.MidiReceiver#onSend(byte[], int, int, long)
         */
        @Override
        public void onSend(byte[] data, int offset, int count, long timestamp)
                throws IOException {
            int sysExStartOffset = (mInSysEx ? offset : -1);

            for (int i = 0; i < count; i++) {
                final byte currentByte = data[offset];
                final int currentInt = currentByte & 0xFF;
                if (currentInt >= 0x80) { // status byte?
                    if (currentInt < 0xF0) { // channel message?
                        mRunningStatus = currentByte;
                        mCount = 1;
                        mNeeded = MidiConstants.getBytesPerMessage(currentByte) - 1;
                    } else if (currentInt < 0xF8) { // system common?
                        if (currentByte == MidiConstants.STATUS_SYSTEM_EXCLUSIVE /* SysEx Start */) {
                            // Log.i(TAG, "SysEx Start");
                            mInSysEx = true;
                            sysExStartOffset = offset;
                        } else if (currentByte == MidiConstants.STATUS_END_SYSEX /* SysEx End */) {
                            // Log.i(TAG, "SysEx End");
                            if (mInSysEx) {
                                mReceiver.send(data, sysExStartOffset,
                                        offset - sysExStartOffset + 1, timestamp);
                                mInSysEx = false;
                                sysExStartOffset = -1;
                            }
                        } else {
                            mBuffer[0] = currentByte;
                            mRunningStatus = 0;
                            mCount = 1;
                            mNeeded = MidiConstants.getBytesPerMessage(currentByte) - 1;
                        }
                    } else { // real-time?
                        // Single byte message interleaved with other data.
                        if (mInSysEx) {
                            mReceiver.send(data, sysExStartOffset,
                                    offset - sysExStartOffset, timestamp);
                            sysExStartOffset = offset + 1;
                        }
                        mReceiver.send(data, offset, 1, timestamp);
                    }
                } else { // data byte
                    if (!mInSysEx) {
                        mBuffer[mCount++] = currentByte;
                        if (--mNeeded == 0) {
                            if (mRunningStatus != 0) {
                                mBuffer[0] = mRunningStatus;
                            }
                            mReceiver.send(mBuffer, 0, mCount, timestamp);
                            mNeeded = MidiConstants.getBytesPerMessage(mBuffer[0]) - 1;
                            mCount = 1;
                        }
                    }
                }
                ++offset;
            }

            // send any accumulatedSysEx data
            if (sysExStartOffset >= 0 && sysExStartOffset < offset) {
                mReceiver.send(data, sysExStartOffset,
                        offset - sysExStartOffset, timestamp);
            }
        }

    }
}
