﻿using System;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace IdleTimer
{
  static class Utility
  {
    [StructLayout(LayoutKind.Sequential)]
    struct LASTINPUTINFO
    {
      [MarshalAs(UnmanagedType.U4)]
      public UInt32 cbSize;
      [MarshalAs(UnmanagedType.U4)]
      public UInt32 dwTime;
    }

    [DllImport("user32.dll")]
    static extern bool GetLastInputInfo(ref LASTINPUTINFO plii);

    public static int GetLastInputTime()
    {
      int idleTime = 0;
      LASTINPUTINFO lastInputInfo = new LASTINPUTINFO {
        cbSize = (UInt32)Marshal.SizeOf(typeof(LASTINPUTINFO)),
        dwTime = 0
      };
      int envTicks = Environment.TickCount;
      if(GetLastInputInfo(ref lastInputInfo)) {
        int lastInputTick = (int)lastInputInfo.dwTime;
        idleTime = envTicks - lastInputTick;
      }
      return ((idleTime > 0) ? (idleTime / 1000) : 0);
    }

    [DllImport("user32.dll")]
    static extern IntPtr GetForegroundWindow();

    [DllImport("user32.dll", SetLastError = true)]
    static extern IntPtr GetWindowThreadProcessId(IntPtr hWnd, out IntPtr lpdwProcessId);

    public static Process GetForegroundProcess()
    {
      IntPtr hWndForegrund = GetForegroundWindow();
      if(hWndForegrund != IntPtr.Zero) {
        IntPtr procId = IntPtr.Zero;
        GetWindowThreadProcessId(hWndForegrund, out procId);
        if(procId != IntPtr.Zero) {
          return Process.GetProcessById((int)procId);
        }
      }
      return null;
    }

    // Trennt den Programmnamen in einer Kommadozeile von den Argumenten.
    // Der Programmname ist das erste Wort des Eingabe-Strings, entweder bis zum ersten
    // Leerzeichen oder bis zum erten schließenden Anführungszeichen.
    public static Tuple<string, string> DissectCommandLine(this string line)
    {
      line = line.Trim();
      string prg = null, arg = null;
      int pos;
      if(string.IsNullOrWhiteSpace(line))
        return Tuple.Create(prg, arg);
      if(line[0] == '"') {
        pos = line.IndexOf('"', 1);
        if(pos != -1) {
          prg = line.Substring(1, pos - 1);
          arg = line.Substring(pos + 1).Trim();
        }
        else {
          prg = line.Substring(1);
        }
      }
      else {
        pos = line.IndexOf(' ');
        if(pos != -1) {
          prg = line.Substring(0, pos);
          arg = line.Substring(pos + 1).Trim();
        }
        else {
          prg = line;
        }

      }
      return Tuple.Create(prg, arg);
    }

  }

}
