﻿using Microsoft.Win32;
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Runtime.CompilerServices;
using System.Threading;
using System.Windows;
using System.Windows.Input;
using System.Windows.Threading;

namespace IdleTimer
{
  /// <summary>
  /// Interaktionslogik für MainWindow.xaml
  /// </summary>
  public partial class MainWindow : Window
  {
    IdleData _idleData = new IdleData();
    DispatcherTimer _watchDogTimer;
    ShowWindowCommand _toggleCommand; // Wird für die TaskbarIcon-Komponente gebraucht
    bool _running = true;
    Process _idleProcess = null;

    public MainWindow()
    {
      this.DataContext = _idleData;
      InitializeComponent();

      _watchDogTimer = new DispatcherTimer();
      _watchDogTimer.Tick += watchDog_Tick;
      _watchDogTimer.Interval = TimeSpan.FromMilliseconds(1000);

      _idleData.IdleExe = Properties.Settings.Default.IdleUrl;
      _idleData.TimeToWait = Properties.Settings.Default.ReloadTimeout;

      _watchDogTimer.Start();
    }

    private void watchDog_Tick(object sender, EventArgs e)
    {
      _idleData.LastInputTime = Utility.GetLastInputTime();
      if(_idleData.TimeRemaining <= 0) {
        // Wartezeit abgelaufen
        try {
          if(_idleData.TimeOffset != 0) {
            // Timer ist mindestens zum zweiten Mal abgelaufen.
            // Den Prozess des letzten Aufrufs beenden.
            if(_idleProcess != null) {
              if(!_idleProcess.CloseMainWindow()) {
                _idleProcess.Kill();
              }
              _idleProcess = null;
              Thread.Sleep(1000);
            }
          }
          var parts = _idleData.IdleExe.DissectCommandLine();
          if(!string.IsNullOrWhiteSpace(parts.Item1)) {
            ProcessStartInfo psi = new ProcessStartInfo(parts.Item1, parts.Item2);
            _idleProcess = Process.Start(psi);
            // Wenn kein neuer Prozess gestartet wurde, ist der, zu dem das Fenster
            // im Vordergrund (nach 2 Sekunden Wartezeit) gehört, der zuständige.
            if(_idleProcess == null) {
              var myProcess = Process.GetCurrentProcess();
              Thread.Sleep(2000);
              var fgProc = Utility.GetForegroundProcess();
              if(fgProc != myProcess)
                _idleProcess = fgProc;
            }
          }
          _idleData.TimeOffset += _idleData.TimeToWait;
        }
        catch(Win32Exception) {
          // Hat nicht geklappt - na und?
        }
      }
      else if(_idleData.TimeRemaining > _idleData.TimeToWait) {
        // Eingabe nach mehrfachem Ablauf der Wartezeit
        _idleData.TimeOffset = 0;
      }
    }

    private void btnSave_Click(object sender, RoutedEventArgs e)
    {
      Properties.Settings.Default.IdleUrl = _idleData.IdleExe;
      Properties.Settings.Default.ReloadTimeout = _idleData.TimeToWait;
      Properties.Settings.Default.Save();
    }

    private void btnBrowse_Click(object sender, RoutedEventArgs e)
    {
      var dlg = new OpenFileDialog {
        Filter = "Programme|*.exe|Alle Dateien|*.*",
        DefaultExt = ".exe",
        Title = "Leerlaufprogramm"
      };
      var parts = _idleData.IdleExe.DissectCommandLine();
      if(!string.IsNullOrWhiteSpace(parts.Item1))
        dlg.FileName = parts.Item1;
      if(dlg.ShowDialog(this) == true) {
        string app = dlg.FileName;
        if(app.IndexOf(' ') != -1)
          app = "\"" + app + "\"";
        if(!string.IsNullOrWhiteSpace(parts.Item2))
          app += " " + parts.Item2;
        _idleData.IdleExe = app;
      }
    }

    private void btnStart_Click(object sender, RoutedEventArgs e)
    {
      try {
        var parts = _idleData.IdleExe.DissectCommandLine();
        if(!string.IsNullOrWhiteSpace(parts.Item1)) {
          ProcessStartInfo psi = new ProcessStartInfo(parts.Item1, parts.Item2);
          _idleProcess = Process.Start(psi);
        }
      }
      catch(Win32Exception wex) {
        MessageBox.Show(wex.Message, "Fehler", MessageBoxButton.OK, MessageBoxImage.Error);
      }
    }

    private void Window_Loaded(object sender, RoutedEventArgs e)
    {
      _toggleCommand = new ShowWindowCommand { Target = this };
      taskBarIcon.DoubleClickCommand = _toggleCommand;
      cmdOpen.Command = _toggleCommand;
#if DEBUG
#else
      this.Visibility = Visibility.Hidden;
#endif
    }

    // Kontextmenübefehl "Schließen" des Tray-Icons
    private void cmdClose_Click(object sender, RoutedEventArgs e)
    {
      this.Close();
    }

    // Kontextmenübefehl "Aktiviert" des Tray-Icons
    private void cmdTimer_Click(object sender, RoutedEventArgs e)
    {
      _running = !_running;
      cmdTimer.IsChecked = _running;
      if(_running)
        _watchDogTimer.Start();
      else
        _watchDogTimer.Stop();
    }

    private void Window_StateChanged(object sender, EventArgs e)
    {
      // Wir minimieren in den Tray
      if(WindowState == WindowState.Minimized) {
        Visibility = Visibility.Hidden;
        WindowState = WindowState.Normal;
      }
    }

  }

  public class IdleData : INotifyPropertyChanged
  {
    // Die nach Ablauf der Wartezeit zu startende Kommandozeile
    private string _idleExe;
    public string IdleExe
    {
      get { return _idleExe; }
      set { SetProperty(ref _idleExe, value); }
    }

    // So viele Millisekunden ist die letzte Eingabe her
    private int _lastInputTime;
    public int LastInputTime
    {
      get { return _lastInputTime; }
      set
      {
        if(SetProperty(ref _lastInputTime, value))
          OnPropertyChanged("TimeRemaining");
      }
    }

    // So viele Millisekunden soll jedes Mal gewartet werden
    private int _timeToWait;
    public int TimeToWait
    {
      get { return _timeToWait; }
      set
      {
        if(SetProperty(ref _timeToWait, Math.Max(2, value)))
          OnPropertyChanged("TimeRemaining");
      }
    }

    // Korrekturwert, um mehrfachen Ablauf der Wartezeit zu brücksichtigen
    private int _timeOffset;
    public int TimeOffset
    {
      get { return _timeOffset; }
      set
      {
        if(SetProperty(ref _timeOffset, value))
          OnPropertyChanged("TimeRemaining");
      }
    }

    public int TimeRemaining
    {
      get { return _timeToWait + _timeOffset - _lastInputTime; }
    }

    #region INotifyPropertyChanged
    public event PropertyChangedEventHandler PropertyChanged;
    protected bool SetProperty<T>(ref T storage, T value, [CallerMemberName] String propertyName = null)
    {
      if(object.Equals(storage, value))
        return false;
      storage = value;
      this.OnPropertyChanged(propertyName);
      return true;
    }

    protected void OnPropertyChanged([CallerMemberName] string propertyName = null)
    {
      var eventHandler = this.PropertyChanged;
      if(eventHandler != null) {
        eventHandler(this, new PropertyChangedEventArgs(propertyName));
      }
    }
    #endregion
  }

  public class ShowWindowCommand : ICommand
  {
    public UIElement Target { get; set; }

    public void Execute(object parameter)
    {
      if(Target != null) {
        if(Target.Visibility == Visibility.Visible) {
          Target.Visibility = Visibility.Hidden;
        }
        else {
          Target.Visibility = Visibility.Visible;
          Target.Focus();
        }
      }
    }

    public bool CanExecute(object parameter)
    {
      return (Target != null);
    }

    // Eigentlich unnötig, aber der Compiler motzt ohne das hier ...
    private EventHandler _canExecuteChanged;
    event EventHandler ICommand.CanExecuteChanged
    {
        add
        {
          _canExecuteChanged += value;
        }
        remove
        {
          _canExecuteChanged -= value;
        }
    }

  }
}
