package com.linkesoft.ctlongitude.common;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.location.Address;
import android.location.Location;
import android.net.Uri;
import android.os.Bundle;
import android.util.Base64;
import android.util.Log;
import android.view.View;

import org.json.JSONException;
import org.json.JSONObject;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

public class Friend {
    public String userid;
    public String name;
    public Location location;
    public Date timestamp;
    public Address address;
    public Uri imageuri; // optional friend image data URI

    public Object marker; // for showing on Google maps
    public View view; // for smartwatch updates

    private static final String NAME = "name";
    private static final String USERID = "userid";
    private static final String AVATAR = "avatar";
    private static final String LAT = "lat";
    private static final String LON = "lng";
    public Friend()
    {
    }
    public Friend(Bundle bundle)
    {
        userid=bundle.getString(USERID);
        name=bundle.getString(NAME);
        location =new Location("ctLongitude");
        location.setLatitude(bundle.getDouble(LAT));
        location.setLongitude(bundle.getDouble(LON));
        if(bundle.getString(AVATAR)!=null)
            imageuri=Uri.parse(bundle.getString(AVATAR));
    }

    public Bundle toBundle()
    {
        Bundle bundle=new Bundle();
        bundle.putString(USERID,userid);
        bundle.putString(NAME,name);
        bundle.putDouble(LAT, location.getLatitude());
        bundle.putDouble(LON, location.getLongitude());
        if(imageuri!=null)
            bundle.putString(AVATAR,imageuri.toString());
        return bundle;
    }

	public void setFromJSON(JSONObject friendobj) throws JSONException {
		userid = friendobj.getString(USERID);
		name = friendobj.getString(NAME);
		location =new Location("ctLongitude");
        location.setLatitude(friendobj.getDouble(LAT));
        location.setLongitude(friendobj.getDouble(LON));
        if(friendobj.has("timestamp"))
		    timestamp = new Date(friendobj.getInt("timestamp")*1000L); // Date expects ms, API gives sec
        if(friendobj.has(AVATAR))
            imageuri=Uri.parse(friendobj.getString(AVATAR));
	}

    public JSONObject toJSON()
    {
        JSONObject friendobj=new JSONObject();
        try {
            friendobj.put(USERID,userid);
            friendobj.put(NAME,name);
            friendobj.put(LAT, location.getLatitude());
            friendobj.put(LON, location.getLongitude());
            if(imageuri!=null)
                friendobj.put(AVATAR,imageuri);
        } catch (JSONException e) {
            Log.e(getClass().getSimpleName(), "Cannot serialize friend to json", e);
        }
        return friendobj;
    }

	@Override
	public String toString() {
		StringBuffer str = new StringBuffer();
		str.append("Friend");
		if (name != null)
			str.append(" " + name);
		if(timestamp!=null)
			str.append(" "+getFormattedDate());
		if (address != null)
			str.append(" " + getFormattedAddress(null));
		return str.toString();
	}

	public String getFormattedDate() {
		Calendar cal=Calendar.getInstance(); // aktueller Zeitpunkt
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND,0);
		// heute? -> nur Uhrzeit
		if(timestamp.after(cal.getTime()))
		{
			// time
			DateFormat sdf = SimpleDateFormat.getTimeInstance(DateFormat.SHORT, Locale.getDefault());
			return  sdf.format(timestamp.getTime());
		}
		else {
			// date
			DateFormat sdf = SimpleDateFormat.getDateInstance();
			return  sdf.format(timestamp.getTime());
		}
	}
	/**
	 * kompakte Adressanzeige
	 * wenn homeAddress angegeben, zeige nur Straße falls in derselben Stadt+Land
	 * Straße+Ort falls in anderer Stadt
	 * Ort+Land falls in anderem Land
	 * wenn keine homeAddress angegeben, nutze Standard-Formatierung 
	 * @param homeAddress Adresse des Nutzers selbst (falls bekannt)
	 * @return formatierte Adresse
	 */
	public String getFormattedAddress(Address homeAddress) {
		// Address-Struktur-Namen:
		// http://www.oasis-open.org/committees/ciq/ciq.html#6
		String town = address.getLocality();
		String street = address.getThoroughfare();
		String country = address.getCountryName();
		if (address.getSubThoroughfare() != null && street != null)
			street += " " + address.getSubThoroughfare();
		StringBuffer buf = new StringBuffer();
		if (homeAddress != null) {
			if (country != null && homeAddress.getCountryName() != null && country.equals(homeAddress.getCountryName())) {
				// gleiches Land
				if (town != null && homeAddress.getLocality() != null && town.equals(homeAddress.getLocality())) {
					// gleiche Stadt
					if (street != null)
						buf.append(street);
				} else {
					// andere Stadt
					if (street != null)
						buf.append(street + "\n");
					if (town != null)
						buf.append(town);
				}
			} else {
				// anderes Land
				if (town != null)
					buf.append(town + "\n");
				if (country != null)
					buf.append(country);
			}
		}
		if (buf.length() == 0) {
			// construct standard address string
			for (int i = 0; i < address.getMaxAddressLineIndex(); i++) {
				if (buf.length() != 0)
					buf.append("\n");
				buf.append(address.getAddressLine(i));
			}
		}
		return buf.toString();

	}

    /**
     * Erzeuge Bitmap aus data image URI wie data:image/png;base64,/9j4AAQSkZJR...
     * @return Bitmap or null
     */
    public Bitmap getBitmap() {
        if(imageuri==null)
            return null;
        // imageuri is base64 encoded
        if(imageuri.getScheme().equals("data")) {
            int ind=imageuri.toString().indexOf("base64,")+"base64".length();
            byte[] data = Base64.decode(imageuri.toString().substring(ind),0);
            return BitmapFactory.decodeByteArray(data, 0, data.length);
        }
        else
        {
            Log.e(getClass().getSimpleName(),"Unsupported image uri "+imageuri);
            return null;
        }
    }
}
