//
//	rucksack.cpp
//	============
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#include "stdafx.h"
#include "../resource.h"

#include <vector>
#include <list>
#include <algorithm>
using namespace std;

#include <limits.h>
#include <float.h>
#include <time.h>
#include <math.h>
#include "../MersenneTwister/SFMT.h"	// "good" random numbers

#ifndef _DEBUG
    #include <omp.h>		// OpenMP support
#endif

#include "../assert.h"
#include "../Splash.h"
#include "../galib.h"
#include "../ga_app.h"
#include "rucksack.h"


// ================================================================================================================
//												Rucksack Application Class
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
class CRucksackApp : public CGaApplication<TGenome, TVisualization, TSelection>
{
protected:
	TGenome		m_BruteForceGenome;			// best solution found by Brute-Force algorithm

public:
	virtual void Init(HINSTANCE hInstance);
	virtual void CalcGenetic();
	virtual void DoCalcBruteForce();
	virtual void MainWndPaint(HWND hWnd, HDC hdc);
	virtual void PopWndPaint(HWND hWnd, HDC hdc);
};


// ================================================================================================================
//												Global Variables
// ================================================================================================================
CRucksackApp<CRsGenome, CGaAppVisualize<CRsGenome>, CNaturalSelection<CRsGenome>> gGeneticApp;


// ================================================================================================================
//											Application Interface Class
//
// This is just glue-code.
// ================================================================================================================
class CAppInterface : public CGaAppInterface
{
public:
	virtual void MainWndCreate() { gGeneticApp.MainWndCreate(); }
	virtual bool MainWndCommand(int wmId, int wmEvent) { return gGeneticApp.MainWndCommand(wmId, wmEvent); }
	virtual void MainWndPaint(HWND hWnd, HDC hdc) { gGeneticApp.MainWndPaint(hWnd, hdc); }
	virtual void MainWndDestroy() { gGeneticApp.MainWndDestroy(); }

	virtual void MeanWndPaint(HWND hWnd, HDC hdc) { gGeneticApp.MeanWndPaint(hWnd, hdc); }
	virtual void PopWndPaint(HWND hWnd, HDC hdc) { gGeneticApp.PopWndPaint(hWnd, hdc); }
};


// ================================================================================================================
//											InitApplication()
// ================================================================================================================
void InitApplication(HINSTANCE hInstance)
{
	gGeneticApp.Init(hInstance);
}


// ================================================================================================================
//											Init()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CRucksackApp<TGenome, TVisualization, TSelection>::Init(HINSTANCE hInstance)
{
	// initialize Application Interface Object
	gpGaAppInterface = new CAppInterface();

	// perform application specific initializations
	InitBitmasks();		// init bitmasks in rucksack.h

	#if 0	// create new source code for the items-array, this can be copied & pasted into rucksack.h
		FILE *fh = fopen("array.txt", "wt");
		for (int i = 0; i < 64; i++)
		{
			fprintf(fh, "\t%d,\t%d,\n", gen_rand32() % 1000, gen_rand32() % 100);
		}
		fclose(fh);
	#endif

	// call base class code
	CGaApplication<TGenome, TVisualization, TSelection>::Init(hInstance);
}


// ================================================================================================================
//												CalcGenetic()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CRucksackApp<TGenome, TVisualization, TSelection>::CalcGenetic()
{
#if NUM_BITS == 32
	// anzahl individuen skaliert auf multi-prozessor!
	m_GeneticAlgo.SetPopulationSize(512);
	m_GeneticAlgo.SetCrossoverProbability(45);
	m_GeneticAlgo.SetMutationProbability(50);
	m_GeneticAlgo.SetElitismPercentage(10);
	m_GeneticAlgo.SetStableGenerations(200);
	m_GeneticAlgo.SetMaxGenerations(600);
#else
	// anzahl individuen skaliert auf multi-prozessor!
	m_GeneticAlgo.SetPopulationSize(512);
	m_GeneticAlgo.SetCrossoverProbability(45);
	m_GeneticAlgo.SetMutationProbability(33);
	m_GeneticAlgo.SetElitismPercentage(10);
	m_GeneticAlgo.SetStableGenerations(200);
	m_GeneticAlgo.SetMaxGenerations(600);
#endif

	CGaApplication<TGenome, TVisualization, TSelection>::CalcGenetic();
}


// ================================================================================================================
//												DoCalcBruteForce()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CRucksackApp<TGenome, TVisualization, TSelection>::DoCalcBruteForce()
{
	CRsGenome genome;
	TGENOME g = 0;
	int top = 0x1000000;
	vector<TGenome> thread_best;
	thread_best.resize(omp_get_num_procs());

	m_BruteForceGenome.Reset();

	do
	{
		// OpenMP does only allow variables of type integer in for-loops
		// so we are working with a smaller integer range from 0 to top
		// and offset it with a base value in g
		#pragma omp parallel for private(genome)
		for (int i = 0; i < top; i++)
		{
			genome.SetGenome(g + i);
			int fi = genome.ComputeFitness();
			int tn = omp_get_thread_num();
			if (fi > thread_best[tn].GetFitness())
				thread_best[tn] = genome;
		}

		if (m_enVisualization != enVisualizeOff)
		{
			// find best genome of all threads
			bool new_best_found = false;
			for (vector<CRsGenome>::iterator it = thread_best.begin(); it != thread_best.end(); it++)
			{
				if (it->GetFitness() > m_BruteForceGenome.GetFitness())
				{
					m_BruteForceGenome = *it;
					new_best_found = true;
				}
			}

			if (new_best_found)
			{
				InvalidateRect(m_hWndMain, NULL, TRUE);
				UpdateWindow(m_hWndMain);
			}

			DispatchAllWindowMessages();
			if (m_bCancel)
				break;
		}

		g += top;
	}
	while (g != 0);

	// find best genome of all threads
	for (vector<CRsGenome>::iterator it = thread_best.begin(); it != thread_best.end(); it++)
	{
		if (it->GetFitness() > m_BruteForceGenome.GetFitness())
			m_BruteForceGenome = *it;
	}
}


// ================================================================================================================
//												MainWndPaint()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CRucksackApp<TGenome, TVisualization, TSelection>::MainWndPaint(HWND hWnd, HDC hdc)
{
	TCHAR s[256];
	const int nLineHeight = 14;
	int y = 10;
	TGenome solution;

	if (m_enCurrentAlgo == enAlgoGenetic)
		solution = m_GeneticAlgo.GetSolution();
	else
		solution = m_BruteForceGenome;

	if (solution.GetFitness() > 0)
	{
		// There is a solution available
		// we select the ANSI_VAR_FONT, which is a bit smaller than the normal system font
		HFONT hOldFont = (HFONT)SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));

		if (m_enCurrentAlgo == enAlgoGenetic)
			_tcscpy(s, _T("Genetic Algorithm"));
		else
			_tcscpy(s, _T("Brute-Force Algorithm"));
		TextOut(hdc, 10, y, s, (int)_tcslen(s));
		y += nLineHeight;

		_stprintf_s(s, _T("Genome: "));

		TGENOME mask = 1;
		for (int i = 0; i < NUM_BITS; i++)
		{
			if (solution.GetGenome() & mask)
				_tcscat(s, _T("1"));
			else
				_tcscat(s, _T("0"));

			mask <<= 1;
		}

		TextOut(hdc, 10, y, s, (int)_tcslen(s));
		y += nLineHeight;

		mask		= 1;
		int nSize	= 0;
		int nCount	= 0;
		for (int i = 0; i < NUM_BITS; i++)
		{
			if (solution.GetGenome() & mask)
			{
				_stprintf_s(s, _T("Item: %d"), i);
				TextOut(hdc, 10, y, s, (int)_tcslen(s));

				_stprintf_s(s, _T("Value: %d"), Items[i].m_nValue);
				TextOut(hdc, 100, y, s, (int)_tcslen(s));

				_stprintf_s(s, _T("Size: %d"), Items[i].m_nSize);
				TextOut(hdc, 200, y, s, (int)_tcslen(s));
				y += nLineHeight;

				nSize += Items[i].m_nSize;
				nCount++;
			}

			mask <<= 1;
		}
		mask = 1;

		y += 10;
		_stprintf_s(s, _T("Sum"));
		TextOut(hdc, 10, y, s, (int)_tcslen(s));

		_stprintf_s(s, _T("Value: %d"), solution.GetFitness());
		TextOut(hdc, 100, y, s, (int)_tcslen(s));

		_stprintf_s(s, _T("Size: %d"), nSize);
		TextOut(hdc, 200, y, s, (int)_tcslen(s));
		y += nLineHeight;

		_stprintf_s(s, _T("#Items = %d"), nCount);
		TextOut(hdc, 10, y, s, (int)_tcslen(s));
		y += nLineHeight;

		if (m_enCurrentAlgo == enAlgoGenetic)
		{
			_stprintf_s(s, _T("Generations %d (%d computations)"), m_GeneticAlgo.GetGenerations(), m_GeneticAlgo.GetGenerations() * m_GeneticAlgo.GetPopulationSize());
			TextOut(hdc, 10, y, s, (int)_tcslen(s));
			y += nLineHeight;

			_stprintf_s(s, _T("Solution found in generation %d"), m_GeneticAlgo.GetBSGeneration());
			TextOut(hdc, 10, y, s, (int)_tcslen(s));
			y += nLineHeight;

			if (!m_GeneticAlgo.GetIsComputing())
			{
				_stprintf_s(s, _T("done."));
				TextOut(hdc, 10, y, s, (int)_tcslen(s));
				y += nLineHeight;

				m_Timer.GetElapsedTime(s);
				TextOut(hdc, 10, y, s, (int)_tcslen(s));
				y += nLineHeight;
			}
		}
		else
		{
			if (m_bBruteForceDone)
			{
				_stprintf_s(s, _T("done."));
				TextOut(hdc, 10, y, s, (int)_tcslen(s));
				y += nLineHeight;

				m_Timer.GetElapsedTime(s);
				TextOut(hdc, 10, y, s, (int)_tcslen(s));
				y += nLineHeight;
			}
			else
			{
				_stprintf_s(s, _T("computing - please wait..."));
				TextOut(hdc, 10, y, s, (int)_tcslen(s));
			}
		}

		SelectObject(hdc, hOldFont);
	}
}


// ================================================================================================================
//													PopWndPaint()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CRucksackApp<TGenome, TVisualization, TSelection>::PopWndPaint(HWND hWnd, HDC hdc)
{
	RECT rcClient;
	GetClientRect(hWnd, &rcClient);
	LONG client_dx = rcClient.right - rcClient.left;
	LONG client_dy = rcClient.bottom - rcClient.top;

	CPopulation<TGenome> *pPopulation = m_GeneticAlgo.GetVisualization().GetPopulation();
	if (!pPopulation || client_dx == 0 || client_dy == 0)
	{
		PatBlt(hdc, 0, 0, client_dx, client_dy, WHITENESS);
		return;
	}

	LONG height = min(client_dy, (LONG)pPopulation->size());
	LONG width = (LONG)TGenome::m_nNumChromosomes;

	if (height == 0)
	{
		PatBlt(hdc, 0, 0, client_dx, client_dy, WHITENESS);
		return;
	}

	// create monochrome bitmap
	BITMAPINFO *pbmi = (BITMAPINFO *)malloc(sizeof(BITMAPINFOHEADER) + 8);
	pbmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pbmi->bmiHeader.biWidth = width;
	pbmi->bmiHeader.biHeight = height;
	pbmi->bmiHeader.biPlanes = 1;
	pbmi->bmiHeader.biBitCount = 1;
	pbmi->bmiHeader.biCompression = BI_RGB;
	pbmi->bmiHeader.biSizeImage = 0;
	pbmi->bmiHeader.biXPelsPerMeter = 0;
	pbmi->bmiHeader.biYPelsPerMeter = 0;
	pbmi->bmiHeader.biClrUsed = 0;
	pbmi->bmiHeader.biClrImportant = 0;
	pbmi->bmiColors[0].rgbBlue = 0;
	pbmi->bmiColors[0].rgbRed = 0;
	pbmi->bmiColors[0].rgbGreen = 0;
	pbmi->bmiColors[0].rgbReserved = 0;
	pbmi->bmiColors[1].rgbBlue = 0xff;
	pbmi->bmiColors[1].rgbRed = 0xff;
	pbmi->bmiColors[1].rgbGreen = 0xff;
	pbmi->bmiColors[1].rgbReserved = 0;

	// Macro to determine the number of bytes in a DWORD aligned DIB scanline
	#define BYTESPERLINE(Width, BPP) ((WORD)((((DWORD)(Width) * (DWORD)(BPP) + 31) >> 5)) << 2)

	DWORD bytes_per_line = BYTESPERLINE(width, pbmi->bmiHeader.biBitCount);
	const size_t bmp_size = bytes_per_line * height;

	TGENOME *pBitmap = (TGENOME *)malloc(bmp_size);
	memset(pBitmap, 0, bmp_size);
	TGENOME *p = pBitmap;
	TGENOME *start = pBitmap;

	LONG line = 0;
	for (vector<TGenome>::iterator it = pPopulation->begin(); it != pPopulation->end(); it++)
	{
		*p = it->GetGenome();

		start += bytes_per_line / sizeof(TGENOME);
		p = start;

		line++;
		if (line >= height)
			break;
	}

	// Blit
	StretchDIBits(
	  hdc,				// handle to DC
	  0,				// x-coord of destination upper-left corner
	  0,				// y-coord of destination upper-left corner
	  client_dx,		// width of destination rectangle
	  client_dy,		// height of destination rectangle
	  0,				// x-coord of source upper-left corner
	  0,				// y-coord of source upper-left corner
	  width,			// width of source rectangle
	  height,			// height of source rectangle
	  pBitmap,			// bitmap bits
	  pbmi,				// bitmap data
	  DIB_RGB_COLORS,	// usage options
	  SRCCOPY			// raster operation code
	);

	free(pbmi);
	free(pBitmap);
}
