//
//	galib.h
//	=======
//
// Genetic Algorithm Template Library
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#if !defined(_OPENMP)
	// OpenMP is not used or unavailable
	#define omp_get_num_procs()		1
	#define omp_get_thread_num()	0
#endif


// ================================================================================================================================
//													class CSampleGenome
//
// Sample-class for Genome-Implementations. Demonstrates the types and methods, which need to be implemented.
// ================================================================================================================================
class CSampleGenome
{
public:
	// Construction
	CSampleGenome() {}

	// type of fitness value
	typedef int TFitnessValueType;

	void Reset() {} // Must reset the Fitness-Value

	// initializes the genome with random chromosomes
	void Randomize() {}

	// computes the fitness-value, the bigger the better
	TFitnessValueType ComputeFitness() {}

	TFitnessValueType	GetFitness() const { return 0; }
	void				SetFitness(const TFitnessValueType /* val */) {}

	// Crossover
	void Crossover(const CSampleGenome &father, const CSampleGenome &mother) {}

	// Mutation
	void Mutate() {}
};


// ================================================================================================================================
//													class CPopulation
//
// Holds a population as well as the next generation of a population.
// ================================================================================================================================
template<class TGenome>
class CPopulation
{
protected:
	vector<TGenome>	m_arGen1;		// Stores one generation
	vector<TGenome>	m_arGen2;		// Stores other generation

	vector<TGenome>	*m_arCurGen;	// This array points to the current generation (either gen1 or gen2)
	vector<TGenome>	*m_arNewGen;	// This array points to the new generation (either gen1 or gen2)
									// We swap the pointers, so there is no need for copying the memory.

public:
	// The parameter "count" is the number of individuals in the population
	CPopulation(size_t count)
	{
		ASSERT(count % 2 == 0);		// number must be even

		m_arCurGen	= &m_arGen1;
		m_arNewGen	= &m_arGen2;

		SetSize(count);
	}

	void SetSize(size_t size)
	{
		m_arGen1.resize(size);
		m_arGen2.resize(size);
	}

	typename vector<TGenome>::iterator begin()			{ return m_arCurGen->begin(); }
	typename vector<TGenome>::iterator end()			{ return m_arCurGen->end(); }
	typename vector<TGenome>::size_type size() const	{ return m_arCurGen->size(); }

	TGenome	&GetGenome(size_t index)
	{
		return m_arCurGen->at(index);
	}

	TGenome	&GetNewGenome(size_t index)
	{
		return m_arNewGen->at(index);
	}

	// swap population arrays, so the new generation becomes the current generation and vice versa
	void NewGeneration()
	{
		vector<TGenome>	*swap = m_arCurGen;
		m_arCurGen = m_arNewGen;
		m_arNewGen = swap;
	}
};


// ================================================================================================================================
//													class CSelectionBase
//
// Base class for all selection algorithms.
// ================================================================================================================================
template<class TGenome>
class CSelectionBase
{
protected:
	class CRouletteItem
	{
	public:
		TGenome *m_pGenome;
		double	m_dblProbability;

		CRouletteItem()
		{
			m_pGenome			= NULL;
			m_dblProbability	= 0;
		}

   		bool operator <(const CRouletteItem &item) const
		{
			return m_dblProbability < item.m_dblProbability;
		}
	};

	double					m_dblSumFitnesses;
	vector<CRouletteItem>	m_Roulette;

public:
	void BuildData(CPopulation<TGenome> *pPopulation)
	{
		CRouletteItem item;

		// build the roulette data, copy fitness values
		m_dblSumFitnesses = 0;
		m_Roulette.clear();
		for (vector<TGenome>::iterator it = pPopulation->begin(); it != pPopulation->end(); it++)
		{
			item.m_pGenome = &*it;
			item.m_dblProbability = it->GetFitness();	// so we can sort by fitness, too
			m_Roulette.push_back(item);
			m_dblSumFitnesses += it->GetFitness();
		}

		// Sort by Fitness, so DoSelectElite() will work
		sort(m_Roulette.begin(), m_Roulette.end());
	}

	// requires that m_Roulette is sorted!
	void DoSelectElite(size_t n, TGenome *&father, TGenome *&mother)
	{
		if (m_Roulette.size() <= n + 1)
			father = m_Roulette.at(0).m_pGenome;
		else
			father = m_Roulette.at(m_Roulette.size() - n - 1).m_pGenome;

		if (m_Roulette.size() <= n + 2)
			mother = m_Roulette.at(0).m_pGenome;
		else
			mother = m_Roulette.at(m_Roulette.size() - n - 2).m_pGenome;
	}
};



// ================================================================================================================================
//													class CTournamentSelection
//
// Chooses the better of two randomly selected individuals.
// ================================================================================================================================
template<class TGenome>
class CTournamentSelection : public CSelectionBase<TGenome>
{
public:
	void PreSelect(CPopulation<TGenome> *pPopulation)
	{
		BuildData(pPopulation);
	}

	// The selection algorithm
	void DoSelect(TGenome *&father, TGenome *&mother)
	{
		size_t count = m_Roulette.size();
		size_t a = gen_rand32() % count;
		size_t b = gen_rand32() % count;
		if (m_Roulette[a].m_pGenome->GetFitness() > m_Roulette[b].m_pGenome->GetFitness())
			father = m_Roulette[a].m_pGenome;
		else
			father = m_Roulette[b].m_pGenome;

		a = gen_rand32() % count;
		b = gen_rand32() % count;
		if (m_Roulette[a].m_pGenome->GetFitness() > m_Roulette[b].m_pGenome->GetFitness())
			mother = m_Roulette[a].m_pGenome;
		else
			mother = m_Roulette[b].m_pGenome;
	}
};



// ================================================================================================================================
//													class CNaturalSelection
//
// See comment in method PreSelect().
//
// Especially the parameter Epsilon depends on the scale of the fitness values and the average fitness deviations of the
// individuals. Unfortunately Epsilon can not be of type double (or float), this is not allowed by Visual-C++.
// Therefore pass Epsilon as integer, which is divided by 10000 internally. (e.g. 15000 = 1.5)
// ================================================================================================================================
template<class TGenome, int nEliminationPercentage = 30, int Epsilon = 10000>
class CNaturalSelection : public CSelectionBase<TGenome>
{
public:
	// remove twins (individuals, whose fitnesses deviate by epsilon from each other)
	void RemoveTwins()
	{
		bool have_last = false;
		vector<CRouletteItem>::iterator it_last;
		vector<CRouletteItem>::iterator it = m_Roulette.begin();

		double dblEpsilon = Epsilon;
		dblEpsilon /= 10000;

		while (it != m_Roulette.end())
		{
			if (have_last && it->m_pGenome->GetFitness() - it_last->m_pGenome->GetFitness() <= dblEpsilon)
			{
				it = m_Roulette.erase(it_last);
				have_last = false;
			}
			else
			{
				it_last = it;
				it++;
				have_last = true;
			}
		}
	}

	void PreSelect(CPopulation<TGenome> *pPopulation)
	{
		BuildData(pPopulation);

		// Hiroaki SENGOKU and Ikuo YOSHIHARA
		// Eliminate R = M x pe / 100 individuals. In the so-called simple GA, the possibility of survival is proportional
		// to the Fitness value, but we pay more attention to the diversity of the population. We eliminate similar individuals
		// to maintain the diversity in order to avoid the immature convergence that is one of the well-known problems in GA.
		// First, sort the individuals in Fitness-value order.
		// Compare the Fitness value of adjoining individuals. If the difference is less than Epsilon (a small positive real
		// number), eliminate preceding individual while the number of eliminated individuals is less than R.
		// Let r be the number of eliminated individuals. Next, if r < R, eliminate R - r individuals in the order of
		// lowest fitness value.
		size_t old = m_Roulette.size();
		size_t R = (int)(m_Roulette.size() * (double)(100 - nEliminationPercentage) / 100.0);
		RemoveTwins();

		size_t r = old - m_Roulette.size();

		vector<CRouletteItem>::iterator it;
		while (r < R)
		{
			it = m_Roulette.begin();
			m_Roulette.erase(it);
			r++;
		}
	}

	// The selection algorithm
	void DoSelect(TGenome *&father, TGenome *&mother)
	{
		size_t a = gen_rand32() % m_Roulette.size();
		father = m_Roulette[a].m_pGenome;

		a = gen_rand32() % m_Roulette.size();
		mother = m_Roulette[a].m_pGenome;
	}
};


// ================================================================================================================================
//													class CRouletteSelection
//
// Creates a virtual roulette wheel, where each fitness-value is assigned a sector. The size of the sector is proportional
// to the fitness value. Therefore individuals with a higher fitness are selected more likely.
//
// Example:
// We have 5 individuals with the fitnesses: 1, 5, 7, 11, 15. So the sum of the fitnesses is 39.
// Individual 1 is assigned a probability range of 1/39									= 0.0000 - 0.0256
// Individual 2 is assigned a probability range of 1/39 + 5/39							= 0.0256 - 0.1538
// Individual 3 is assigned a probability range of 1/39 + 5/39 + 7/39					= 0.1538 - 0.3333
// Individual 4 is assigned a probability range of 1/39 + 5/39 + 7/39 + 11/39			= 0.3333 - 0.6154
// Individual 5 is assigned a probability range of 1/39 + 5/39 + 7/39 + 11/39 + 15/39	= 0.6154 - 1.0000
//
// Hint: Elitism should be turned off for this selection algorithm. The elite should not be over-emphasized.
// ================================================================================================================================
template<class TGenome>
class CRouletteSelection : public CSelectionBase<TGenome>
{
public:
	// compute selection probabilities
	void ComputeProbabilities()
	{
		double sum_probabilities = 0;

		for (vector<CRouletteItem>::iterator it = m_Roulette.begin(); it != m_Roulette.end(); it++)
		{
			it->m_dblProbability = sum_probabilities + (it->m_pGenome->GetFitness() / m_dblSumFitnesses);
			sum_probabilities = it->m_dblProbability;
		}

		sort(m_Roulette.begin(), m_Roulette.end());
	}

	void PreSelect(CPopulation<TGenome> *pPopulation)
	{
		BuildData(pPopulation);
		ComputeProbabilities();
	}

	TGenome *GetRandomGenome()
	{
		CRouletteItem item;

		item.m_dblProbability = genrand_real1();
		vector<CRouletteItem>::iterator it = lower_bound(m_Roulette.begin(), m_Roulette.end(), item);
		
		return it->m_pGenome;
	}

	// The selection algorithm requires that probabilities are computed and m_Roulette is sorted!
	void DoSelect(TGenome *&father, TGenome *&mother)
	{
		father = GetRandomGenome();
		mother = GetRandomGenome();
	}
};


// ================================================================================================================================
//													class CRankSelection
//
// Based on the roulette selection, but assigns each individual a fitness value based on its position in the population array,
// which is sorted by fitness first. This smoothes the selection probabilities.
//
// Hint: Elitism should be turned off for this selection algorithm. The elite should not be over-emphasized.
// ================================================================================================================================
template<class TGenome>
class CRankSelection : public CRouletteSelection<TGenome>
{
public:
	void PreSelect(CPopulation<TGenome> *pPopulation)
	{
		BuildData(pPopulation);

		// Rank
		int n = 1;
		m_dblSumFitnesses = 0;
		for (vector<CRouletteItem>::iterator it = m_Roulette.begin(); it != m_Roulette.end(); it++)
		{
			it->m_pGenome->SetFitness(n);
			m_dblSumFitnesses += n;
			n++;
		}

		ComputeProbabilities();
	}
};


// ================================================================================================================================
//													class CQuadraticSelection
//
// Based on the roulette selection, but assigns each individual the square-root of its fitness value first.
// This smoothes the selection probabilities.
//
// Hint: Elitism should be turned off for this selection algorithm. The elite should not be over-emphasized.
// ================================================================================================================================
template<class TGenome>
class CQuadraticSelection : public CRouletteSelection<TGenome>
{
public:
	void PreSelect(CPopulation<TGenome> *pPopulation)
	{
		BuildData(pPopulation);

		m_dblSumFitnesses = 0;
		for (vector<CRouletteItem>::iterator it = m_Roulette.begin(); it != m_Roulette.end(); it++)
		{
			TGenome::TFitnessValueType val = (TGenome::TFitnessValueType)sqrt((double)it->m_pGenome->GetFitness());
			it->m_pGenome->SetFitness(val);
			m_dblSumFitnesses += val;
		}

		ComputeProbabilities();
	}
};


// ================================================================================================================================
//													class CVisualization
//
// Dummy visualization class, which does nothing.
// ================================================================================================================================
template<class TGenome>
class CVisualization
{
public:
	// is called once at startup of the genetic algorithm
	void Init() {}

	// is called for each new generation
	void Visualize(const TGenome &solution, bool solution_changed, int generation, double mean_value, CPopulation<TGenome> *population) {}

	// support for slowing down the visualization, implementors should sleep() msec milliseconds
	// before exiting the Visualize() method
	void SetDelay(int msec) {}
};


// ================================================================================================================================
//													class CGeneticAlgo
//
// The genetic algorithm.
// ================================================================================================================================
template<class TGenome, class TVisualization = CVisualization<TGenome>, class TSelection = CTournamentSelection<TGenome>>
class CGeneticAlgo
{
protected:
	TVisualization	m_Visualization;				// The visualization callback. Default is no visualization.
	bool			m_bEnableVisualization;			// Turn visualization on / off. Default is on.
	TSelection		m_Selection;					// The selection algorithm. Default is tournament selection.
	int				m_nCrossoverProbability;		// A value between 0 and 100 = probability in percent. Default is 60%.
	int				m_nMutationProbability;			// A value between 0 and 100 = probability in percent. Default is 1%.
	int				m_nElitismPercentage;			// A value between 0 and 100 = percentage of fittest individuals, which
													// survive unchanged. Default is 10%.
	unsigned int	m_nStableGenerations;			// If for this number of generations the fitness-value of the best found genome
													// is stable, then the algorithm will abort. Default is 200.
	unsigned int	m_nMaxGenerations;				// Maximum number of created generations. Default is 1000.

	unsigned int	m_nGenerations;					// This value can be read and holds the number of created generations
	double			m_dblMeanValue;					// Mean value of all fitnesses of the last created generation

	bool			m_bIsComputing;					// If true, the Compute() method is currently running
	bool			m_bCancelled;					// If true, the Compute() method will abort

	CPopulation<TGenome>	*m_pPopulation;			// The population
	TGenome					m_BestSolution;			// Current best solution found
	int						m_nBSGeneration;		// The generation number where the best solution was found

public:
	// The parameter "count" is the number of individuals in the population
	CGeneticAlgo(size_t count = 128)
	{
		m_nCrossoverProbability	= 60;		// 60%
		m_nMutationProbability	= 1;		//  1%
		m_nElitismPercentage	= 10;		// 10%
		m_nStableGenerations		= 200;
		m_nMaxGenerations		= 1000;

		m_bEnableVisualization	= true;

		m_nGenerations	= 0;
		m_dblMeanValue	= 0;

		m_bIsComputing	= false;
		m_bCancelled	= false;

		m_pPopulation = new CPopulation<TGenome>(count);
		m_nBSGeneration = 0;
	}

	~CGeneticAlgo()
	{
		delete m_pPopulation;
	}

	TGenome			GetSolution() const { return m_BestSolution; }
	int				GetBSGeneration() const { return m_nBSGeneration; }		// the generation, where the best solution was found
	size_t			GetPopulationSize() const { return m_pPopulation->size(); }
	void			SetPopulationSize(size_t size) { m_pPopulation->SetSize(size); }

	TVisualization	&GetVisualization() { return m_Visualization; }
	void			SetEnableVisualization(bool val) { m_bEnableVisualization = val; }
	void			SetVisualizationDelay(int msec) { m_Visualization.SetDelay(msec); }

	int		GetCrossoverProbability() const { return m_nCrossoverProbability; }
	void	SetCrossoverProbability(int percent) { m_nCrossoverProbability = percent; }

	int		GetMutationProbability() const { return m_nMutationProbability; }
	void	SetMutationProbability(int percent) { m_nMutationProbability = percent; }

	int		GetElitismPercentage() const { return m_nElitismPercentage; }
	void	SetElitismPercentage(int val) { m_nElitismPercentage = val; }

	void	SetStableGenerations(unsigned int val) { m_nStableGenerations = val; }
	void	SetMaxGenerations(unsigned int val) { m_nMaxGenerations = val; }

	bool	GetIsComputing() const { return m_bIsComputing; }

	void	SetCancel() { m_bCancelled = true; }

	unsigned int	GetGenerations() const { return m_nGenerations; }
	double			GetMeanValue() const { return m_dblMeanValue; }

	// Compute - The genetic algorithm itself
	TGenome Compute()
	{
		m_bIsComputing = true;
		m_BestSolution.Reset();
		m_nBSGeneration = 0;		// the generation number where the best solution was found
		m_nGenerations = 0;
		unsigned int stable_generation_count = 0;

		// Compute the number of elite individuals. div 2, because at each loop two individuals are chosen (father and mother)
		int elite_elements = (int)(m_pPopulation->size() * ((double)m_nElitismPercentage / 100.0)) / 2;

		m_Visualization.Init();

		// Init Population with random genomes
		// OpenMP does only allow variables of type integer in for-loops
		#pragma omp parallel for
		for (int i = 0; i < (int)m_pPopulation->size(); i++)
			m_pPopulation->GetGenome(i).Randomize();

		do
		{
			m_nGenerations++;

			// Compute the fitness value for each genome, and find the best fitness value.
			// To avoid critical sections, which slow down performance a lot, we keep the best found genome separately for each thread.
			vector<TGenome> thread_best;
			thread_best.resize(omp_get_num_procs());
			double sum = 0;

			// OpenMP does only allow variables of type integer in for-loops
			#pragma omp parallel for reduction(+ : sum)
			for (int i = 0; i < (int)m_pPopulation->size(); i++)
			{
				double fi = m_pPopulation->GetGenome(i).ComputeFitness();
				sum += fi;

				int tn = omp_get_thread_num();
				if (fi > thread_best[tn].GetFitness())
					thread_best[tn] = m_pPopulation->GetGenome(i);
			}

			if (m_bCancelled)
				return m_BestSolution;

			// find best genome of all threads
			TGenome current_best;
			for (vector<TGenome>::iterator it = thread_best.begin(); it != thread_best.end(); it++)
			{
				if (it->GetFitness() > current_best.GetFitness())
					current_best = *it;
			}

			// compute the fitness mean value of the current generation
			m_dblMeanValue = sum / m_pPopulation->size();

			if (current_best.GetFitness() > m_BestSolution.GetFitness())
			{
				// we found a better solution in the current generation
				m_BestSolution = current_best;
				m_nBSGeneration = m_nGenerations;
				stable_generation_count = 0;
			}
			else
			{
				// we did not find a better solution in the current generation
				stable_generation_count++;
			}

			if (stable_generation_count < m_nStableGenerations && m_nGenerations < m_nMaxGenerations)
			{
				// create new generation
				m_Selection.PreSelect(m_pPopulation);	// let the chosen selection algorithm perform its initialization

				// visualization
				if (m_bEnableVisualization)
					m_Visualization.Visualize(m_BestSolution, stable_generation_count == 0, m_nGenerations, m_dblMeanValue, m_pPopulation);

				TGenome *father, *mother;
				TGenome *child1, *child2;

				// compute the new generation
				// OpenMP does only allow variables of type integer in for-loops
				#pragma omp parallel for private(father, mother, child1, child2)
				for (int i = 0; i < (int)m_pPopulation->size() / 2; i++)
				{
					// Selection
					if (i < elite_elements)
						m_Selection.DoSelectElite(i, father, mother);
					else
						m_Selection.DoSelect(father, mother);

					child1 = &m_pPopulation->GetNewGenome(i * 2);
					child2 = &m_pPopulation->GetNewGenome(i * 2 + 1);

					// Crossover, with a likelihood of m_nCrossoverProbability
					int n = gen_rand32() % 100;
					if (n < m_nCrossoverProbability)
					{
						// Create two children by crossover.
						child1->Crossover(*father, *mother);
						child2->Crossover(*mother, *father);
					}
					else
					{
						// No crossover, copy parents to new generation
						*child1 = *father;
						*child2 = *mother;
					}

					// Mutation, first child, with a likelihood of m_nMutationProbability
					int k = gen_rand32() % 100;
					if (k < m_nMutationProbability)
						child1->Mutate();

					// Mutation, second child, with a likelihood of m_nMutationProbability
					k = gen_rand32() % 100;
					if (k < m_nMutationProbability)
						child2->Mutate();
				}

				// swap population arrays, so the new generation becomes the current generation and vice versa
				m_pPopulation->NewGeneration();
			}
		}
		while (stable_generation_count < m_nStableGenerations && m_nGenerations < m_nMaxGenerations);	// abort-condition

		m_bIsComputing = false;

		return m_BestSolution;
	};
};
