//
//	ga_app.cpp
//	==========
//
//	Genetic Application Framework for Windows
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#include "stdafx.h"
#include "resource.h"

#include <vector>
#include <list>
#include <algorithm>
using namespace std;

#include <limits.h>
#include <float.h>
#include <time.h>
#include <math.h>
#include "./MersenneTwister/SFMT.h"	// "good" random numbers

#ifndef _DEBUG
    #include <omp.h>		// OpenMP support
#endif

#include "assert.h"
#include "Splash.h"
#include "galib.h"
#include "ga_app.h"


// ================================================================================================================
//													Globals
// ================================================================================================================
CGaAppInterface *gpGaAppInterface = NULL;


// ================================================================================================================
//													Assert()
// ================================================================================================================
void Assert(long arg, long line, const char *file)
{
	if (!arg)
	{
		char msg[256];
		sprintf(msg, "assertion failed! file %s, line %ld\n", file, line);
		MessageBoxA(NULL, msg, "Assert", MB_OK);
	}
}


// ================================================================================================================
//													GaCreateWindow()
// ================================================================================================================
HWND GaCreateWindow(WNDCLASSEX &wcex, const TCHAR *szTitle)
{
	HWND hWnd = CreateWindow(wcex.lpszClassName, szTitle, WS_OVERLAPPEDWINDOW,
							CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, NULL, NULL, wcex.hInstance, NULL);
	if (!hWnd)
		return NULL;

	ShowWindow(hWnd, TRUE);
	UpdateWindow(hWnd);

	return hWnd;
}


// ================================================================================================================
//												MainWndProc()
// ================================================================================================================
LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch (message)
	{
	case WM_CREATE:
		gpGaAppInterface->MainWndCreate();
		break;

	case WM_COMMAND:
		if (!gpGaAppInterface->MainWndCommand(LOWORD(wParam), HIWORD(wParam)))
			return DefWindowProc(hWnd, message, wParam, lParam);
		return 0;

	case WM_PAINT:
		hdc = BeginPaint(hWnd, &ps);
		gpGaAppInterface->MainWndPaint(hWnd, hdc);
		EndPaint(hWnd, &ps);
		break;

	case WM_DESTROY:
		gpGaAppInterface->MainWndDestroy();
		PostQuitMessage(0);
		break;

	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}

	return 0;
}


// ================================================================================================================
//												MeanWndProc()
// ================================================================================================================
LRESULT CALLBACK MeanWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch (message)
	{
	case WM_PAINT:
		hdc = BeginPaint(hWnd, &ps);
		gpGaAppInterface->MeanWndPaint(hWnd, hdc);
		EndPaint(hWnd, &ps);
		break;

	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}

	return 0;
}


// ================================================================================================================
//												PopulationWndProc()
// ================================================================================================================
LRESULT CALLBACK PopulationWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch (message)
	{
	case WM_PAINT:
		hdc = BeginPaint(hWnd, &ps);
		gpGaAppInterface->PopWndPaint(hWnd, hdc);
		EndPaint(hWnd, &ps);
		break;

	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}

	return 0;
}


// ================================================================================================================
//											DispatchAllWindowMessages()
//
// Keeps the Windows mnessage queue alive, whilst performing computations.
// Must be called periodically. 
// ================================================================================================================
BOOL DispatchAllWindowMessages()
{
	MSG msg;

	while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
	{
		if (msg.message == WM_QUIT)	// WM_QUIT is posted by the main-application via PostQuitMessage()
		{
			return TRUE;	// cancel
		}

		// No cancel, dispatch message:
		PeekMessage(&msg, NULL, 0, 0, PM_REMOVE);
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	return FALSE;
}


// ================================================================================================================
//													main()
// ================================================================================================================
int APIENTRY _tWinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPTSTR    lpCmdLine,
                     int       nCmdShow)
{
	UNREFERENCED_PARAMETER(hPrevInstance);
	UNREFERENCED_PARAMETER(lpCmdLine);

	init_gen_rand((uint32_t)time(0));	// init mersenne twister

	MSG msg;
	HACCEL hAccelTable;

	// Initialize global strings

	// Perform application initialization:
	InitApplication(hInstance);

	hAccelTable = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDC_MENU));

	// Main message loop:
	while (GetMessage(&msg, NULL, 0, 0))
	{
		if (!TranslateAccelerator(msg.hwnd, hAccelTable, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	return (int) msg.wParam;
}
