﻿using Actien.Data;
using System;
using System.Collections.ObjectModel;
using System.Linq;
using System.Runtime.InteropServices;
using Windows.Foundation;
using Windows.UI.ApplicationSettings;
using Windows.UI.Popups;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using Windows.UI.Xaml.Navigation;
using Actien.UI;
using System.Collections.Generic;
using Windows.Storage;

// Die Elementvorlage "Leere Seite" ist unter http://go.microsoft.com/fwlink/?LinkId=234238 dokumentiert.

namespace Actien
{
    /// <summary>
    /// Eine leere Seite, die eigenständig verwendet werden kann oder auf die innerhalb eines Rahmens navigiert werden kann.
    /// </summary>
    public sealed partial class MainPage : Page
    {
        public ObservableCollection<StockQuote> Quotes
        {
            get; private set;
        }

        public MainPage()
        {
            Quotes = new ObservableCollection<StockQuote>();
            LoadSettings();
            this.InitializeComponent();

            SettingsPane.GetForCurrentView().CommandsRequested += MainPage_CommandsRequested;
            ApplicationData.Current.DataChanged += new TypedEventHandler<ApplicationData, object>(RoamingDataChanged);
        }

        private void RoamingDataChanged(ApplicationData sender, object args)
        {
            LoadSettings();
            UpdateQuotes(true);
        }

        /// <summary>
        /// Wird aufgerufen, wenn diese Seite in einem Rahmen angezeigt werden soll.
        /// </summary>
        /// <param name="e">Ereignisdaten, die beschreiben, wie diese Seite erreicht wurde. Die
        /// Parametereigenschaft wird normalerweise zum Konfigurieren der Seite verwendet.</param>
        protected override async void OnNavigatedTo(NavigationEventArgs e)
        {
            if (e.NavigationMode == NavigationMode.New)
            {
                try
                {
                    QuoteQuery query = new QuoteQuery(_symbols);
                    var quotes = await query.GetAnswerAsync();
                    foreach (var q in quotes)
                    {
                        Quotes.Add(q);
                        q.LoadHistory(DateTime.Today.AddDays(-_historyDays), DateTime.Today);
                    }
                }
                catch (Exception ex)
                {
                    MessageDialog md = new MessageDialog(ex.Message, "Fehler");
                    md.ShowAsync();
                }
            }
            else
            {
                UpdateQuotes(true);
            }
        }

        private void UpdateQuotes(bool updateHistory = false)
        {
            foreach (var quote in Quotes)
            {
                quote.LoadData();
                if(updateHistory)
                    quote.LoadHistory(DateTime.Today.AddDays(-_historyDays), DateTime.Today);
            }
        }

        List<string> _symbols;
        int _historyDays;

        private void LoadSettings()
        {
            var settings = ApplicationData.Current.RoamingSettings;

            int nSymbols = 0;
            if (settings.Values.ContainsKey("Symbols"))
                nSymbols = (int)settings.Values["Symbols"];
            if (nSymbols > 0)
            {
                _symbols = new List<string>(nSymbols);
                for (int i = 0; i < nSymbols; i++)
                {
                    string key = string.Format("Symbol{0:D3}", i);
                    _symbols.Add(settings.Values[key] as string);
                }
            }
            else
                _symbols = new List<string>() { "MSFT", "AAPL", "GOOG", "FB", "VOW.DE", "DAI.DE", "DBK.DE", "DTE.DE", "LHA.DE", "005930.KS", "^GDAXI", "^DJI", "^NDX", "^HSI", "^N225" };
            
            if (settings.Values.ContainsKey("HistoryDays"))
                _historyDays = (int)settings.Values["HistoryDays"];
            else
                _historyDays = 30;
        }

        private void SaveSettings()
        {
            var settings = ApplicationData.Current.RoamingSettings;
            List<string> symbols = (from sq in Quotes
                                    select sq.Symbol).ToList();
            settings.Values["Symbols"] = symbols.Count;
            for (int i = 0; i < symbols.Count; i++)
            {
                string key = string.Format("Symbol{0:D3}", i);
                settings.Values[key] = symbols[i];
            }
            settings.Values["HistoryDays"] = _historyDays;
        }

        private void Refresh_Click(object sender, RoutedEventArgs e)
        {
            UpdateQuotes(false);
            bottomAppBar.IsOpen = false;
        }

        private void Sort_Click(object sender, RoutedEventArgs e)
        {
            var order = Quotes.OrderBy(q => q.Symbol).ToList();
            Quotes.Clear();
            foreach (var q in order)
            {
                Quotes.Add(q);
            }
            bottomAppBar.IsOpen = false;
            SaveSettings();
        }

        private void Settings_Click(object sender, RoutedEventArgs e)
        {
            SettingsPane.Show();
        }

        private void Remove_Click(object sender, RoutedEventArgs e)
        {
            var selectedStock = stockList.SelectedItem as StockQuote;
            if (selectedStock != null)
                Quotes.Remove(selectedStock);
            bottomAppBar.IsOpen = false;
            SaveSettings();
        }

        private async void Add_Click(object sender, RoutedEventArgs e)
        {
            MessageDialog md = new MessageDialog("Funktion noch nicht implementiert.", "Sorry");
            await md.ShowAsync();
            bottomAppBar.IsOpen = false;
        }

        private void bottomAppBar_Opened(object sender, object e)
        {
            bool bStockSelected = stockList.SelectedItem != null;
            btnRemove.Visibility = bStockSelected ? Visibility.Visible : Visibility.Collapsed;
        }

        Popup _settingsPopup;
        double _settingsWidth = 346;

        void MainPage_CommandsRequested(SettingsPane sender, SettingsPaneCommandsRequestedEventArgs args)
        {
            SettingsCommand cmd = new SettingsCommand("stockoptions", "Optionen", (x) =>
            {
                Rect windowBounds = Window.Current.Bounds;

                _settingsPopup = new Popup();
                _settingsPopup.Closed += OnPopupClosed;
                Window.Current.Activated += OnWindowActivated;
                _settingsPopup.IsLightDismissEnabled = true;
                _settingsPopup.Width = _settingsWidth;
                _settingsPopup.Height = windowBounds.Height;

                SettingsHolder options = new SettingsHolder();
                options.Quotes = this.Quotes;
                options.HistoryDays = _historyDays;
                StockSettings settingsPane = new StockSettings(options);
                settingsPane.Width = _settingsWidth;
                settingsPane.Height = windowBounds.Height;

                _settingsPopup.Child = settingsPane;
                _settingsPopup.SetValue(Canvas.LeftProperty, windowBounds.Width - _settingsWidth);
                _settingsPopup.SetValue(Canvas.TopProperty, 0);
                _settingsPopup.IsOpen = true;
            });

            args.Request.ApplicationCommands.Add(cmd);
        }

        private void OnWindowActivated(object sender, Windows.UI.Core.WindowActivatedEventArgs e)
        {
            if (e.WindowActivationState == Windows.UI.Core.CoreWindowActivationState.Deactivated)
            {
                _settingsPopup.IsOpen = false;
            }
        }


        void OnPopupClosed(object sender, object e)
        {
            StockSettings settingsPane = _settingsPopup.Child as StockSettings;
            if (settingsPane != null)
            {
                SettingsHolder options = settingsPane.Options;
                if (options.HistoryDays != _historyDays)
                {
                    _historyDays = options.HistoryDays;
                    UpdateQuotes(true);
                }
            }
            SaveSettings();

            Window.Current.Activated -= OnWindowActivated;
        }

    }
}
