/* syncbox.c (C) Oliver Diedrich / c't Magazin fuer Computertechnik */
/* This file is licensed under GPL v2, see file LICENSE */

#define _LARGEFILE64_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/fanotify.h>

#define MBYTE 1048576       /* read and write buffer size */
#define REMOTE_TO_LOCAL 1   /* direction of synchronization in sync_file() */
#define LOCAL_TO_REMOTE 0



int initialize(char *source, char *dest)
/* syncs data from source (host/directory) to dest (host/directory) */
/* source and dest must be given in rsync/ssh syntax */
/* newer files are never overwritten (--update) */
/* 1: ok, 0: error */
{
  char rsync[FILENAME_MAX] = "\0";
  int status;

  sprintf(rsync, "%s '%s' '%s'",
         "/usr/bin/rsync -rpgoDvz --update --backup --timeout=5",
         source, dest);
  status = system(rsync);

  if (-1 == status)
  {
    perror("system()");
    return 0;
  }
  else
  {
    status = WEXITSTATUS(status);
    if (0 == status) return 1;   /* no error */
    else
    {
      if (status >= 30) printf("rsync timed out\n");
      else printf("rsync error, exit status: %d\n", status);
      return 0;
    }
  }
}   /* initialize() */



int get_fname(int fd, char *fname, int basename)
/* places name for file descriptor fd in fname. If basename is false, */
/* fname contains full path name; else only file name */
/* Enough memory must be allocated for fname! */
/* 0: error, 1: ok */
{
  int len;
  char buf[FILENAME_MAX];
  char *c;
  
  sprintf(fname, "/proc/self/fd/%d", fd); /* link to local path name */
  len = readlink(fname, buf, FILENAME_MAX-1);   /* real path name is placed in buf */
  if (len == 0)
  {
    fname[0] = '\0';
    return 0;
  }
  buf[len] = '\0';
  if (basename)
  {
    c = buf + len;
    while (*c != '/') c--;   /* find last / in buf */
    c++;                     /* points to file name without directory */
  }
  else c = buf;
  sprintf(fname, "%s", c);
  return 1;
}   /* get_fname() */



int cp_to_tmp(int fd, char *tmpname)
/* copies content of file descriptor fd to a temp file "/tmp/PID.syncbox" */
/* temp file name is put in tmpname which must be large enough */
/* 0: error, 1: ok */
{
  char buf[MBYTE];
  int tmp, i;
  struct stat st;

/* create temporary file in /tmp */
  sprintf(tmpname, "/tmp/%d.syncbox", (int) getpid());
  tmp = open(tmpname, O_WRONLY|O_CREAT, S_IRWXU);
  if (-1 == tmp)
  {
    perror("open()");
    remove(tmpname);
    return 0;
  }

/* copy fd to tmp */
  while ( (i = read(fd, buf, MBYTE)) > 0)
  {
    if ( (i = write(tmp, buf, i)) <= 0)
    {
      perror("write()");
      close(tmp);
      remove(tmpname);
      return 0;
    }
  }
  if (-1 == i)
  {
    perror("read()");
    close(tmp);
    remove(tmpname);
    return 0;
  }

/* preserve owner and group id of original file */
  fstat(fd, &st);
  fchown(tmp, st. st_uid, st.st_gid);
  close(tmp);
  return 1;
}   /* cp_to_tmp() */



int cp_from_tmp(int fd, char *tmpname)
/* copies content from file tmpname to file descriptor fd */
/* 0: error, 1: ok */
{
  char buf[MBYTE];
  int tmp, i;

  tmp = open(tmpname, O_RDONLY);
  if (-1 == tmp)
  {
    perror("open()");
    return 0;
  }
  if (0 != lseek(fd, 0, SEEK_SET))
  {
    perror("lseek()");
    close(tmp);
    return 0;
  }
  if (-1 == ftruncate(fd, 0))
  {
    perror("ftruncate()");
    close(tmp);
    return 0;
  }

  while ( (i = read(tmp, buf, MBYTE)) > 0)
  {
    if ( (i = write(fd, buf, i)) <= 0)
    {
      perror("write()");
      close(tmp);
      return 0;
    }
  }
  if (-1 == i)
  {
    perror("read()");
    close(tmp);
    return 0;
  }
  return 1;
}   /* cp_from_tmp() */



int sync_file(int fd, char *remote, int from)
/* fd: open file descriptor of file to sync; remote: remote directory name  */
/* from: direction (1: sync remote to local, 0: sync local to remote)       */
/* Because we can't sync to a file which is watched (the open() call from   */
/* rsync would be blocked by fanotify, leading to a deadlock), fd content   */
/* must be copied to a temporary file, synced, and copied back to fd        */
/* return value: 0: error (error message is printed); else: ok              */
{
  int i;
  char tmpname[FILENAME_MAX];      /* temporary file to wich fd is copied */
  char fname[FILENAME_MAX];        /* name of file to sync (without path) */
  char rsync[FILENAME_MAX];        /* rsync call for system() */

/* determine name of file to sync */
  if (!get_fname(fd, fname, 1))
  {
    printf("can't determine file to sync\n");
    return 0;
  }

/* copy file content to temporary file */
  if (!cp_to_tmp(fd, tmpname))
  {
    printf("copy error: %s to %s\n", fname, tmpname);
    return 0;
  }

/* sync content of temporary file from remote */
  if (REMOTE_TO_LOCAL == from)
  {
    sprintf(rsync, "%s '%s%s' '%s'",
           "/usr/bin/rsync -pgoDqcz --timeout=3",
           remote, fname, tmpname);
  }
  else if (LOCAL_TO_REMOTE == from)
  {
    sprintf(rsync, "%s '%s' '%s%s'",
           "/usr/bin/rsync -pgoDqcz --timeout=3",
           tmpname, remote, fname);
  }
  else
  {
    printf("sync_file called wrong parameter %d\n", from);
    remove(tmpname);
    return 0;
  }
    
  printf(">>> calling %s\n", rsync); fflush(stdout);
  i = system(rsync);
  if (-1 == i)
  {
    perror("system()");
    remove(tmpname);
    return 0;
  }

  i = WEXITSTATUS(i);
  if (i > 0)    /* rsync error */
  {
    if (i >= 30) printf("rsync timed out\n");
    else printf("rsync error, exit status: %d\n", i);
    remove(tmpname);
    return 0;
  }

/* copy updated temporary file content back to local file */
  if (!cp_from_tmp(fd, tmpname))
  {
    printf("copy error: %s to %s\n", tmpname, fname);
    remove(tmpname);
    return 0;
  }

  remove(tmpname);
  return 1;
}   /* sync_from() */



void usage(char *argv0)
{
  printf("%s -d directory -r user@remote:path\n", argv0);
  printf("  -d	local directory to sync\n");
  printf("  -r	remote host in ssh syntax, e.g. rsync@syncbox.my.net:\n");
   _exit(1);
}



int main(int argc, char** argv)
{
  char local[FILENAME_MAX] = "\0";              /* local directory to watch */
  char remote[FILENAME_MAX] = "\0"; /* remote end of sync (USER@HOST:[DIR]) */
  int fanotify_fd;           /* file descriptor returned by fanotify_init() */
  struct fanotify_event_metadata event;         /* structures read from and */
  struct fanotify_response response;              /* written to fanotify_fd */
  u_int64_t mask = FAN_OPEN_PERM | FAN_CLOSE_WRITE | FAN_EVENT_ON_CHILD;
                                          /* event mask for fanotify_mark() */
  char fname[FILENAME_MAX];                         /* name of file to sync */
  int i;
  char *c;
 
/* check command line arguments */
  while ((i = getopt(argc, argv, "d:r:")) != -1)
  {
    switch (i)
    {
      case 'd': realpath(optarg, local); break;
      case 'r': strncpy(remote, optarg, 4094); break;
      default: usage(argv[0]);
    }
  }
  if ('\0' == local[0]) usage(argv[0]);
  if ('\0' == remote[0]) usage(argv[0]);

/* remote must be suitable to be concatenated with a filename and give a  */
/* usable remote host/path name for rsync; so we add a '/' if it does not */
/* end with a '/' or ':' ("user@host:" without directory) */
  c = remote;
  while (*c != '\0') c++;
  c--;
  if ( (*c != ':') && (*c != '/') )
  {
    c++;
    *c = '/';
    c++;
    *c = '\0';
  }

/* local must end with '/' to ensure that directory content is sync'ed in */
/* initialize */
  c = local;
  while (*c != '\0') c++;
  *c = '/';
  c++;
  *c = '\0';

  printf("local directory: %s, remote: %s\n", local, remote);

/* we can't sync /tmp because temporary files are created there */
  if (0 == strncmp(local, "/tmp", 4))
  {
    printf("directory '/tmp' can't be synced, sorry\n");
    return 1;
  }

/* initialize fanotify */
  fanotify_fd = fanotify_init(FAN_CLASS_PRE_CONTENT, O_RDWR);
  if (fanotify_fd < 0)
  {
    perror("fanotify_init()");
    return 1;
  }

/* transfer local files to remote host, except they are newer on remote */
  if (!initialize(local, remote)) return 1;
/* transfer files from remote host to local directory, except they are */
/* newer in the local directory */
  if (!initialize(remote, local)) return 1;

/* mark local directory */
  if (fanotify_mark(fanotify_fd, FAN_MARK_ADD, mask, AT_FDCWD, local) != 0)
  {
    perror("fanotify_mark()");
    return 1;
  }

/* fanotify event processing */
  i = 0;
  while (read(fanotify_fd, &event, sizeof(event)) == sizeof(event))
  {
    i++;
    if (event.fd >= 0)
    {
      get_fname(event.fd, fname, 0);
      printf(">>> event #%d: vers %hu, mask 0x%02lX, fd %d (%s), pid %d\n", 
             i, event.vers, (unsigned long) event.mask, event.fd, 
    	     fname, event.pid);
      if (event.mask & FAN_OPEN_PERM)        /* snyc local file from remote, */
      {                                      /* than allow open              */
        sync_file(event.fd, remote, REMOTE_TO_LOCAL); /* we ignore the return*/
        response.fd = event.fd;                       /* value and always    */
        response.response = FAN_ALLOW;                /* allow access        */
        if (write(fanotify_fd, &response, sizeof(response)) < 0) 
         perror("write(fanotify_fd)");
      }
      if (event.mask & FAN_CLOSE_WRITE)         /* sync local file to remote */
      {
        sync_file(event.fd, remote, LOCAL_TO_REMOTE);
      }        
      close(event.fd);   /* else file descriptors are eaten up */
    }
  }
  close(fanotify_fd);
  return 0;
}
