// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// Alle Rechte vorbehalten.
// $Id: vector.h 03d56a8f9ad2 2011/10/24 08:02:28 Oliver Lau <oliver@von-und-fuer-lau.de> $

#ifndef __VECTOR_H_
#define __VECTOR_H_

#include "helper.h"

class Point;

/// Die Klasse Vector reprsentiert einen dreidimensionalen Vektor.
class Vector {
public:
    /// Konstruktor.
    Vector(float u = 0.0f, float v = 0.0f, float w = 0.0f) : mU(u), mV(v), mW(w)
    { /* ... */ }

    /// Konstruktor.
    Vector(const Point&, const Point&);

    /// Copy-Konstruktor.
    Vector(const Vector& o) : mU(o.u()), mV(o.v()), mW(o.w())
    { /* ... */ }

    /// Skalarprodukt aus zwei Vektoren bilden.
    friend float operator*(const Vector&, const Vector&);

    /// Einen Vektor von einem anderen abziehen.
    friend Vector operator-(const Vector&, const Vector&);

    /// Zwei Vektoren addieren.
    friend Vector operator+(const Vector&, const Vector&);

    /// Vektor durch Gleitkommazahl teilen.
    friend Vector operator/(const Vector&, float);

    /// Vektor mit Gleitkommazahl multiplizieren.
    friend Vector operator*(const Vector&, float);
    friend Vector operator*(float, const Vector&);

    /// u zurckgeben.
    /// @return u
    inline float u(void) const { return mU; }

    /// v zurckgeben.
    /// @return v
    inline float v(void) const { return mV; }

    /// w zurckgeben.
    /// @return w
    inline float w(void) const { return mW; }

    /// Lnge des Vektors als doppeltgenaue Gleitkommazahl zurckgeben.
    /// @return Lnge
    inline operator float() { return this->length(); }

    /// Lnge des Vektors als Ganzzahl zurckgeben.
    /// @return Lnge
    inline operator int() { return (int)this->length(); }

    /// Lnge des Vektors zurckgeben.
    /// @return Lnge
    float length(void) const;

    /// Normalisierten Vektor zurckgeben.
    /// @return normalisierter Vektor
    Vector normalized(void) const;

    /// Prfen, ob alle Komponenten des Vektor auf 0 stehen.
    /// @return true, wenn Vektor gleich Null
    inline bool isNull(void) const {
        return (mU == 0.0f) && (mV == 0.0f) && (mW == 0.0f);
    }

    float angleTo(const Vector& vec) const;

private:
    /// u
    float mU;

    /// v
    float mV;

    /// w
    float mW;

};

#endif // __VECTOR_H_
