// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: point.h eb870b13e595 2011/08/31 11:17:52 Oliver Lau <oliver@von-und-fuer-lau.de> $

#ifndef __POINT_H_
#define __POINT_H_

#include "helper.h"
#include <QtCore/QRect>
#include <QtCore/QDebug>

class Line;
class Vector;

class Point {
public:
    Point(float x = 0.0f, float y = 0.0f, float z = 0.0f, int state = 0) : mX(x), mY(y), mZ(z), mState(state)
    { /* ... */ }

    Point(const Point& o) : mX(o.x()), mY(o.y()), mZ(o.z()), mState(o.state())
    { /* ... */ }

    inline float x(void) const { return mX; }
    inline float y(void) const { return mY; }
    inline float z(void) const { return mZ; }
    inline int state(void) const { return mState; }

    /// Einen Punkt von einem anderen subtrahieren.
    /// Das Ergebnis der Subtraktion ist ein Vektor.
    /// @param p1 Punkt, von dem p2 abgezogen werden soll
    /// @param p2 Punkt, der von p1 abgezogen werden soll
    /// @return Vektor, der die Differenz zwischen den beiden Punkten reprsentiert.
    friend Vector operator-(const Point& p1, const Point& p2);

    /// Einen Vektor von einem Punkt subtrahieren.
    /// Das Ergebnis der Subtraktion ist ein Punkt.
    /// @param p Punkt, von dem abgezogen werden soll.
    /// @param v Vektor, der abgezogen werden soll
    /// @return Punkt, der die Differenz reprsentiert.
    friend Point operator-(const Point& p, const Vector& v);

    /// Einen Vektor auf einen Punkt addieren.
    /// Das Ergebnis der Addition ist ein Punkt.
    /// @param p zu addierender Punkt
    /// @param v zu addierender Vektor
    /// @return Punkt, der die Summe reprsentiert.
    friend Point operator+(const Point& p, const Vector& v);

    /// Zwei Punkte addieren.
    /// Das Ergebnis der Addition ist ein Punkt.
    /// @param p1 zu addierender Punkt
    /// @param p2 zu addierender Punkt
    /// @return Punkt, der die Summe reprsentiert.
    friend Point operator+(const Point& p1, const Point& p2);

    /// Einen Punkt mit einem Skalar multiplizieren.
    /// Das Produkt ist ein Punkt.
    /// @param p Punkt
    /// @param c Skalar
    /// @return Punkt, der die Summe reprsentiert.
    friend Point operator*(const Point& p, float c);

    /// Einen Punkt mit einem Skalar multiplizieren.
    /// Das Produkt ist ein Punkt.
    /// @param c Skalar
    /// @param p Punkt
    /// @return Punkt, der das Produkt reprsentiert.
    friend Point operator*(float c, const Point& p);

    /// Einen Punkt durch einen Skalar teilen.
    /// Das Produkt ist ein Punkt.
    /// @param p Punkt
    /// @param c Skalar
    /// @return Quotient aus Punkt und Skalar.
    friend Point operator/(const Point& p, float c);

    /// Zwei Punkte miteinander vergleichen.
    /// @param p1 zu vergleichender Punkt
    /// @param p2 zu vergleichender Punkt
    /// @return true, wenn Punkte identisch sind, sonst false
    friend bool operator==(const Point& p1, const Point& p2);

    /// Zwei Punkte miteinander vergleichen.
    /// @param p1 zu vergleichender Punkt
    /// @param p2 zu vergleichender Punkt
    /// @return true, wenn Punkte sich unterscheiden, sonst false
    friend bool operator!=(const Point& p1, const Point& p2);

    /// Einen Vektor von einem Punkt abziehen.
    /// Das Ergebnis der Subtraktion ist ein Punkt.
    /// @param v zu subtrahierender Vektor
    /// @return Punkt, der die Differenz reprsentiert
    Point& operator-=(const Vector& v);

    /// Einen Vektor auf einen Punkt addieren.
    /// Das Ergebnis der Addition ist ein Punkt.
    /// @param v zu addierender Vektor
    /// @return Punkt, auf den die Spitze des Vektorpfeils zeigt.
    Point& operator+=(const Vector& v);

    /// Einen Punkt mit einem Skalar multiplizieren.
    /// Das Produkt ist ein Punkt.
    /// @param c Skalar
    /// @return Punkt, der die Summe reprsentiert.
    Point& operator*=(float c);

    /// Einen Punkt durch einen Skalar teilen.
    /// Das Produkt ist ein Punkt.
    /// @param c Skalar
    /// @return Quotient aus Punkt und Skalar.
    Point& operator/=(float c);

    bool isInsideSphere(const Point& center, float radius) const;

    /// Den senkrechten Abstand des Punktes zu der gedachten Geraden
    /// durch zwei andere Punkte berechnen.
    /// @param p1 Erster Punkt, durch den die gedachte Gerade verluft.
    /// @param p2 Zweiter Punkt, durch den die gedachte Gerade verluft.
    /// @return Senkrechter Abstand
    float perpendicularDistanceToLine(const Point& p1, const Point& p2) const;

    /// Den senkrechten Abstand des Punktes zu einer Geraden berechnen.
    /// @param line Gerade
    /// @return Senkrechter Abstand
    float perpendicularDistanceToLine(const Line& line) const;

    /// Den senkrechten Abstand des Punktes zu der gedachten Strecken
    /// zwischen zwei andere Punkten berechnen.
    /// @param p1 Erster Punkt, durch den die gedachte Strecke verluft.
    /// @param p2 Zweiter Punkt, durch den die gedachte Strecke verluft.
    /// @return Senkrechter Abstand
    float perpendicularDistanceToSegment(const Point& p1, const Point& p2) const;

    /// Den senkrechten Abstand des Punktes zu einer Strecke berechnen.
    /// @param line Strecke
    /// @return Senkrechter Abstand
    float perpendicularDistanceToSegment(const Line& line) const;

    inline bool isNull(void) const { return mX == 0.0f && mY == 0.0f && mZ == 0.0f; }

    /// X-Wert setzen.
    /// @param x X-Wert
    inline void setX(float x) { mX = x; }

    /// Y-Wert setzen.
    /// @param y Y-Wert
    inline void setY(float y) { mY = y; }

    /// Z-Wert setzen.
    /// @param z Z-Wert
    inline void setZ(float z) { mZ = z; }

private:
    float mX;
    float mY;
    float mZ;
    int mState;
};


#ifndef QT_NO_DEBUG_STREAM
QDebug operator<<(QDebug, const Point &);
#endif


#endif // __POINT_H_
