//-*- coding: utf8 -*-
// Copyright (c) 2011 Oliver Lau <ola@ct.de>, Heise Zeitschriften Verlag.
// Alle Rechte vorbehalten.

var Metronome = (function() {
    // Konstanten
    var TimerInterval = 40; // 1000/TimerInterval ms
    var PhiMax = Math.PI/5; // maximaler Ausschlagwinkel des Pendels
    var G = 3;              // "Schwerkraft"
    var PI2 = 2*Math.PI;    // 2 Pi
    var Epsilon = 3e-2;     // Winkel ums Pendel, in dem man es "einfangen" kann

    // Variablen
    var canvas = null; // das Canvas-Element
    var ctx = null; // 2D-Kontext des Canvas-Elements
    var t = 0; // aktuelle Zeit
    var t0 = 0; // Startzeitpunkt des Pendels
    var t00 = 0;
    var T = 0; // aktuelle Periodendauer des Pendels
    var phi0 = 0; // Startwinkel des Pendels
    var width = 0; // Breite des Canvas
    var height = 0; // Hhe des Canvas
    var counter = 0; // Anzahl der Nulldurchgnge seit dem Start des Metronoms
    var timer = null; // Timer-Objekt
    var audios = {};
    var sounds = {};
    var axis = {
            r: 8,
            x: undefined,
            y: undefined
            };
    var pendulum = {
            // Frequenz [Hz] des Pendels
            frequency: 26,
            // Winkel des Pendels
            phi: 0,
            lastPhi: 0,
            moving: false,
            length: 0,
            dragging: false,
            touching: false
            };
    var weight = {
            grabbing: false,
            pushing: false,
            dist: 0,
            width: 16,
            height: 16
            };
    var mouse = { isDown: false, pos: null };
    var sound = 'Clock'; // aktuell gewhlter Sound
    var startStopHandler = null;
    var countHandler = null;
    var freqChangedHandler = null;

    function getCursorPosition(e) {
        var pos = { x: 0, y: 0 };
        if (e.pageX != undefined && e.pageY != undefined) {
            pos.x = e.pageX;
            pos.y = e.pageY;
        }
        else {
            pos.x = e.clientX + document.body.scrollLeft + document.documentElement.scrollLeft;
            pos.y = e.clientY + document.body.scrollTop + document.documentElement.scrollTop;
        }
        pos.x -= canvas.offsetLeft;
        pos.y -= canvas.offsetTop;
        return pos;
    }

    function getPhiByPos(pos) {
        var x = pos.x-axis.x;
        var y = pos.y-axis.y;
        var z = Math.sqrt(x*x+y*y);
        return Math.asin(x/z);
    }

    function getDistByPos(pos) {
        var x = pos.x-axis.x;
        var y = pos.y-axis.y;
        return Math.sqrt(x*x+y*y);
    }

    function isPendulumTouched(pos) {
        return Math.abs(getPhiByPos(pos) - pendulum.phi) < Epsilon;
    }
    
    function isWeightTouched(pos) {
        var mPhi = getPhiByPos(pos);
        var mDist = getDistByPos(pos);
        return Math.abs(mPhi - pendulum.phi) < weight.width/8*Epsilon
            && mDist > weight.dist*pendulum.length
            && mDist < weight.dist*pendulum.length+weight.height; 
    }

    function onmousedown(e) {
        if (e.which == 1) {
            var pos = getCursorPosition(e);
            if (isPendulumTouched(pos))
                pendulum.dragging = true;
            weight.pushing = isWeightTouched(pos);
            mouse.isDown = true;
            mouse.pos = pos;
        }
    }
    
    function onmouseup(e) {
        if (e.which == 1) {
            var pos = getCursorPosition(e);
            var mPhi = getPhiByPos(pos);
            if (pendulum.dragging) {
                if (Math.abs(mPhi) < PhiMax/8) {
                    snapBack();
                }
                else {
                    phi0 = mPhi;
                    if (Math.abs(phi0) > PhiMax)
                        phi0 = PhiMax * phi0.sign();
                    Metronome.start();
                }
            }
            pendulum.dragging = false;
            weight.pushing = false;
            mouse.isDown = false;
        }
    }
    
    function onmousemove(e) {
        var pos = getCursorPosition(e);
        pendulum.touching = isPendulumTouched(pos);
        weight.grabbing = isWeightTouched(pos);         
        if (pendulum.dragging) {
            pendulum.phi = getPhiByPos(pos);
            if (Math.abs(pendulum.phi) > PhiMax)
               pendulum.phi = PhiMax * pendulum.phi.sign();
        }
        if (weight.pushing) {
            var d = getDistByPos(pos)-weight.height/2;
            if (d < 2)
                return;
            weight.dist = d/pendulum.length;
            Metronome.setFrequency(60/(Math.sqrt(weight.dist/G)*PI2));
        }
        if (mouse.isDown && pendulum.moving && pendulum.touching)
            Metronome.stop();
        updateCanvas();
    }
    
    function updateCanvas() {
        // Canvas-Hintergrund zeichnen
        var bkgGrad = ctx.createLinearGradient(0, 0, 0, height);
        bkgGrad.addColorStop(0, '#373E5C');
        bkgGrad.addColorStop(0.4, '#1A1D2B');
        ctx.fillStyle = bkgGrad;
        ctx.fillRect(0, 0, width, height);

        // Transformationsmatrix sichern
        ctx.save();
        
        // Ursprung des Koordinatensystems auf
        // die Mitte der Pendelachse verschieben
        ctx.translate(axis.x, axis.y);

        // Gehuse zeichnen
        ctx.beginPath();
        var contourGrad = ctx.createLinearGradient(-33, -pendulum.length, 100, 0);
        contourGrad.addColorStop(0, '#D0D0F4');
        contourGrad.addColorStop(1, '#7171BC');
        ctx.fillStyle = 'rgb(8, 24, 80)';
        ctx.strokeStyle = contourGrad;
        ctx.lineWidth = 2;
        ctx.moveTo(-100, axis.r+7);
        ctx.lineTo(100, axis.r+7);
        ctx.lineTo(35, -pendulum.length-5);
        ctx.quadraticCurveTo(0, -pendulum.length-15,
          -35, -pendulum.length-5);
        ctx.closePath();
        ctx.fill();
        ctx.stroke();

        // Glanz zeichnen
        var caseGrad = ctx.createLinearGradient(-33, -pendulum.length, 0, 0);
        caseGrad.addColorStop(0, '#424685');
        caseGrad.addColorStop(1, 'rgb(8, 24, 80)');
        ctx.fillStyle = caseGrad;
        ctx.beginPath();
        ctx.moveTo(-99, axis.r+6);
        ctx.quadraticCurveTo(12, -pendulum.length/2,
          34, -pendulum.length-5);
        ctx.quadraticCurveTo(0, -pendulum.length-13,
          -34, -pendulum.length-5);
        ctx.closePath();
        ctx.fill();

        // Skala zeichnen
        ctx.beginPath();
        ctx.lineWidth = 1;
        ctx.strokeStyle = '#6666ff';
        ctx.moveTo(0, -axis.r-5);
        ctx.lineTo(0, -axis.r-pendulum.length+15);
        ctx.fillStyle = '#6666ff';
        ctx.textBaseline = 'middle';
        var tickHeight = 0,
        f = 100,
        n = 0,
        scale = -G*pendulum.length/(PI2*PI2);
        while (tickHeight < pendulum.length && f > 0) {
            var t = 60/f;
            tickHeight = Math.round(t*t*scale);
            if (n%2) {
                ctx.moveTo(-12, tickHeight);
                ctx.lineTo(-1, tickHeight);
                ctx.textAlign = 'right';
                if (f%10 == 0) {
                    ctx.fillText(2*f, -10, tickHeight);
                    ++n;
                }
            }
            else {
                ctx.moveTo(1, tickHeight);
                ctx.lineTo(12, tickHeight);
                ctx.textAlign = 'left';
                if (f%10 == 0) {
                    ctx.fillText(2*f, 10, tickHeight);
                    ++n;
                }
            }
            f -= 5;
        }
        ctx.closePath();
        ctx.stroke();

        // ab hier alle Elemente um den Winkel des Pendels rotieren
        ctx.rotate(pendulum.phi);

        // Pendel zeichnen
        ctx.fillStyle = (pendulum.touching)? '#99f' : '#66f';
        ctx.fillRect(-2, 11, 4, -pendulum.length-11);
        ctx.strokeStyle = contourGrad;
        ctx.lineWidth = 1;
        ctx.strokeRect(-2, 11, 4, -pendulum.length-11);

        // Gewicht zeichnen
        ctx.beginPath();
        ctx.fillStyle = (weight.grabbing)? '#f00' : '#900';
        ctx.strokeStyle = (weight.grabbing)? '#f33' : '#c00';
        ctx.moveTo(-weight.width/1.4, -weight.dist*pendulum.length);
        ctx.lineTo(+weight.width/1.4, -weight.dist*pendulum.length);
        ctx.lineTo(+weight.width/2.6, -weight.dist*pendulum.length-weight.height);
        ctx.lineTo(-weight.width/2.6, -weight.dist*pendulum.length-weight.height);
        ctx.lineTo(-weight.width/1.4, -weight.dist*pendulum.length);
        ctx.closePath();
        ctx.fill();
        ctx.stroke();
        
        // Pendelachse zeichnen
        ctx.beginPath();
        ctx.fillStyle = '#339';
        ctx.lineWidth = 2;
        ctx.strokeStyle = '#66f';
        ctx.arc(0, 0, axis.r, 0, PI2, false);
        ctx.closePath();
        ctx.stroke();
        ctx.fill();
        ctx.restore();
    }
    
    function tick() {
        if (mouse.isDown && isPendulumTouched(mouse.pos)) {
            Metronome.hold();
            pendulum.touching = true;
            pendulum.dragging = true;
        }
        else {
            t = 1e-3 * (new Date().getTime() - t0);
            pendulum.phi = PhiMax * -Math.cos(Math.sqrt(G/weight.dist)*t);
            if (pendulum.phi.sign() != pendulum.lastPhi.sign())
                count();
            pendulum.lastPhi = pendulum.phi;
        }
        updateCanvas();
    }

    function updateStats() {
        var now = new Date().getTime();
        if (t00 == 0)
            t00 = now;
        if (typeof countHandler === 'function')
            countHandler(counter, now-t00);
    }
    
    function snapBack() {
        counter = 0;
        t00 = 0;
        pendulum.phi = 0;
        updateStats();
        updateCanvas();
    }

    function scaleBy(delta) {
        var d = delta/10;
        if ((d < 0 && G > d) || d > 0)
            G += d;
        Metronome.onchange();
    }
    
    function onwheel(event){
        var delta = 0;
        if (!event) event = window.event;
        if (event.wheelDelta) {
            delta = event.wheelDelta/120;
            if (window.opera)
                delta = -delta;
        }
        else if (event.detail) {
            delta = -event.detail/3;
        }
        if (delta) scaleBy(delta);
        if (event.preventDefault) event.preventDefault();
        event.returnValue = false;
    }
    
    function makeSoundUrl(name) {
        return name + '.mp3';
    }
    
    function count() {
        if (Metronome.audioAvailable) {
            audios[sound].currentTime = 0;
            audios[sound].play();
        }
        else { // fall-back to Flash sound
            sounds[sound].play();
        }
        updateStats();
        ++counter;
    }
    
    return {
        DefaultSound: 'Clock',
        AvailableSounds: { 'Clock': 'Uhr' },
        audioAvailable: (function() { return false; })(),
        init: function(params) {
            if (params != null && typeof params === 'object') {
                startStopHandler = params.startStopHandler;
                countHandler = params.countHandler;
                freqChangedHandler = params.freqChangedHandler;
            }
            if (Metronome.audioAvailable) {
                for (var k in Metronome.AvailableSounds) {
                    audios[k] = document.createElement('audio');
                    audios[k].src = makeSoundUrl(k);
                }
            }
            else { // fall-back to Flash sound
                soundManager.url = '/';
                soundManager.flashVersion = 9;
                soundManager.useFlashBlock = true;
                soundManager.onready(function(status) {
                    if (status.success) {
                        for (var k in Metronome.AvailableSounds)
                            sounds[k] = soundManager.createSound({
                                id: k,
                                url: makeSoundUrl(k),
                                autoLoad: true,
                                stream: false,
                            });
                    }
                });
            }
            canvas = $('metronom');
            canvas.style.display = 'block';
            attachEvent(canvas, 'mousedown', onmousedown);
            attachEvent(canvas, 'mouseup', onmouseup);
            attachEvent(canvas, 'mousemove', onmousemove);
            attachEvent(window, 'DOMMouseScroll', onwheel);
            attachEvent(window, 'mousewheel', onwheel);
            attachEvent(document, 'mousewheel', onwheel);
            attachEvent(window, 'scroll', onwheel);
            attachEvent(document, 'scroll', onwheel);
            ctx = canvas.getContext('2d');
            ctx.save();
            Metronome.setFrequency(pendulum.frequency);
            Metronome.onchange();
            updateStats();
        },
        onchange: function() {
            width = window.innerWidth;
            height = window.innerHeight;
            canvas.width = width;
            canvas.height = height;
            axis.x = Math.round(width/2);
            axis.y = height-axis.r-11;
            pendulum.length = Math.round(height-axis.r/2-30);
            T = 60/pendulum.frequency;
            weight.dist = T*T/(PI2*PI2)*G;
            if (pendulum.moving) {
                Metronome.stop();
                Metronome.start();
            }               
            updateCanvas();
        },
        start: function(angle) {
            if (typeof angle === 'number')
                phi0 = angle;
            pendulum.moving = true;
            var dt = Math.acos(-phi0/PhiMax)/Math.sqrt(G/weight.dist);
            t0 = new Date().getTime() + ((phi0 < 0)? -1000*dt : 1000*dt-T/2);
            t00 = 0;
            pendulum.lastPhi = phi0;
            counter = 0;
            timer = setInterval(tick, TimerInterval);
            if (typeof startStopHandler === 'function') 
                startStopHandler(pendulum.moving);
        },
        hold: function() {
            pendulum.moving = false;
            clearTimeout(timer);
            timer = null;
            if (typeof startStopHandler === 'function') 
                startStopHandler(pendulum.moving);
        },
        stop: function() {
            Metronome.hold();
            pendulum.phi = 0;
            phi0 = 0;
            counter = 0;
            t00 = 0;
            updateCanvas();
            updateStats();
            if (typeof startStopHandler === 'function') 
                startStopHandler(pendulum.moving);
        },
        restart: function() {
            if (pendulum.moving) {
                Metronome.stop();
                Metronome.start();
                Metronome.onchange();
            }
            updateCanvas();
        },
        startStop: function(angle) {
            if (pendulum.moving) Metronome.stop();
            else Metronome.start(angle);
        },
        running: function() {
            return pendulum.moving;
        },
        increaseFrequency: function() {
            Metronome.setFrequency(pendulum.frequency+0.5);
        },
        decreaseFrequency: function() {
            Metronome.setFrequency(pendulum.frequency-0.5);
        },
        setFrequency: function(f) {
            if (f < 0.5 || f > 120)
                return;
            var ff = parseInt(f);
            pendulum.frequency = (f-ff > 0.4999)? ff+0.5 : ff;
            if (typeof freqChangedHandler === 'function')
                freqChangedHandler(2*pendulum.frequency);
            Metronome.onchange();
        },
        selectSound: function(newSound) {
            sound = newSound;
        },
    };
})();
