// $Id$
// Copyright (c) 2008 Oliver Lau <ola@ctmagazin.de>, Heise Zeitschriften Verlag. Alle Rechte vorbehalten.

#define BENCHMARK

#ifdef BENCHMARK
#include <Windows.h>
#endif

#include <string>
#include <iostream>
#include <type_traits>
#include <cassert>
#include <vector>

using namespace std;
using namespace std::tr1;

typedef const char* STR;

const int SZ = 50000000;

#ifdef BENCHMARK
LARGE_INTEGER freq;
LARGE_INTEGER t0;
#endif


void start(string msg)
{
#ifdef BENCHMARK
	cout << "  " << msg << " ... ";
	QueryPerformanceCounter(&t0);
#endif
}


void stop(LONGLONG& ms)
{
#ifdef BENCHMARK
	LARGE_INTEGER t;
	QueryPerformanceCounter(&t);
	LONGLONG duration = 1000LL * (t.QuadPart - t0.QuadPart) / freq.QuadPart;
	ms += duration;
	cout << duration << " ms " << endl;
#endif
}


struct IntVal
{
    int x;
};


namespace clever {

    template <bool b>
    struct copier
    {
        template<typename I1, typename I2>
        static I2 do_copy(I1 first, I1 last, I2 out);
    };

    template <bool b>
    template<typename I1, typename I2>
    I2 copier<b>::do_copy(I1 first, I1 last, I2 out)
    {
        while (first != last)
            *out++ = *first++;
        return out;
    }

    template <>
    struct copier<true>
    {
        template<typename I1, typename I2>
        static I2* do_copy(I1* first, I1* last, I2* out)
        {
            memcpy(out, first, (last-first) * sizeof(I2));
            return out + (last - first);
        }
    };

    template<typename I1, typename I2>
    inline I2 copy(I1 first, I1 last, I2 out)
    {
        typedef typename remove_cv<typename std::iterator_traits<I1>::value_type>::type v1_t;
        typedef typename remove_cv<typename std::iterator_traits<I2>::value_type>::type v2_t;
        enum { can_opt = 
            is_same<v1_t, v2_t>::value      &&
            is_pointer<I1>::value           &&
            is_pointer<I2>::value           && 
            has_trivial_assign<v1_t>::value 
        };
        return clever::copier<can_opt>::do_copy(first, last, out);
    }

};


int main(int argc, char* argv[]) 
{ 
#ifdef BENCHMARK
	QueryPerformanceFrequency(&freq);
#endif

	const int TRIES = 5;
	
	LONGLONG t_copy = 0;
    LONGLONG t_memcpy = 0;
    LONGLONG t_copy_vector = 0;
    LONGLONG t_copy_array = 0;

    vector<IntVal> big(SZ);
    vector<IntVal> bigcopy(SZ);

    int* arr = new int[SZ];
    int* arrcopy = new int[SZ];

    STR msg = "hallo";

    cout << "char_traits<char>::length = " << char_traits<char>::length(msg) << endl;

    cout << "is_pointer<STR> = " << boolalpha << is_pointer<STR>::value << endl;
    cout << "is_reference<STR> = " << boolalpha << is_reference<STR>::value << endl;
    cout << "is_same<STR, char*> = " << boolalpha << is_same<STR, char*>::value << endl;
    cout << "is_same<STR, const char*> = " << boolalpha << is_same<STR, const char*>::value << endl;

    for (int j = 0; j < TRIES; ++j)
	{
		cout << "Durchlauf " << j+1 << endl;

        start("::copy");
        std::copy(big.begin(), big.end(), bigcopy.begin());
        stop(t_copy);

        start("memcpy");
        memcpy(arrcopy, arr, SZ * sizeof(int));
        stop(t_memcpy);

        start("copier (iter)");
        clever::copy(big.begin(), big.end(), bigcopy.begin());
        stop(t_copy_vector);

        start("copier (memcpy)");
        clever::copy(arr, arr+SZ, arrcopy);
        stop(t_copy_array);

        cout << endl;
    }

	cout << "Mittelwerte:" << endl;
    cout << "  std::copy               = " << t_copy   / TRIES << " ms" << endl;
    cout << "  std::memcpy             = " << t_memcpy / TRIES << " ms" << endl;
    cout << "  clever::copier (iter)   = " << t_copy_vector / TRIES << " ms" << endl;
    cout << "  clever::copier (memcpy) = " << t_copy_array / TRIES << " ms" << endl;

    delete [] arrcopy;
    delete [] arr;

    return 0; 
} 
