﻿#!/usr/bin/perl -w

use strict;
use warnings;
use utf8;
use Getopt::Long;
use Pod::Usage;

sub encode_simple($);
sub encode_extended($);

my @GOOGLE_CHART_API_ENCODING =
    qw(
       A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
       a b c d e f g h i j k l m n o p q r s t u v w x y z
       0 1 2 3 4 5 6 7 8 9
       - .
       );
my $max_value = -1e308;
my $min_value = +1e308;
my $encoding_method = 's'; # oder 'e'
my @values;
my $HELP;
my $LONGHELP;
my $file = undef;

GetOptions(
           "encoding|e=s" => \$encoding_method,
           "lo|min=s"     => \$min_value,
           "hi|max=s"     => \$max_value,
           "file=s"       => \$file,
           "help|h|?"     => \$HELP,
           "longhelp"     => \$LONGHELP,
           );

pod2usage(1) if $HELP || !defined($file);
pod2usage(-verbose => 2)  if $LONGHELP;

open IN, "<$file"
    or die "Kann $file nicht zum Lesen öffnen.";

while (<IN>)
{
    chomp;
    tr/,/./;
    if ( $_ ) {
        if ( $_ > $max_value ) { $max_value = $_ }
        if ( $_ < $min_value ) { $min_value = $_ }
        push @values, $_;
    }
    else {
        push @values, undef;
    }
}

close IN;

my $range = $max_value - $min_value;
if ( $range == 0 ) { $range = 1; }

printf "min/max = %f/%f\n", $min_value, $max_value;

if ( $encoding_method =~ /s(imple)?/ )
{
    print "\nchd=s:",
        map{ defined $_? encode_simple(61 * ($_ - $min_value) / $range) : '_' } @values;
}
elsif ( $encoding_method =~ /e(xt(ended)?)?/ )
{
    print "\nchd=e:",
        map{ defined $_? encode_extended(4095 * ($_ - $min_value) / $range) : '__' } @values;
}
else
{
    die "Fehler: unbekannte Kodierungsmethode";
}

print "\n";


sub encode_simple($)
{
    my $val = shift;
    return $GOOGLE_CHART_API_ENCODING[$val % 62];
}

sub encode_extended($)
{
    my $val = shift;
    return $GOOGLE_CHART_API_ENCODING[$val / 64]
         . $GOOGLE_CHART_API_ENCODING[$val % 64];
}

0;

__END__

=pod

=head1 NAME

C<gchartencode.pl> - Konvertieren von Zahlendaten ins einfache und erweiterte Format von Google Charts (L<http://code.google.com/apis/chart/>).

=head1 SYNOPSIS

    gchartencode.pl Parameter

Parameter:

  --encoding       Kodierung wählen
  --file           Daten aus der angegebenen Datei lesen
  --min            Minimum setzen
  --max            Maximum setzen
  --help           diese Kurzhilfe ausgeben
  --longhelp       komplette Dokumentation ausgeben

=head1 OPTIONS

=over 8

=item B<--encoding=simple>

in einfache Kodierung wandeln

=item B<--encoding=extended>

in erweiterte Kodierung wandeln

=item B<--file=datei.dat>

Daten aus datei.dat lesen

=item B<--min=Minimum>

den angegebenen Wert als Minimum verwenden, anstatt es automatisch zu ermitteln

=item B<--max=Maximum>

den angegebenen Wert als Maximum verwenden, anstatt es automatisch zu ermitteln

=back

=head1 DESCRIPTION

B<gchartencode.pl> liest zeilenweise Zahlen aus einer Datei und kodiert sie so, dass sie sich in einer Google-Charts-URL als darzustellende Daten angeben lassen (vgl. S<c't 15/08>, S<S. 176>).


Ohne Vorgabe von Minimum und Maximum würde das Programm die Zahlen aus folgender Kolonne (DAX-Schlusskurse eines nicht näher benannten Zeitraums) so kodieren, dass der kleinste darin enthaltene Wert (6182.30) in den Code mit der geringsten Wertigkeit gewandelt wird, also "A" in einfacher beziehungsweise "AA" in erweiterter Kodierung, und der größte Wert (8076.12) in den Code mit der höchsten Wertigkeit, also "9" in einfacher beziehungsweise ".." in erweiterter Kodierung:

=over 8

7723.66 7765.19 7870.52 7837.26 7808.94 7944.77 7940.58 7994.07
8033.36 8009.42 B<8076.12> 7928.31 7948.36 7825.44 7850.74 7837.32
7869.19 8002.67 8038.60 8067.32 7949.11 7908.41 7808.69 7817.17
7849.99 7782.71 7713.09 7717.95 7732.02 7566.38 7471.57 7413.53
7314.17 6790.19 6769.47 6439.21 6821.07 6816.74 6818.85 6892.96
6875.35 6851.75 6968.67 7000.49 6765.25 6847.51 6733.72 6767.28
6743.54 6967.84 6973.67 6962.28 6832.43 6967.55 7002.29 6899.68
6904.85 6806.29 6882.56 6985.97 6997.85 6862.52 6748.13 6689.95
6545.04 6683.71 6591.31 6513.99 6448.08 6524.57 6599.37 6500.56
6451.90 B<6182.30> 6393.39 6361.22 6319.99 6524.71 6489.26 6578.06
6559.90 6534.97 6720.33 6777.44 6741.72 6763.39 6821.03 6771.98
6721.36 6704.32 6603.57 6554.49 6585.05 6702.84 6681.81 6843.08
6786.55 6728.30 6795.03 6821.32 6896.58 6925.33 6885.34 6948.82
7043.23 7052.08 7017.10 7076.25 7071.90 7003.17 7035.95 7060.19
7083.24 7081.05 7156.55 7225.94 7118.50 7040.83 7070.33 6944.05
6953.84 6958.66 7033.84

=back

Der Aufruf von

    gchartencode.pl --encoding=extended --file=dax.dat

erzeugt die Ausgabe

    min/max = 6182.300000/8076.120000

    chd=e:0E1e5C36297i7Z9N-i9u..6.7q3g4X364.9g-t.r7s6U283P4W2Ezu
    z40WuwrjpmmPUiT1IrVlVbVgYAXaWnakbpTsWeSoTwS9aiavaWV9ahbtYPYa
    VFXqbJbjW-THRJMQQ8N0LNI-LkOFKwJGAAHIGCEpLkKXNXMwL6SLUGS5ToVl
    T7SNRoOOMkNmRlQ4WUUaScUsVlYIZGXwZ5dFdYcNeMeDbuc1dqeceXg6jQfo
    dAeAZvaEaOcx

Wenn das nicht gewünscht ist, etwa weil die Achse, auf die die Werte abgetragen werden sollen, anders eingeteilt werden soll, muss man Minimum und Maximum auf der Kommandozeile vorgeben. Folgender Aufruf liest die F<dax.dat> und legt als untere und obere Grenze 6000 beziehungsweise 8500 fest:

    gchartencode.pl --lo=6000 --hi=8500 --encoding=extended --file=dax.dat

Die Ausgabe lautet:

    min/max = 6000.000000/8500.000000

    chd=e:sHtLv3vBuTxxxqzC0Czb1IxWx3uuvXvBv1zQ0L06x4w1uSugvWtor2
    r-sVoFlqkLhoUOTsLPVAU5U9W2WZVzYyZmTlVsSxToTBYxY6YoVTYwZpXBXK
    UoWlZPZiWETJRqN8RfPINJLdNbPVMzLkEqKEJPIMNbMhOyOVNsSbT5S-TiVA
    TwSdSBPcOMO-R.RcVkUISoUWVBW8XrWqYSasa7aCbibbZragbIbubqdmfYco
    aobZYKYaYiad

=head1 COPYRIGHT

Copyright (c) 2008 Oliver Lau E<lt>ola@ct.heise.deE<gt>, Heise Zeitschriften Verlag. Alle Rechte vorbehalten.

=cut
