// *****************************************************************
// Filename:  DatabaseEntry.cpp
// Copyright: Pedram Azad, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:	  Pedram Azad
// Date:      02.05.2008
// *****************************************************************



// *****************************************************************
// includes
// *****************************************************************

#include "DatabaseEntry.h"
#include "VideoCapture/BitmapSequenceCapture.h"
#include <math.h>



CDatabaseEntry::CDatabaseEntry(const char *pName, const char *pPath)
{
	m_pImages = new CDynamicArray(500);

	m_sName = "";
	m_sName += pName;

	m_sPath = "";
	m_sPath += pPath;

	m_nImageWidth = 0;
	m_nImageHeight = 0;
}

CDatabaseEntry::~CDatabaseEntry()
{
	delete m_pImages;
}

int CDatabaseEntry::GetNumberOfSamples() const
{
	return m_pImages->GetSize();
}

const float* CDatabaseEntry::GetImageData(int nIndex) const
{
	CDataElement *pDataElement = (CDataElement *) m_pImages->GetElement(nIndex);
	return (const float *) pDataElement->data;
}

bool CDatabaseEntry::AddImage(const CByteImage *pImage)
{
	if (m_nImageWidth != pImage->width || m_nImageHeight != pImage->height)
	{
		printf("Fehler: Bildgroessen stimmen nicht ueberein\n");
		return false;
	}

	const int nInputDimension = m_nImageWidth * m_nImageHeight;
	const unsigned char *pixels = pImage->pixels;

	CDataElement *pDataElement = new CDataElement(nInputDimension);

	for (int j = 0; j < nInputDimension; j++)
 		pDataElement->data[j] = pixels[j];
		
	NormalizeIntensity(pDataElement->data, nInputDimension);

	m_pImages->AddElement(pDataElement);

	return true;
}

bool CDatabaseEntry::ReadImages()
{
	m_pImages->Clear();

	CBitmapSequenceCapture capture(m_sPath.c_str());
	if (!capture.OpenCamera())
	{
		printf("Fehler: Ansichten in '%s' konnten nicht geladen werden\n", m_sPath.c_str());
		return false;
	}
	
	m_nImageWidth = capture.GetWidth();
	m_nImageHeight = capture.GetHeight();
	
	CByteImage image(m_nImageWidth, m_nImageHeight, CByteImage::eGrayScale);
	CByteImage *pImage = &image;
	
	int i;
		
	for (i = 0;; i++)
	{
		if (!capture.CaptureImage(&pImage))
			break;

		AddImage(&image);
	}

	printf("Info:   %i Bilder wurden fuer den Namen '%s' geladen\n", i, m_sName.c_str());
	
	return true;
}

void CDatabaseEntry::NormalizeIntensity(float *pData, int nLength)
{
	double mean = 0, factor = 0;
	int i;

	for (i = 0; i < nLength; i++)
		mean += pData[i];
		
	mean /= nLength;
		
	for (i = 0; i < nLength; i++)
	{
		pData[i] -= mean;
		factor += pData[i] * pData[i];
	}

	factor = (float) (1 / sqrt(factor));
		
	for (i = 0; i < nLength; i++)
		pData[i] *= factor;
}

int CDatabaseEntry::GetNumberOfSamples(const char *pFileName, int &width, int &height)
{
	// count number of rows
	CBitmapSequenceCapture capture(pFileName);
	if (!capture.OpenCamera())
		return 0;
	
	width = capture.GetWidth();
	height = capture.GetHeight();
	CByteImage image(width, height, CByteImage::eGrayScale);
	CByteImage *ppImage[] = { &image };
	
	int i;
	
	for (i = 0;; i++)
	{
		if (!capture.CaptureImage(ppImage))
			break;
	}
	
	return i;
}
