// Copyright (c) 2008 Oliver Lau <ola@ctmagazin.de>
// Heise Zeitschriften Verlag, Hannover, Germany

#include "ThreadPool.h"
#include "WorkerThread.h"



ThreadPool::ThreadPool(int maxThreads, QThread::Priority p)
{
	canceled = false;
	finishedSignalSent = false;
	numThreadsCompleted = 0;
	numJobs0 = 0;
	priority = p;
	requestedPoolSize = -1;
	setMaxThreadCount(maxThreads);
}


ThreadPool::~ThreadPool()
{
	waitForTerminated();
	pool.clear();
    jobs.clear();
}


// Append a job to the job queue.
// This function is thread-safe.
void ThreadPool::enqueue(Job* job)
{
    jobMutex.lock();
    jobs.enqueue(job);
    jobMutex.unlock();
}


// Append jobs to the job queue.
// This function is thread-safe.
void ThreadPool::enqueue(const JobQueue& jobq)
{
    jobMutex.lock();
	jobs << jobq;
    jobMutex.unlock();
}


void ThreadPool::run(void)
{
	for (WorkerThreadList::iterator i = pool.begin(); i < pool.end(); ++i)
	{
        (*i)->start();
	}
    signalAll();
}


void ThreadPool::quitAll(void)
{
	for (WorkerThreadList::iterator i = pool.begin(); i < pool.end(); ++i)
    {
        (*i)->terminate();
    }
}


void ThreadPool::cancel(void)
{
    jobMutex.lock();
	canceled = !jobs.empty();
	jobs.clear();
	jobMutex.unlock();
}


bool ThreadPool::isCanceled(void)
{
    return canceled;
}


void ThreadPool::signalAll(void)
{
	jobMutex.lock();
	int numJobs = numJobs0 = jobs.size();
	finishedSignalSent = false;
	numThreadsCompleted = 0;
	jobMutex.unlock();
	emit progressRangeChanged(0, numJobs);
	for (WorkerThreadList::iterator i = pool.begin(); i < pool.end(); ++i)
	{
        (*i)->trigger();
	}
}


/// Wait for threads to have finished their jobs.
void ThreadPool::waitForFinished(void)
{
	// Wait until all threads have signaled that they have
	// finished their jobs. Signalling is done via
	// calling jobsDone(), which will release a single
	// resource in the QSemaphore "ready".
	ready.acquire(pool.size());
}


void ThreadPool::jobsDone(void)
{
	qDebug("ThreadPool::jobsDone() %d threads", pool.size());
	ready.release();
	jobMutex.lock();
	if (jobs.isEmpty())
	{
		++numThreadsCompleted;
	}
	bool allThreadsCompleted = (numThreadsCompleted == pool.size());
	jobMutex.unlock();
	if (allThreadsCompleted && !finishedSignalSent)
	{
		finishedSignalSent = true;
		numJobs0 = 0;
		emit finished();
	}
	qDebug("ThreadPool::jobsDone() %d threads completed", numThreadsCompleted);
}


void ThreadPool::eraseThread(WorkerThreadList::iterator i)
{
	(*i)->terminate();
	(*i)->wait();
	pool.erase(i);
}


void ThreadPool::jobDone(WorkerThread* src)
{
	jobMutex.lock();
	int numJobsReady = numJobs0 - jobs.size();
	jobMutex.unlock();
	qDebug("ThreadPool::jobDone() emitting progressValueChanged(%d)", numJobsReady);
	emit progressValueChanged(numJobsReady);
	if (requestedPoolSize > pool.size())
	{
		poolMutex.lock();
		for (WorkerThreadList::iterator i = pool.begin(); i < pool.end(); ++i)
		{
			if (src == *i)
			{
				eraseThread(i);
				break;
			}
		}
		poolMutex.unlock();
	}
}


void ThreadPool::setPriority(QThread::Priority p)
{
	priority = p;
	for (WorkerThreadList::iterator i = pool.begin(); i < pool.end(); ++i)
    {
		(*i)->setPriority(priority);
	}
}


void ThreadPool::waitForTerminated(void)
{
	for (WorkerThreadList::iterator i = pool.begin(); i < pool.end(); ++i)
    {
        (*i)->wait();
    }
}


// Get next job from queue.
// Return NULL if job queue is empty.
// This function is thread-safe.
Job* ThreadPool::nextJob(void)
{
    Job* job = NULL;
    jobMutex.lock();
    if (!jobs.isEmpty())
    {
        job = jobs.dequeue();
    }
    jobMutex.unlock();
    return job;
}


void ThreadPool::setMaxThreadCount(int n)
{
	qDebug("ThreadPool::setMaxThreadCount(%d)", n);
	poolMutex.lock();
	if (pool.size() > n)
	{
		int reduceCount = pool.size() - n;
		for (WorkerThreadList::iterator i = pool.begin(); (i < pool.end()) && (reduceCount > 0); ++i, --reduceCount)
		{
			if ((*i)->isWaiting())
			{
				eraseThread(i);
			}
			// if no thread is currently waiting, rely on jobDone()
			// to terminate inactive threads.
			requestedPoolSize = n;
		}
	}
	else
	{
		for (int i = pool.size(); i < n; ++i)
		{
			WorkerThread* newWorker = new WorkerThread(this);
			pool.append(newWorker);
			newWorker->start(priority);
			newWorker->trigger();
		}
	}
	poolMutex.unlock();
}
