/****************************  disasm3.cpp   ********************************
* Author:        Andreas Stiller
* Date created:  2007-10-15
* Last modified: 2007-10-15;
* Project:       objconv
* Module:        disasm3.cpp
* Description:
* Module for Class CDisassembler containing FDO Generation & Instructionset Statistics
*
* (c) 2007 GNU General Public License www.gnu.org/copyleft/gpl.html
*****************************************************************************/
#include "stdafx.h"
/**********************  Warning and error texts   ***************************
These texts are inserted in disassembled code in case of warnings or errors.

The occurrence of an error makes the disassembler mark the code block between
the nearest known code labels as dubious. This means that the byte sequence
might be data in the code segment or the disassembler might be out of phase
with instruction boundaries. Dubious code will be shown both as code and as
data.

A warning will be shown as 'Note:' before the instruction it applies to.
This might indicate suboptimal coding or a possible cause for concern.

The criteria for distinguishing between warnings and errors is not the 
severity of consequences, but whether the condition is likely to be caused
by common programming errors or by data in the code segment.

Some of the warning messages are quite benign, e.g. an unnecessary prefix.
Other warning messages can have severe consequences, e.g. a function missing
a return statement.

Still other warnings are no case for concern, but a condition requiring 
attention. For example the message: "Multi-byte NOP. Replace with ALIGN",
might actually indicate a well optimized code. But it requires attention
because the assembler cannot re-create the multi-byte NOP if the code
is assembled again. The programmer needs to decide what level of alignment
is optimal and replace the NOP with an align statement.

*****************************************************************************/
/* FDO-file stores all branches and labels of the disassembled file 
/* in an easy to read form for profiling and debugging and feed back optimization 


Each entry is a structure of four uint32, byte 0 is the type byte 

Type 0x63h: Signature, first Entry in File 
0: "ctfd" sign for the fdo filetype 
1: uint8 Version (=10), uint8 Entrysize(=4), 0,0   
2: int32 cmd.inputType 
3: int32 cmd.DesiredWordSize

Type 0x64h: Imagefilename 
0: uint8 0x64h, 0,uint16 namesize in Byte
1: char[4] Imagefilename [1..4]
2: char[4] Imagefilename [5..8]
3: char[4] Imagefilename [9..12]

Type 0x65h: Imagefilename Continue
0: uint8 0x64h, 0,uint16 namesize in Byte
1: char[4] Imagefilename  ... 
2: char[4] Imagefilename  ...
3: char[4] Imagefilename  ...

Type 0x66h: Imagefile Date/Time
0: 0x66h,0,0,uint8 Datetype 
1: uint32 Date
2: uint32 Time 
3: 0 

Type 0x67h: Imagefile Size 
0: 0x67h,0,0,0 
1: uint32 FileSize lo
2: uint32 FileSize hi 
3: 0 

Type 0x68h: Imagefile Checksum  
0: 0x68h,0,0,checksumtype
1: uint32 checksum lo
2: uint32 checksum hi 
3: 0 




Type 1: Image Info, second Entry in File
0: uint8  1,0,0,0 
1: uint32 image Base (low 32Bit) 
2: uint32 image Base (hi 32 Bit) 

Type 2: Section/Segment 32 Bit info in a list at the beginning  (usefull for allocation) 
0:uint8  2,wordsize, uint16 sectiontype  
1:uint32 Section Startaddr   
2:uint32 Section Endaddr
3: reserved (=0) 

Type 3: Section/Segment 32 Bit info  as section limiter in the file
0:uint8  3,wordsize, uint16 sectiontype  
1:uint32 Section Startaddr   
2:uint32 Section Endaddr
3: reserved (=0) 

Type 4:  jcc & jmp Branch Instruction with rel8 32 Bit 
0: uint8  4, MFlags,Prefix,InstructionLen (total) 
1: uint32 addr 
2: uint8  op1, rel8, 0,0         
3: uint32 targetaddr (taken) 

Type 5:  jcc Instruction with rel16/32 32 Bit 
0: uint8  5, MFlags,Prefix,InstructionLen (total) 
1: uint32 addr 
2: uint8  op1,op2,0,0             
3: uint32 targetaddr (taken) 

Type 6:  abs JMP/Call Instruction with irel16/32 32 Bit 
0: uint8  6, MFlags,Prefix,InstructionLen (total) 
1: uint32 addr 
2: uint8  op1, 0, 0, 0             
3: uint32 targetaddr 

Type 7: indirect JMP/Call Instruction  32 Bit
0: uint8  7, MFlags,Prefix,InstructionLen (total) 
1: uint32 addr 
2: uint8  0xFF, op2, SIB(if exist), 0             
3: uint32 offset (if exist)   

Type 8:InterSegment JMP/Call  32 Bit
0: uint8  8, MFlags,Prefix,InstructionLen (total) 
1: uint32 addr 
2: uint8  op1, op2,  uint16 selector 
3: uint32 offset    

Type 9: indirect Intersegment JMP/Call 32 Bit
0: uint8  9, MFlags,Prefix,InstructionLen (total) 
1: uint32 addr 
2: uint8  0xFF, op2, SIB(if exist), 0             
3: uint32 offset (if exist)   

Type 0x10: Target  or Label  
0: uint8  0x10, 0, uint16 datatype 
1: uint32 addr (lo)
2: uint32 addr (hi)
3: reserved 

64 Bit not defined yet 
*/
/*
 s.MFlags                                       // Memory operand flags:
                                                 // 1 = has memory operand, 
                                                 // 2 = has mod/reg/rm byte, 
                                                 // 4 = has SIB byte, 
                                                 // 8 = has DREX byte (AMD SSE5 instructions),
                                                 // 0x10 = is rip-relative

*/

void CDisassembler::InitFdo() {
     uint32 imagefilenamelen;
	 uint32 towrite;
	 strcpy (fdofile,cmd.InputFile);
	 strcat (fdofile,".fdo");
     
	 hfdo=fopen(fdofile,"wb");
	 if (hfdo==NULL)
		 printf ("oops cant open %s",fdofile);
	 else 
	 {												   //Firstentry: File info 
	 fdobuffer[0]=0x64667463;                          //sign "ctfd" for ctfdo file 
	 fdobuffer[1]=0x04100000 ;                         //Version 1.0, 4 Dwords per Entry    
	 fdobuffer[2]=cmd.InputType; 	                   //FileType
	 fdobuffer[3]=cmd.DesiredWordSize;
 	 fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);
	 // Store Filename 
	 imagefilenamelen=(uint32) strlen(cmd.fileinfo.name);
     fdobuffer [0]=0x65+(imagefilenamelen<<16);
	 while (imagefilenamelen>0) {
     memset (&fdobuffer[1],0,12);
	 towrite=imagefilenamelen;
	 if (towrite > 12) towrite=12;   
	 memcpy (&fdobuffer[1],&cmd.fileinfo.name,towrite); 
     fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);
	 imagefilenamelen -=towrite;
     fdobuffer [0]=0x66;
	 }
     // Store Filesize
     fdobuffer [0]=0x67;
	 fdobuffer [1]=cmd.fileinfo.size;
	 fdobuffer [2]=0; 
	 fdobuffer [3]=0; 
     fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);

    // Store FileDate/Time
     fdobuffer [0]=0x68+(1L<<24);         //Type Windows time_create Date & Time  
	 memcpy (&fdobuffer[1],&cmd.fileinfo.time_create,8);
	 fdobuffer [3]=0; 
     fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);


	 fdobuffer[0]=1;                                   //Image info   
	 fdobuffer[1]=uint32(ImageBase);                                  
     fdobuffer[2]=uint32(ImageBase>>32);
	 fdobuffer[3]=0; 
	 fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);
	 }
}

void  CDisassembler::WriteSegmenttoFdoFile (uint32 Section,uint32 mark){
   
   fdobuffer[0]=mark+(Sections[Section].WordSize << 8) + (Sections[Section].Type<<16);
   fdobuffer[1]=Sections[Section].SectionAddress+(uint32)ImageBase;
   fdobuffer[2]=Sections[Section].TotalSize+Sections[Section].SectionAddress+(uint32)ImageBase;
   fdobuffer[3]=0; // reserved 
   fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);
}

void  CDisassembler::WriteAllCodeSegmentstoFdoFile (){
   for (uint32 aSection = 1; aSection < Sections.GetNumEntries(); aSection++) {
   if ((Sections[aSection].Type & 0xFF)==1)  // write only  valid Code Segments  
    WriteSegmenttoFdoFile(aSection,2); 
   }
}
void  CDisassembler::WriteTargettoFdoFile (uint32 datatype){
   int64 adr=IBegin+Sections[Section].SectionAddress+ImageBase;
   fdobuffer[0]=0x10 +((uint16)datatype <<16); 
   fdobuffer[1]=(uint32) adr;
   fdobuffer[2]=(uint32) (adr >> 32);
   fdobuffer[3]=0; // reserved 
   fwrite (&fdobuffer,sizeof(fdobuffer[0]),4,hfdo);
}

void  CDisassembler::AnalyzeInstructionforFdo() { 
#define info fdobuffer[0] 
#define addr fdobuffer[1]
#define opcodes fdobuffer[2]
#define target fdobuffer[3]

uint8 op1=Get<uint8>(s.OpcodeStart1);
uint8 op2=Get<uint8>(s.OpcodeStart1+1);
/*
#define store (typ) \
	{info= typ +(s.MFlags<<8)+(s.Prefixes[0]<<16)+(InstructionLen<<24);}\ 
	{addr=IBegin + SectionAddress + (uint32)ImageBase;}\
	{opcodes=op1+(op2<<8);}                                             
*/

	switch (op1) {
		case 0x70:case 0x71:case 0x72:case 0x73:case 0x74:case 0x75:case 0x76:case 0x77:
		case 0x78:case 0x79:case 0x7A:case 0x7B:case 0x7C:case 0x7D:case 0x7E:case 0x7F:
		case 0xE0:case 0xE1:case 0xE2:case 0xE3:case 0xEB:
			info=4 +(s.MFlags<<8)+(s.Prefixes[0]<<16)+(InstructionLen<<24); 
		    addr=IBegin + SectionAddress + (uint32)ImageBase;               // Startaddr (with prefix) 
			opcodes=op1+(op2<<8);                                              // op1 and imm8 
			target=addr+(int8)(op2) +InstructionLen;                           // taken, calc Target, no 16-Bit-segment-Wraparound yet!
			fwrite (&fdobuffer,sizeof(info),4,hfdo);
			break;
		case 0xE8:case 0xE9:
			info=6 +(s.MFlags<<8)+(s.Prefixes[0]<<16)+(InstructionLen<<24); 
			addr=IBegin + SectionAddress + (uint32)ImageBase;                   // Startaddr (with prefix) 
			opcodes=op1 +(op2<<8);                                               // op1 and  lo(byte) of addr
			if (WordSize==16) target=addr+Get<int16>(s.OpcodeStart1+1) +InstructionLen;    // taken  no 16-Bit-segment-Wraparound yet! 
			else if (WordSize==32) target=addr+Get<int32>(s.OpcodeStart1+1) +InstructionLen;
			fwrite (&fdobuffer,sizeof(info),4,hfdo);
			break;

		case 0xff:  //indirect jmp/call
			if ((s.Reg>=2) && (s.Reg <=5)) {                                       //with reg=2..5 is JMP or CALL  
				info=7+(s.MFlags<<8)+(s.Prefixes[0]<<16)+(InstructionLen<<24);
				if ((s.Reg==3) ||(s.Reg=5))info +=2;                              //0xFF/3 or/5) jmp/call intersegment with call/taskgate is quite unusual for Usermode-Apps in 32/64-Bit
				
				addr=IBegin + SectionAddress + (uint32)ImageBase;                 // Startaddr (with prefix) 
				opcodes=op1+(op2<<8);                                              // 0xFF, Mod/reg/rm, SIB  
				if (s.MFlags &4) 
					opcodes +=Get<uint8>(s.OpcodeStart1+2)<<16;      // if with SIB-Byte, store it here
				memcpy(&target,&Buffer[s.AddressField],s.AddressFieldSize);
				fwrite (&fdobuffer,sizeof(info),4,hfdo);
			}  
			break; 
		case 0x0f:
			if ((op2>= 0x80) && (op2<=0x8f)) {
				info=5 +(s.MFlags<<8)+(s.Prefixes[0]<<16)+(InstructionLen<<24); 
				addr=IBegin + SectionAddress + (uint32)ImageBase;               // Startaddr (with prefix) 
				opcodes=op1+(op2<<8);                                           // op1 and imm8 
				//Target Address for Taken
				switch (s.AddressFieldSize) {
					case 1: target=addr+Get<int8> (s.OpcodeStart1+1)+InstructionLen; break;
					case 2: target=addr+Get<int16>(s.OpcodeStart1+1)+InstructionLen; break;
					case 4: target=addr+Get<int32>(s.OpcodeStart1+1)+InstructionLen; break;
				}
				fwrite (&fdobuffer,sizeof(info),4,hfdo);
			}
			break;
		case 0xEA:case 0x9A:      
			//call/int intersegment with call/taskgate is quite unusual for 32/64 Bit Mode 
				info=5 +(s.MFlags<<8)+(s.Prefixes[0]<<16)+(InstructionLen<<24); 
				addr=IBegin + SectionAddress + (uint32)ImageBase;               // Startaddr (with prefix) 
			switch (s.AddressFieldSize) {
				case 4: opcodes=op1+(Get<uint16> (s.OpcodeStart1+3)<< 16); 
					    target =Get<uint16> (s.OpcodeStart1+1);break;
				case 6: opcodes=op1+(Get<uint16> (s.OpcodeStart1+5)<< 16); 
					    target =Get<uint32> (s.OpcodeStart1+1);break;
			}	
			fwrite (&fdobuffer,sizeof(info),4,hfdo);
			break;
	} // switch
}  


// #as added for statistics

void CDisassembler::ResetInstructionStatistics() 
{
   InstructionCount = 0;  
   instructions = 0;                      // Total number of instructions
   PrivilInstr = 0;                       // Number of privileged instructions
   UndocInstr = 0;                        // Number of undocumented instructions
   UnknownInstr = 0;                      // Number of undocumented instructions
   TotalLen = 0;                          //       
   memset (&InstructionStat,0,sizeof (InstructionStat));
}
void CDisassembler::UpdateStatistics() {
if (cmd.DoStatistics == 2) {
   OutFile.Put(" #");
   OutFile.PutDecimal (InstructionLen,0);                      // #as output Instructionlen 
	 }
   TotalLen +=InstructionLen;								   // #as  
   InstructionCount++;                                         // #as
   PutinInstruction(cmd.DoStatistics==2);					   // #as for Statistics
 }

void CDisassembler::PutinInstruction(bool doprint) { 
	// Count instructiontypes as defined in opcodes.cpp
	// in instruction statistic array "InstructionStat"
    uint8 Dest; 
	uint8 InstSet = s.OpcodeDef->InstructionSet & 0xFF;
    if (s.OpcodeDef->InstructionSet & iPriv) PrivilInstr++;  // Privileged instruction 
    if (s.OpcodeDef->InstructionSet & i64Bit) InstSet=i64;
	if ((WordSize == 64) && (InstSet <i64)) {
        InstructionStat[InstSet]++;
		switch (s.OperandSize) {
			case 64:InstSet=i64b64; break; 
			case 32:InstSet=i64b32; break;
			case 16:InstSet=i64b16; break; 
			case  8:InstSet=i64b8;  break;
		}
	}
    if ((WordSize == 32) && (s.OperandSize==32) && (InstSet <i386)) InstSet=i386;  
	if (s.OpcodeDef->AllowedPrefixes==0xE00) {   
		//Change some SSE instructions to SSE2 if 66 or F2 prefix 
		if ((s.Prefixes[5] == 0x66) || (s.Prefixes[5] == 0xF2)) InstSet=iSSE2;
	}

	if ((InstSet &0xE0) == ioffMMX && s.Prefixes[5] == 0x66) {
		// Change MMX to SSE2 if 66 prefix
		InstSet = iSSE2;
	}


	if ( (InstSet == iSSSE3) && (s.OpcodeDef->AllowedPrefixes == 0x200) && s.Prefixes[5] != 0x66)
		// Change iSSSE3 to MMX4 if not 66 prefix  
		InstSet = iMMX4;
        Dest=0; 
	if ((InstSet &0xE0) == (ioffSSE)) { //for special SSE statistics 
        Dest=s.OpcodeDef->Destination & 0xFF;
		if (Dest==0x6F) {
			switch (s.Prefixes[5]) {
				case 0x00: Dest=0x6B; break; 
				case 0x66: Dest=0x6C; break; 
				case 0xF2: Dest=0x4C; break; 
				case 0xF3: Dest=0x4B; break;
		    }
		}

		if (InstSet==iSSE) {
			switch (Dest) {
			case 0x4B: InstructionStat[iSSEss]++; break;
			case 0x6B: InstructionStat[iSSEps]++; break;
			}
		}
		if (InstSet==iSSE2) {
			switch (Dest) {
			case 0x4B: InstructionStat[iSSE2ss]++; break;
			case 0x4C: InstructionStat[iSSE2sd]++; break;
			case 0x6B: InstructionStat[iSSE2ps]++; break;
			case 0x6C: InstructionStat[iSSE2pd]++; break;
			}
		}
		if (InstSet==iSSE3) {
			switch (Dest) {
			case 0x4B: InstructionStat[iSSE3ss]++; break;
			case 0x4C: InstructionStat[iSSE3sd]++; break;
			case 0x6B: InstructionStat[iSSE3ps]++; break;
			case 0x6C: InstructionStat[iSSE3pd]++; break;
			}
		}
	}


	InstructionStat[InstSet]++;
	const char * is = "unknown instr";
	if (doprint) 
	{
		switch (InstSet) {

		   case i86 : is="8086";break;
		   case i186: is="186";break; 
		   case i286: is="286";break;
		   case i386: is="386";break; 
		   case i486: is="486";break;
		   case iP5:  is="P5";break; 
		   case iP6:  is="P6";break;
		   case iPII: is="PII";break;
		   case iPIII:is="PIII";break;
		   case iP4:  is="P4";break; 
		   case iP4E: is="P4E";break;
		   case i64:  is="AMD64";break;
	       case i64b64:is="AMD64-64";break;
		   case i64b32:is="AMD64-32";break;
		   case i64b16:is="AMD64-16";break;
		   case i64b8: is="AMD64-8";break;
		   case i87:  is="8087";break;
		   case i287: is="287";break; 
		   case i387: is="387";break;
		   case iFPUP6: is="FPUP6"; break;
		   case iFPUPNI:is="FPUPNI";break;
		   case iMMX: is="MMX";break; 
		   case iMMX2: is="MMX2 (SSE)";break;
		   case iMMX3: is="MMX3 (SSE2)"; break;
		   case iMMX4: is="MMX4 (SSSE3)"; break; 
		   case iSSE:   is="SSE";break; 
		   case iSSE2:  is="SSE2";break;
		   case iSSE3:  is="SSE3";break;
		   case iSSSE3: is="SSSE3";break; 
		   case iSSE41: is="SSE4.1";break;
		   case iSSE42: is="SSE4.2";break; 
		   case iMemSSE: case iMemSSE2: case iMemSSE3: is="Mem"; break;
		   case iK3D:  is="3Dnow!";break; 
		   case iK3D2: is="3Dnow2";break; 
		   case iSSE4a:is="SSE4a";break;
		   case iSSE5: is="SSE5";break;
		}
	

		OutFile.Put (",");
		OutFile.Put (is);
	     switch (Dest) {
			case 0x4B: OutFile.Put(" ss"); break;
			case 0x4C: OutFile.Put(" sd"); break;
			case 0x6B: OutFile.Put(" ps"); break;
			case 0x6C: OutFile.Put(" pd"); break;
			}
	}       
}

void CDisassembler::WriteStatistics() {
char  S[80];
OutFile.NewLine();
sprintf(S,"% d Instructions, %d Bytes, %3.1f Bytes/Instruction",InstructionCount,TotalLen, ((float)TotalLen)/(float)InstructionCount);
printf ("\n%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"86..286: %9d",InstructionStat[i86]+InstructionStat[i186]+InstructionStat[i286]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"386/486: %9d",InstructionStat[i386]+InstructionStat[i486]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

if (WordSize==64) {
sprintf (S,"i64    : %9d, op=64: %d, op=32: %d, op=16: %d, op=8: %d",
		InstructionStat[i64],InstructionStat[i64b64],InstructionStat[i64b32],InstructionStat[i64b16],InstructionStat[i64b8]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();
}

sprintf (S,"P5     : %9d",InstructionStat[iP5]); 
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"P6     : %9d",InstructionStat[iP6]); 
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"PII    : %9d",InstructionStat[iPII]); 
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"FPU    : %9d",InstructionStat[i87]+InstructionStat[i387]+InstructionStat[iFPUP6]+InstructionStat[iFPUPNI]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"MMX    : %9d",InstructionStat[iMMX]+InstructionStat[iMMX2]+InstructionStat[iMMX3]+InstructionStat[iMMX4]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE    : %9d, ps: %6d, ss: %6d",InstructionStat[iSSE],InstructionStat[iSSEps],InstructionStat[iSSEss]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE2   : %9d, ps: %6d, ss: %6d, pd: %6d, sd: %6d",InstructionStat[iSSE2],InstructionStat[iSSE2ps],InstructionStat[iSSE2ss],InstructionStat[iSSE2pd],InstructionStat[iSSE2sd]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE3   : %9d, ps: %6d, ss: %6d, pd: %6d, sd: %6d",InstructionStat[iSSE3],InstructionStat[iSSE3ps],InstructionStat[iSSE3ss],InstructionStat[iSSE3pd],InstructionStat[iSSE3sd]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSSE3  : %9d",InstructionStat[iSSSE3]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE4.1 : %9d",InstructionStat[iSSE41]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE4.2 : %9d",InstructionStat[iSSE42]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"Mem    : %9d",InstructionStat[iMemSSE]+InstructionStat[iMemSSE2]+InstructionStat[iMemSSE3]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"3Dnow  : %9d",InstructionStat[iK3D]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"3Dnow2 : %9d",InstructionStat[iK3D2]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE4a  : %9d",InstructionStat[iSSE4a]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"SSE5   : %9d",InstructionStat[iSSE5]);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

sprintf (S,"nn     : %9d",UnknownInstr);
printf ("%s\n",S);OutFile.Put(CommentSeparator);OutFile.Put(S);OutFile.NewLine();

printf ("\n");OutFile.NewLine();

//CountMapInstructions(); 
}