/*
 * lev100.c
 * --------
 *
 * Levenshtein function and the like.
 *
 * Copyright (c):
 * 2007:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *
 * SCCS: @(#) lev100.c  1.0  2007-09-10
 *
 * This file is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this file; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLA.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. modify the underlying logic or translate this program into another
 * programming language), you must also release the changes under the terms
 * of the LGPL.
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "umlaut_a.h"
#define  __IS_LEV100_C__
#include "addr_ext.h"



/************************************************************/
/****  macros with umlauts (do not change)  *****************/
/************************************************************/

#define INTL_AE     ''
#define INTL_OE     ''
#define INTL_UE     ''

#define SCAND_AA    ''
#define SCAND_AE    ''
#define FRENCH_OE   ''
#define GERMAN_SS   ''

#define MATCHES_ALL   -1000

#define WORD_SEPARATORS   "+-/*( )&,'`.:"



static char *zip_german_cities[] =
     {
       "10 BERLIN",
       "12 BERLIN",
       "13 BERLIN",
       "14 BERLIN",
       "20 HAMBURG",
       "21 HAMBURG",
       "22 HAMBURG",
       "50 KOELN",
       "51 KOELN",
       "60 FRANKFURT",
       "65 FRANKFURT",
       "68 MANNHEIM",
       "69 MANNHEIM",
       "80 MUENCHEN",
       "81 MUENCHEN",
       NULL
     };


/****  external variables  ****/
int  conv_strings_initialized = 0;
char up_and_conv[HASH_COUNT];
char sortchar [HASH_COUNT];
char sortchar2[HASH_COUNT];
char upperchar[HASH_COUNT];
char lowerchar[HASH_COUNT];
int  isxletter[HASH_COUNT];




/************************************************************/
/****  private (static) functions  **************************/
/************************************************************/

static struct LEV_RESULT calculate_points
     (int diff, int limit, int unknown, int empty_diff, int max_points)
{
  struct LEV_RESULT l_res = { 0, 0, 0, 0 };
  int points = 0;

  if (empty_diff != 0)
    {
     if (empty_diff == MATCHES_ALL  ||  empty_diff == 3)
       {
        diff = 0;
        max_points = 0;
        points = max_points;
        empty_diff = 0;
       }
     else
       {
        diff = 0;
        max_points = (int) ((max_points+2) / 8);
        points = 0;
        empty_diff = max_points;
       }
    }
  else
    {
     if (diff <= 0)
       {
        points = 1000;
       }
     else if (diff + diff <= limit)
       {
        points = 700 + (int) (300L * (limit - diff - diff) / limit);
       }
     else if (diff <= limit)
       {
        points = 300 + (int) (800L * (limit - diff) / limit);
       }
     else if (diff <= limit +50  &&  limit >= 150)
       {
        points = 200 + (int) (100L * (limit + 50L - diff) / 50L);
       }

     if (points > 0)
       {
        if (limit < 70)
          {
           max_points = (int) ((long)max_points * (long)limit / 70L);
          }

        if (unknown > 0)
          {
           points = (int) ((long)points * (1000L - unknown) / 1000L);
          }
        points = (int) ((long)points * (long)max_points / 1000L);
       }

     if (unknown > 0)
       {
        unknown = (int) ((long)unknown * (long)max_points / 1000L);
       }
    }

  l_res.points = points;
  l_res.max_points = max_points;
  l_res.diff = diff;
  l_res.empty_diff = empty_diff;

  return (l_res);
}





static int lev_2_name (char text1[], char text2[], int run_mode)
{
  struct LEV_RESULT l_res;
  int  i,n,limit;
  char c,*s,*s2;
  char tmp_string [LENGTH_WHOLE_NAME+1];
  char *tt = tmp_string;

  /****  check for name like "John" vs. "John Carl"  ****/
  i = (int) strlen (text1);
  n = (int) strlen (text2);

  if (i >= n+4  &&  n >= 4)
    {
     s = text1;
     s2 = text2;
    }
  else if (n >= i+4  &&  i >= 4)
    {
     i = n;
     s = text2;
     s2 = text1;
    }
  else
    {
     /****  return "no match"  ****/
     return (10000);
    }

  if (i > LENGTH_WHOLE_NAME)
    {
     tt = (char *) malloc ((unsigned) (i+1));
     if (tt == NULL)
       {
        if (run_mode & (TRACE_ADDR | TRACE_ERRORS | TRACE_LEV))
          {
           printf ("Error: \"malloc\" for %d Bytes failed.\n", i+1);
          }
        return (10000);
       }
    }
  strcpy (tt,s);
  s = tt;
  i = 10000;

  l_res = calculate_limit (text2, text1, run_mode);
  limit = l_res.points;

  if (limit > 0  &&  l_res.empty_diff != MATCHES_ALL)
    {
     limit = (int)(limit/2) + 100;
    }

  /****  search for word separator  ****/
  while (*s != '\0'  &&  *s != '.'  &&  *s != '-'  &&  *s != ' ')
    {
     s++;
    }

  if (*s != '\0')
    {
     /****  evaluate leading name  ****/
     if (*s == '.')
       {
        s++;
       }
     c = *s;
     *s = '\0';

     if ((int)strlen (tt) >= 4)
       {
        n = lev_diff (s2,tt, limit,
               ((run_mode | LEV_COMPARE_INTERNAL) & ~TRACE_LEV));

        if (n <= limit)
          {
           i = n;
          }
       }
     *s = c;

     /****  search for trailing name  ****/
     while (*s == '.'  ||  *s == '-'  ||  *s == ' ')
       {
        s++;
       }

     /****  evaluate trailing name  ****/
     if (*s != '\0'  &&  (int) strlen (s) >= 4)
       {
        /****  evaluate trailing name  ****/
        n = lev_diff (s2,s, limit,
               ((run_mode | LEV_COMPARE_INTERNAL) & ~TRACE_LEV));

        if (n < i  &&  n <= limit)
          {
           i = n;
          }
       }
    }

  if (tt != tmp_string)
    {
     free (tt);
    }

  return (i);
}





/************************************************************/
/****  "external" functions  ********************************/
/************************************************************/


int initialize_conv_strings (int run_mode)
{
  int  i,k,n;
  char *s,*s2,*s3;

  if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
    {
     if ((int)strlen (letters_a_to_z) > 26)
       {
        if (run_mode & (TRACE_ADDR | TRACE_LEV))
          {
           printf ("Error: %s  is not allowed\n",
                "strlen (letters_a_to_z) > 26");
          }
        return (-1);
       }
     if ((int)strlen (letters_a_to_z) != (int)strlen (letters_A_to_Z))
       {
        if (run_mode & (TRACE_ADDR | TRACE_LEV))
          {
           printf ("Error: %s  is not allowed\n",
                "strlen(letters_a_to_z) != strlen(letters_a_to_z)");
          }
        return (-1);
       }

     if ((int)strlen (umlaut_lower) != (int)strlen (umlaut_upper))
       {
        if (run_mode & (TRACE_ADDR | TRACE_LEV))
          {
           printf ("Error: %s  is not allowed\n",
                "strlen(umlaut_lower) != strlen(umlaut_upper)");
          }
        return (-1);
       }

     if ((int)strlen (umlaut_lower) != (int)strlen (umlaut_sort)
     ||  (int)strlen (umlaut_lower) != (int)strlen (umlaut_sort2))
       {
        if (run_mode & (TRACE_ADDR | TRACE_LEV))
          {
           printf ("Error: %s  is not allowed\n",
                "strlen(umlaut_sort*) != strlen(umlaut_lower)");
          }
        return (-1);
       }

     conv_strings_initialized |= CONV_STRINGS_ARE_INITIALIZED;

     /****  generate arrays "isxletter", "sortchar", "sortchar2", ...  ****/
     for (i=0; i< HASH_COUNT; i++)
       {
         isxletter[i] = 0;
         sortchar[i] = (char) i;
         sortchar2[i] = '\0';
         up_and_conv[i] = (char) i;
         upperchar[i] = (char) i;
         lowerchar[i] = (char) i;
       }
     sortchar [(unsigned char) '-']  = ' ';
     up_and_conv [(unsigned char) '-'] = ' ';
     upperchar [(unsigned char) '-']  = ' ';
     lowerchar [(unsigned char) '-']  = ' ';

     s = letters_a_to_z;
     s2 = letters_A_to_Z;

     for (i=0; s[i] != '\0'; i++)
       {
         k = (unsigned char) s2[i];  /** "s2" **/
         isxletter[k] |= _IS_UPPER_;
         sortchar[k] = s2[i];
         up_and_conv[k] = s2[i];
         upperchar[k] = s2[i];
         lowerchar[k] = s[i];

         k = (unsigned char) s[i];   /** "s" **/
         isxletter[k] |= _IS_LOWER_;
         sortchar[k] = s2[i];
         up_and_conv[k] = s2[i];
         upperchar[k] = s2[i];
         lowerchar[k] = s[i];
       }

     s = umlaut_lower;
     s2 = umlaut_sort;
     s3 = umlaut_sort2;

     for (i=0; s[i] != '\0'; i++)
       {
        n = (unsigned char) umlaut_conv[i];

        k = (unsigned char) umlaut_upper[i];
        isxletter[k] |= _IS_UPPER_ + _IS_UMLAUT_;
        up_and_conv[k] = (char) n;
        upperchar[k] = (char) k;
        lowerchar[k] = s[i];

        sortchar[k] = s2[i];
        isxletter [(unsigned char) s2[i]] |= _IS_SORTCHAR_;
        if (s3[i] != ' ')
          {
            sortchar2[k] = s3[i];
            isxletter [(unsigned char) s3[i]] |= _IS_SORTCHAR2_;
          }

        k = (unsigned char) s[i];   /** "s" **/
        isxletter[k] |= _IS_LOWER_ + _IS_UMLAUT_;
        up_and_conv[k] = (char) n;
        lowerchar[k] = s[i];

        sortchar[k] = s2[i];
        n = lowerchar [(unsigned char) s2[i]];
        isxletter [(unsigned char) n] |= _IS_SORTCHAR_;
        if (s3[i] != ' ')
          {
            sortchar2[k] = s3[i];
            n = lowerchar [(unsigned char) s3[i]];
            isxletter [(unsigned char) n] |= _IS_SORTCHAR2_;
          }

        n = (unsigned char) umlaut_upper[i];
        upperchar[k] = (char) n;
       }

     s = "0123456789";
     for (i=0; s[i] != '\0'; i++)
       {
         k = (unsigned char) s[i];
         isxletter[k] |= _IS_DIGIT_;
       }
    }

  return (0);
}




void print_number (char *text, int number, int mode)
{
  char *s;

  if (mode >= 0  &&  number == MATCHES_ALL)
    {
     printf ("%s = %s", text, "MATCHES_ALL");
     return;
    }

  s = "";
  if (number < 0)
    {
     s = "-";
     number = -number;
    }
  printf ("%s = %s%d", text,s, (int)(number/100));

  if (number % 100 != 0)
    {
     number = number % 100;
     printf (".%d", (int)(number/10));
     
     if (number % 10 != 0)
       {
        printf ("%d", number % 10);
       }
    }
}




struct LEV_RESULT calculate_limit 
            (char pattern[], char text[], int run_mode)
/****  calculate limit for the Levenshtein function  ****/
{
  struct LEV_RESULT l_res = { 0, 0, 0, 0 };
  char *s;
  int  i,j,k,n,x;
  int  max_points, limit;
  int  asterisk_found, unknown;

  char wildcard_any_string = MATCHES_ANY_STRING;
  char wildcard_any_char = MATCHES_ANY_CHAR;

  if (run_mode & DB_WILDCARDS_FOR_LIKE)
    {
     wildcard_any_string = DB_LIKE_ANY_STRING;
     wildcard_any_char = DB_LIKE_ANY_CHAR;
    }

  max_points = 0;
  unknown = 0;

  for (j=1; j<=2; j++)
    {
     s = pattern;
     if (j == 2)
       {
        s = text;
       }

     if (s == NULL)
       {
        s = "";
       }
     if (*s == '\0')
       {
        l_res.empty_diff += j;
       }

     asterisk_found = 0;
     x = 0;
     i = 0;
     while (s[i] != '\0')
       {
        /****  look for word separators and wildcard_any_string/char  ****/
        k = 0;
        n = 0;
        while (s[i] == wildcard_any_string
        ||  s[i] == '.'  ||  s[i] == wildcard_any_char)
          {
           if (s[i] == wildcard_any_string
           ||  s[i] == '.')
             {
              asterisk_found = 1;
              n = 300;
             }
           if (s[i] == wildcard_any_char)
             {
              k += 100;
             }

           if (s[i+1] != wildcard_any_string
           &&  s[i+1] != '.'  &&  s[i] != wildcard_any_char)
             {
              break;
             }
           i++;
          }

        if (n > k)
          {
           k = n;
          }
        unknown += k;

        if (strchr (WORD_SEPARATORS, s[i]) == NULL)
          {
           /****  count "letters"  ****/
           x += 100;
          }
        else if (s[i] != wildcard_any_string
        &&  s[i] != '.'  &&  s[i] != wildcard_any_char)
          {
           x += 40;
          }

        i++;
       }

     if (asterisk_found  &&  unknown < 600 - x)
       {
        unknown = 600 - x;
       }

     if (j == 1  ||  x < max_points)
       {
        max_points = x;
       }
     if (j == 1  ||  unknown > l_res.diff)
       {
        l_res.diff = unknown;
       }

     if (run_mode & DATABASE_SELECT)
       {
        /****  evaluate "pattern" only  ****/
        if (i == 0
        || (x == 0  &&  strchr (pattern,wildcard_any_string) != NULL))
          {
           l_res.empty_diff = MATCHES_ALL;
          }

        break;
       }
    }

  limit = max_points;
  if (limit > 0)
    {
     if (limit > 1200)
       {
        limit = 1200;
       }

     limit = (int) ((limit +40) / 3);
    }

  l_res.max_points = max_points;
  l_res.points = limit;
  l_res.diff = unknown;

#ifdef qweqweasd
  if (min_length <= l_res.unknown)
    {
     l_res.unknown = 1000;
    }
  else if (l_res.unknown > 0)
    {
      l_res.unknown = (int)((long) l_res.unknown * 1000L / (long) min_length);
    }
#endif

  return (l_res);
}




int up_expand (char src[], char dest[], int len, int run_mode)

/****  Function converts "src" to upper case,             ****/
/****  expands umlauts and compresses successive blanks,  ****/
/****  i.e. "<ae>" will be converted to "AE".             ****/

/****  The conversion is done using the arrays "sortchar" ****/
/****  and "sortchar2". The second array is needed only   ****/
/****  for umlauts and contains null values for all       ****/
/****  other chars.                                       ****/

/****  (len = size of "dest" incl. '\0').                 ****/
/****  This function returns "strlen (dest)".             ****/
{
  int  i,n;
  char c;

  if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
    {
      initialize_conv_strings (run_mode);

      if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
        {
         /****  internal error  ****/
         dest[0] = '\0';
         return (-1);
        }
    }

  i = 0;
  n = 0;
  while (*src != '\0'  &&  n < len-1)
    {
      /****  find next char in "src"  ****/
      c = '\0';
      if (i != 0)
        {
          c = sortchar2 [(unsigned char) *src];
          i = 0;
          src++;
        }
      if (c == '\0')
        {
          c = sortchar [(unsigned char) *src];
          i = 1;
        }

      if (c != '\0'
      && (c != ' '  ||  n <= 0  ||  dest[n-1] != ' '))
        {
          if ((run_mode & COMPRESS_MULTIPLE_CHARS)
          &&  n > 0  &&  c == dest[n-1])
            {
             n--;
            }
          dest[n] = c;
          n++;
        }
    }

  dest[n] = '\0';
  return (n);
}





int lev_diff (char *pattern, char *text, int limit, int run_mode)

/****  Function for calculating Levenshtein distance         ****/
/****  (in hundredths of points) with respect to wildcards   ****/
/****  ("text" is compared with "pattern"; allowed wildcards ****/
/****  are <wildcard_any_string> and <wildcard_any_char>)    ****/
/****  (function works for arbitrary string lengths)         ****/

/****  This function does not discriminate between           ****/
/****  lower / upper char, including umlauts                 ****/
/****  and is also "symmetric": shorter string = pattern;    ****/
/****  "limit" may be re-calculated                          ****/

/****  If "limit >= 0", the function returns "no match",     ****/
/****  if the minimum distance (as given by "col_min")       ****/
/****  is greater than limit.                                ****/

/****  Available values for "run_mode" (bit mask):           ****/
/****  LEV_COMPARE_NORMAL :  "normal" Levenshtein distance   ****/
/****  LEV_SKIP_UPEXPAND  :  do not convert strings to       ****/
/****                       upper char (incl. umlaut_expand) ****/
/****  LEV_COMPARE_GERMAN :  use simple phonetic algorithm   ****/
/****                        for German                      ****/
/****  TRACE_LEV          :  activate trace option           ****/
/****  (Do not use "LEV_COMPARE_INTERNAL" - internal use only) ****/
{
 struct LEV_RESULT l_res;
 int  col_min,
      lim_1,
      p,q,r,
      lt,lp,
      d1,d2,
      i,k,n,
      x1,x2;
 char ct,cp,cp2,
      *sp,*st,
      p_string[51],
      t_string[51];
 int  d_array[51];
 int  *d = d_array;
 char *pp = p_string;
 char *tt = t_string;

 char wildcard_any_string = MATCHES_ANY_STRING;
 char wildcard_any_char = MATCHES_ANY_CHAR;

 if (run_mode & DB_WILDCARDS_FOR_LIKE)
   {
    wildcard_any_string = DB_LIKE_ANY_STRING;
    wildcard_any_char = DB_LIKE_ANY_CHAR;
   }

 st = text;
 sp = pattern;

 if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
   {
     initialize_conv_strings (run_mode);

     if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
       {
        /****  internal error  ****/
        return (10000);
       }
   }

 if (run_mode & LEV_SKIP_UPEXPAND)
   {
    lt = (int) strlen (text);
    lp = (int) strlen (pattern);
   }
 else
   {
    /****  convert strings to upper char  ****/
    lt = up_expand (text, tt,51, run_mode);
    lp = up_expand (pattern, pp,51, run_mode);

    if (lt >= 50)
      {
       lt = 2 * (int) strlen (text) + 3;
       tt = (char *) malloc ((unsigned) lt);

       if (tt == NULL)
         {
          if (run_mode & (TRACE_ADDR | TRACE_ERRORS | TRACE_LEV))
            {
             printf ("Error: \"malloc\" for %d Bytes failed.\n", lt);
            }
          return (10000);
         }
       lt = up_expand (text, tt, lt-1, run_mode);
      }

    if (lp >= 50)
      {
       lp = 2 * (int) strlen (pattern) + 3;
       pp = (char *) malloc ((unsigned) lp);

       if (pp == NULL)
         {
          if (run_mode & (TRACE_ADDR | TRACE_ERRORS | TRACE_LEV))
            {
             printf ("Error: \"malloc\" for %d Bytes failed.\n", lp);
            }
          if (tt != t_string)
            {
             free (tt);
            }
          return (10000);
         }
       lp = up_expand (pattern, pp, lp-1, run_mode);
      }

    text = tt;
    pattern = pp;
   }

 if (lt < 0  ||  lp < 0)
   {
     /****  internal error  ****/
     if (run_mode & (TRACE_ADDR | TRACE_LEV))
       {
        printf ("Internal error: could not initialize conv strings.\n");
       }
     return (10000);
   }

 if (limit == 0
 &&  *pattern != *text
 &&  *pattern != wildcard_any_string
 &&  *pattern != wildcard_any_char)
   {
    /****  no match  ****/
    if (tt != t_string)
      {
       free (tt);
      }
    if (pp != p_string)
      {
       free (pp);
      }
    return (10000);
   }

 if (lt < lp-1)
   {
    /****  switch "text" and "pattern"  ****/
    text = pp;
    pattern = tt;
    i = lt;
    lt = lp;
    lp = i;

    /****  re-calculate limit  ****/
    if (limit > 0)
      {
       l_res = calculate_limit (pattern,NULL,0);
       if (l_res.points < limit)
         {
          limit = l_res.points;
         }
      }
   }

 if (limit < 0)
   {
    limit = 0;
   }
 lim_1 = limit;
 if (lim_1 >= 7  &&  lim_1 <= 30)
   {
     lim_1 += 10;
   }

 if (lt >= 50)
   {
    i = (lt+1) * sizeof (int);
    d = (int *) malloc ((unsigned) i);

    if (d == NULL)
      {
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS | TRACE_LEV))
         {
          printf ("Error: \"malloc\" for %d Bytes failed.\n", i);
         }
       if (tt != t_string)
         {
          free (tt);
         }
       if (pp != p_string)
         {
          free (pp);
         }
       return (10000);
      }
   }

 if (run_mode & TRACE_LEV)
   {
    printf ("\nLevenshtein:  Strings = '%s'%s'", pattern, text);
    print_number (",  Limit", limit,-1);
    printf ("\n         ");

    for (k=1; k<=lt; k++)
      {
       printf ("   '%c'", text[k-1]);
      }
    printf ("\n");
   }

 /****  calculate initial values ( = zero'th column)  ****/
 d[0] = 0;
 for (k=1; k<=lt; k++)
   {
    d[k] = 10000;
   }

 /****  calculate distance matrix  ****/
 for (i=0; i<=lp; i++)
   {
    cp = (i == 0) ?  '\0' : *(pattern +i-1);
    p = (cp == wildcard_any_string  ||  cp == '.'  ||  cp == wildcard_any_char) ?   0 : 100;
    q = (cp == wildcard_any_string  ||  cp == '.') ?   0 : 100;
    r = (cp == wildcard_any_string  ||  cp == '.') ?   0 : 100;

    if (q > 0  &&  limit > 0)
      {
       if (i >= 2  &&  cp == pattern[i-2])
         {
          /****  "<x><x>"  ****/
          q = 30;
         }

       switch (cp)
         {
          case ' ':
          case '-':
          case '+':
          case '&':
          case '\'': q = 20;
                 break;
         }

       if (run_mode & LEV_COMPARE_GERMAN)
         {
          switch (cp)
            {
             case 'T':  if (i >= 2  &&  pattern[i-2] == 'D') q = 30;   /****  "DT"  ****/
                        if (pattern[i] == 'Z') q = 30;                 /****  "TZ"  ****/
                   break;

             case 'S':  if (i > 2  &&  pattern[i-3] == 'C'
                        && (pattern[i-2] == 'H'  ||  pattern[i-2] == 'K'))
                          {
                           /****  "CHS" or "CKS"  ****/
                           q = 0;
                          }
                        if (i >= 2  &&  pattern[i-2] == 'G') n = 0;   /****  "GS"  ****/
                        if (i >= 2  &&  pattern[i-2] == 'K') n = 0;   /****  "KS"  ****/
                   break;

             case 'K':  if (i > 1     /****  "CKS"  ****/
                        &&  pattern[i-2] == 'C'  &&  pattern[i] == 'S')
                          {
                           q = 0;
                          }
                        if (i >= 2  &&  pattern[i-2] == 'C') q = 0;   /****  "CK"  ****/
                   break;

             case 'H':  if (i >= 1)
                          {
                           q = 70;     /****  "^H"  ****/
                          }
                        if (i > 1     /****  "CHS"  ****/
                        &&  pattern[i-2] == 'C'  &&  pattern[i] == 'S')
                          {
                           q = 0;
                          }

                        if (i > 1  &&  pattern[i-2] == 'P')
                          {
                           /****  "PH"  ****/
                           q = 0;
                          }
                   break;

             case 'F':  if (i > 1  &&  pattern[i-2] == 'P')
                          {
                           /****  "PF"  ****/
                           q = 0;
                          }
                   break;

             case 'E':  if (i > 1  &&  (pattern[i-2] == 'A'
                        ||  pattern[i-2] == 'O'  ||  pattern[i-2] == 'U'))
                          {
                           /****  "AE", "OE" or "UE"  ****/
                           q = 0;
                          }
                        if (i >= 2  &&  pattern[i-2] == 'I') q = 30;   /****  "IE"  ****/
                   break;
            }
         }
      }

    d2 = d[0];
    d[0] = (i == 0) ?  0 : (d2 + q);
    col_min = d[0];

    for (k=1; k<=lt; k++)
      {
       /****  d[k] = minimum of three numbers  ****/
       d1 = d2;
       d2 = d[k];
       ct = text[k-1];
       n = (cp == text[k-1]) ?  0 : p;

       if (i >= 2  &&  k >= 2  &&  n != 0  &&  limit > 0)
         {
          /****  look for transpositions (e.g. "AB" -> "BA")  ****/
          cp2 = pattern[i-2];
          if (cp == text[k-2]  &&  cp2 == text[k-1]
          &&  cp != wildcard_any_string  &&  cp != '.'
           &&  cp != wildcard_any_char
          &&  cp2 != wildcard_any_string  &&  cp2 != '.'
           &&  cp2 != wildcard_any_char)
            {
             /****  transposition found  ****/
             n = 0;
            }
         }

       if (n > 0  &&  limit > 0)
         {
          switch (cp)
            {
             case ' ':
             case '-':
             case '+':
             case '&':
             case '\'': if (ct == ' '  ||  ct == '-'  ||  ct == '+'
                        ||  ct == '&'  ||  ct == '\''
                        ||  ct == wildcard_any_string)
                          {
                           /****  separator found  ****/
                           n = 20;
                          }
                   break;
            }
         }

       if ((run_mode & LEV_COMPARE_GERMAN)
       &&  n > 0  &&  limit > 0)
         {
          switch (cp)
            {
             case 'D':  if (ct == 'T') n = 60;  break;
             case 'T':  if (ct == 'D') n = 60;  break;

             case 'X':  if (ct == 'C'  &&  text[k+1] == 'S'
                        && (text[k] == 'H'  ||  text[k] == 'K'))
                          {
                           n = 60;
                          }
                        if (ct == 'K'  &&  text[k] == 'S')
                          {
                           n = 60;
                          }
                    break;

             case 'S':  if (ct == GERMAN_SS) n = 30;
                    break;
             case GERMAN_SS :  if (ct == 'S') n = 30;
                    break;

             case 'E':  if (ct == INTL_AE) n = 30;
                        if (ct == SCAND_AE) n = 30;
                        if (ct == 'A'  &&  text[k] == 'E') n = 30;

                        if (i > 0  &&  ct == 'O'  &&  pattern[i-1] == 'U'
                        && (text [k-1] == 'I'   ||   text [k-1] == 'Y'))
                          {
                           n = 0;
                          }
                        if (i > 0  &&  ct == 'A'
                        && (pattern[i-1] == 'I'  ||  pattern[i-1] == 'Y')
                        && (text [k-1] == 'I'  ||   text [k-1] == 'Y'))
                          {
                           n = 0;
                          }
                    break;

             case 'O':  if (ct == SCAND_AA) n = 70;
                        if (i > 0  &&  ct == 'E'  &&  text[k-1] == 'U'
                        && (pattern[i-1] == 'I'  ||  pattern[i-1] == 'Y'))
                          {
                           n = 0;
                          }
                    break;

             case 'A':  if (i > 0  &&  ct == 'E'
                        && (pattern[i-1] == 'I'  ||  pattern[i-1] == 'Y')
                        && (text [k-1] == 'I'   ||   text [k-1] == 'Y'))
                          {
                           n = 0;
                          }
                        if (ct == 'E'  &&  pattern[i] == 'E') n = 30;
                    break;

             case INTL_AE :
             case SCAND_AE :  if (ct == 'E') n = 30;
                    break;

             case SCAND_AA :  if (ct == 'O') n = 70;
                    break;

             case 'Y':  if (ct == 'I'  ||  ct == 'J'  ||  ct == INTL_UE)
                          {
                           n = 60;
                          }
                    break;

             case 'I':  if (ct == 'J'  ||  ct == 'Y') n = 60;  break;
             case 'J':  if (ct == 'I'  ||  ct == 'Y') n = 60;  break;
             case INTL_UE :  if (ct == 'Y') n = 60;  break;

             case 'C':  if (ct == 'X'  &&  pattern[i+1] == 'S'
                        && (pattern[i] == 'H'  &&  pattern[i] == 'H'))
                          {
                           n = 60;
                          }
                        if (ct == 'K'  ||  ct == 'Z') n = 60;
                    break;
             case 'K':  if (ct == 'C') n = 60;
                        if (ct == 'X'  &&  pattern[i] == 'S') n = 60;
                        if (ct == 'G') n = 70;
                        if (ct == 'Q') n = 30;
                    break;
             case 'Q':  if (ct == 'K') n = 30;  break;
             case 'Z':  if (ct == 'C') n = 60;  break;
             case 'G':  if (ct == 'K') n = 70;  break;

             case 'B':  if (ct == 'P') n = 60;  break;
             case 'P':  if (ct == 'B') n = 60;
                        if (ct == 'F'  &&  pattern[i] == 'H') n = 40;
                        if (ct == 'F'  &&  pattern[i] == 'F') n = 40;
                    break;
             case 'F':  if (ct == 'V') n = 60;
                        if (ct == 'P'  &&  text[k] == 'H') n = 40;
                    break;
             case 'V':  if (ct == 'F'  ||  ct == 'W') n = 60;
                    break;
             case 'W':  if (ct == 'V') n = 60;  break;
            }

          if (n >= 100
          &&  up_and_conv [(unsigned char) ct]
           == up_and_conv [(unsigned char) cp])
            {
             n = 30;
             if (ct == up_and_conv [(unsigned char) cp]
             ||  cp == up_and_conv [(unsigned char) ct])
               {
                n = 10;
               }
            }
         }

       x1 = d1 + n;
       x2 = d2 + q;

       if (x2 < x1)
         {
          x1 = x2;
         }

       n = r;
       if (n > 0  &&  limit > 0)
         {
          if (k >= 2  &&  ct == text[k-2])
            {
             /****  "<x><x>"  ****/
             n = 30;
            }

          if (ct == wildcard_any_string)
            {
             n = 20;
            }

          switch (ct)
            {
          /** case wildcard_any_string: **/
             case ' ':
             case '-':
             case '+':
             case '&':
             case '\'': n = 20;
                   break;
            }

          if (run_mode & LEV_COMPARE_GERMAN)
            {
             switch (ct)
               {
                case 'T':  if (k >= 2  &&  text[k-2] == 'D') n = 30;   /****  "DT"  ****/
                           if (text[k] == 'Z') n = 30;                 /****  "TZ"  ****/
                      break;

                case 'S':  if (k > 2     /****  "CHS"  ****/
                           &&  text[k-3] == 'C'  &&  text[k-2] == 'H')
                             {
                              n = 0;
                             }
                           if (k >= 2  &&  text[k-2] == 'G') n = 0;   /****  "GS"  ****/
                           if (k >= 2  &&  text[k-2] == 'K') n = 0;   /****  "KS"  ****/
                      break;

                case 'K':  if (k > 1      /****  "CKS"  ****/
                           &&  text[k-2] == 'C'  &&  text[k] == 'S')
                             {
                              n = 0;
                             }
                           if (k >= 2  &&  text[k-2] == 'C') n = 0;   /****  "CK"  ****/
                      break;

                case 'H':  if (k >= 1)
                             {
                              n = 70;      /****  "^H"  ****/
                             }
                           if (k > 1      /****  "CHS"  ****/
                           &&  text[k-2] == 'C'  &&  text[k] == 'S')
                             {
                              n = 0;
                             }
                           if (k > 1  &&  text[k-2] == 'P')
                             {
                              /****  "PH"  ****/
                              n = 0;
                             }
                      break;

                case 'F':  if (k > 1  &&  text[k-2] == 'P')
                             {
                              /****  "PF"  ****/
                              n = 0;
                             }
                      break;

                case 'E':  if (k > 1  &&  (text[k-2] == 'A'
                           ||  text[k-2] == 'O'  ||  text[k-2] == 'U'))
                             {
                              /****  "AE", "OE" or "UE"  ****/
                              n = 0;
                             }
                           if (k >= 2  &&  text[k-2] == 'I') n = 30;   /****  "IE"  ****/
                      break;
               }
            }
         }

       x2 = d[k-1] + n;
       d[k] = (x1 < x2) ?  x1 : x2;

       if (d[k] < col_min)
         {
          col_min = d[k];
         }
      }

    if (run_mode & TRACE_LEV)
      {
       if (i == 0)
         {
          printf ("    ");
         }
       else
         {
          printf ("'%c' ",cp);
         }

       for (k=0; k<=lt; k++)
         {
          if (d[k] <= limit)
            {
             printf (" %2d.%02d", (int)(d[k]/100), d[k]%100);
            }
          else
            {
             printf ("  ----");
            }
         }
       printf ("\n");
      }

    if (col_min > limit)
      {
       break;
      }
   }

 if (tt != t_string)
   {
     free (tt);
   }
 if (pp != p_string)
   {
     free (pp);
   }
 if (d != d_array)
   {
     free (d);
   }

  /****  make corrections for phonetic or semantic similarity  ****/
  col_min = d[lt];
  x1 = 0;
  x2 = 0;

  if (d[lt] >= 150  &&  ! (run_mode & LEV_COMPARE_INTERNAL))
    {
      i = 0;
      while (text[i] == pattern[i]  &&  text[i] != '\0')
        {
          i++;
        }

      if (i >= 2  &&  (text[i] == '\0'  ||  pattern[i] == '\0'))
        {
          /****  begin of one string matches the other  ****/
          /****  (e.g.  "Miller" vs. "Miller-Smith")    ****/
          x1 = 1;

          /****  one string ends, therefore:  ****/
          k = 0;
          if (text[i] == '\0')
            {
              while (pattern[i] != '\0')
                {
                  if (strchr ("- .", pattern[i]) == NULL
                  &&  pattern[i] != wildcard_any_string)
                    {
                      k++;
                    }
                  i++;
                }
            }
          else if (pattern[i] == '\0')
            {
              while (text[i] != '\0')
                {
                  if (strchr ("- .", text[i]) == NULL
                  &&  text[i] != wildcard_any_string)
                    {
                      k++;
                    }
                  i++;
                }
            }

          if (k == 0)
            {
              k++;
            }
          k = 50 * (k+1);    /** 100 * (k+1) / 2 **/

          if (k < d[lt])
            {
              d[lt] = k;
              if (run_mode & TRACE_LEV)
                {
                  printf ("Extra 1:  lev = %d.%02d\n",
                      (int) (d[lt] / 100), d[lt] % 100);
                }
            }
        }
    }

  if (d[lt] >= 100  && d[lt] <= 800  &&  x1 == 0
  && d[lt] <= lim_1  && ! (run_mode & LEV_COMPARE_INTERNAL))
    {
      if (d[lt] >= 200)
        {
          /****  check for multiple insert  ****/
          i = 0;
          while (text[i] == pattern[i]  &&  text[i] != '\0')
            {
              i++;
            }
          col_min = d[lt];
          k = (int) (d[lt] / 100);

          if (lt == lt + k
          &&  strcmp (text+i+k, pattern+i) == 0)
            {
              if (i >= 2
              ||  text[i+k] == ' '  ||  text[i+k] == '-')
                {
                  /****  multiple insert  ****/
                  if (k >= 4  &&  (pattern[i] == '\0'
                  ||  text[i+k] == ' '  ||  text[i+k] == '-'))
                    {
                      d[lt] = 50 * (k+1);    /** 100 * (k+1) / 2 **/

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 2:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 100);
                        }
                    }
                  else
                    {
                      d[lt] -= 100;
                      if (k >= 6)
                        {
                          d[lt] -= 100;
                        }
                      col_min = d[lt];

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 3:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 100);
                        }
                    }
                }
              else if (i == 0  &&  text[k] != '-'
              &&  text[k] != ' '  &&  text[k] != '/')
                {
                  /****  begin of strings differ  ****/
                  d[lt] += 100;
                  if (k >= 6)
                    {
                      d[lt] += 100;
                    }
                  col_min = d[lt];

                  if (run_mode & TRACE_LEV)
                    {
                      printf ("Extra 4:  lev = %d.%02d\n",
                          (int) (d[lt] / 100), d[lt] % 100);
                    }
                }
            }

          else if (lp == lt + k
          &&  strcmp (text+i, pattern+i+k) == 0)
            {
              if (i >= 2
              ||  pattern[i+k] == ' '  ||  pattern[i+k] == '-')
                {
                  /****  multiple insert  ****/
                  if (k >= 4  &&  (text[i] == '\0'
                  ||  pattern[i+k] == ' '  ||  pattern[i+k] == '-'))
                    {
                      d[lt] = 50 * (k+1);    /** 100 * (k+1) / 2 **/

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 5:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 10);
                        }
                    }
                  else
                    {
                      d[lt] -= 100;
                      if (k >= 6)
                        {
                          d[lt] -= 100;
                        }
                      col_min = d[lt];

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 6:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 100);
                        }
                    }
                }
              else if (i == 0  &&  pattern[k] != '-'
              &&  pattern[k] != ' '  &&  pattern[k] != '/')
                {
                  /****  begin of strings differ  ****/
                  d[lt] += 100;
                  if (k >= 6)
                    {
                      d[lt] += 100;
                    }
                  col_min = d[lt];

                  if (run_mode & TRACE_LEV)
                    {
                      printf ("Extra 7:  lev = %d.%02d\n",
                          (int) (d[lt] / 100), d[lt] % 100);
                    }
                }
            }
          else
            {
              if ((int)strlen (st) < lt  ||  (int)strlen (sp) < lp)
                {
                  /****  umlauts have been expanded  ****/
                  i = lev_diff (sp,st, limit,
                         ((run_mode | LEV_COMPARE_INTERNAL) & ~TRACE_LEV));

                  if (i < d[lt])
                    {
                      /****  non-expanded umlauts generate smaller lev  ****/
                      d[lt] = i;

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 8:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 100);
                        }
                    }
                }

              i = lt - 1;
              k = lp - 1;
              while (text[i] == pattern[k]  &&  i > 0  &&  k > 0)
                {
                  i--;
                  k--;
                }
              if (i+1 == d[lt]  ||  k+1 == d[lt])
                {
                  i = 0;
                  while (text[i] == pattern[i]  &&  text[i] != '\0')
                    {
                      i++;
                    }

                  if (i == 0  &&  (run_mode & LEV_COMPARE_GERMAN))
                    {
                      /****  begin of strings differ  ****/
                      if ((strncmp (text, "AN DEM ",7) == 0
                      ||   strncmp (text, "ZUM ",4) == 0)
                      && (strncmp (pattern, "AN DEM ",7) == 0
                      ||  strncmp (pattern, "ZUM ",4) == 0))
                        {
                          i = 1;
                        }
                      if ((strncmp (text, "AN DER ",7) == 0
                      ||   strncmp (text, "ZUR ",4) == 0)
                      && (strncmp (pattern, "AN DER ",7) == 0
                      ||  strncmp (pattern, "ZUR ",4) == 0))
                        {
                          i = 1;
                        }

                      if ((strncmp (text, "IN DE",5) == 0
                      ||   strncmp (text, "ZU DE",5) == 0)
                      &&  strchr ("MNR",text[6]) != NULL
                      && (strncmp (pattern, "IN DE",5) == 0
                      ||  strncmp (pattern, "ZU DE",5) == 0)
                      &&  strchr ("MNR",pattern[6]) != NULL)
                        {
                          i = 1;
                        }

                      if (i != 0)
                        {
                          /****  semantic similarity  ****/
                          d[lt] -= 100;

                          if (run_mode & TRACE_LEV)
                            {
                              printf ("Extra 9:  lev = %d.%02d\n",
                                  (int) (d[lt] / 100), d[lt] % 100);
                            }
                        }
                    }

                  if (i == 0)
                    {
                      /****  begin of strings differ       ****/
                      /****  (e.g. "Hedwig" and "Ludwig")  ****/
                      d[lt] += 100;

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 10:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 100);
                        }
                    }
                }

              if (d[lt] >= 200  &&  i > 0  &&  lt >= lp-1  &&  lt <= lp-1)
                {
                  strncpy (tt,text,4);
                  tt[4] = '\0';
                  strncpy (pp,pattern,4);
                  pp[4] = '\0';

                  i = lev_diff (pp,tt, limit,
                         ((run_mode | LEV_COMPARE_INTERNAL) & ~TRACE_LEV));

                  if (i >= 200  ||  text[0] != pattern[0])
                    {
                      d[lt] += 100;

                      if (run_mode & TRACE_LEV)
                        {
                          printf ("Extra 11:  lev = %d.%02d\n",
                              (int) (d[lt] / 100), d[lt] % 100);
                        }
                    }
                }
            }
        }

      /****  check for differences in ' ' or '-'  ****/
      while (*text == *pattern  &&  *text != '\0')
        {
          text++;
          pattern++;
        }

      /****  delete ' ' and '-'  ****/
      i = 0;
      k = 0;
      while (text[i] != '\0')
        {
          if (text[i] != ' '  &&  text[i] != '-')
            {
             text[k] = text[i];
             k++;
            }
          i++;
        }
      text[k] = '\0';
      x1 = k - i;

      i = 0;
      k = 0;
      while (pattern[i] != '\0')
        {
          if (pattern[i] != ' '  &&  pattern[i] != '-')
            {
             pattern[k] = pattern[i];
             k++;
            }
          i++;
        }
      pattern[k] = '\0';
      x1 = k - i;

      if (x1 != 0  ||  x2 != 0)
        {
          i = lev_diff (pattern,text, limit,
                 ((run_mode | LEV_COMPARE_INTERNAL) & ~TRACE_LEV));

          if (i < col_min)
            {
              d[lt] -= 100;
              if (i + 200 < d[lt])
                {
                  d[lt] -= 50;
                }
              if (i + 300 < d[lt])
                {
                  d[lt] -= 50;
                }

              if (run_mode & TRACE_LEV)
                {
                  printf ("Extra 12:  lev = %d.%02d\n",
                      (int) (d[lt] / 100), d[lt] % 100);
                }
            }
        }
    }
 n = d[lt];
 if (n > limit)
   {
    n = 10000;
   }
 if (run_mode & TRACE_LEV)
   {
    print_number ("Levenshtein distance", n,-1);
    printf ("\n\n");
   }

 return (n);
}





struct LEV_RESULT lev_x (char *pattern,
          char *text, char *desc, int max_points, int run_mode)
{
  struct LEV_RESULT l_res;
  int limit,diff;
  int empty_diff;
  int points,unknown;

  if (max_points < 100)
    {
     max_points = 100;
    }

  l_res = calculate_limit (pattern, text, run_mode);

/***
  max_points = l_res.max_points;
***/
  limit = l_res.points;
  empty_diff = l_res.empty_diff;
  unknown = l_res.diff;

  if (empty_diff != 0)
    {
     l_res = calculate_points
                  (100, limit, unknown, empty_diff, max_points);

     points = l_res.points;
     max_points = l_res.max_points;
     diff = l_res.diff;
     empty_diff = l_res.empty_diff;
    }
  else
    {
     /****  compare "pattern" with "text"  ****/
     diff = 0;
     if (strcmp (text,pattern) != 0)
       {
        diff = lev_diff (pattern, text, limit+50, 
                  (run_mode | LEV_COMPARE_INTERNAL));
       }

     points = 0;
     if (diff <= limit + 200)
       {
        l_res = calculate_points (diff, limit, unknown, 0, max_points);

        points = l_res.points;

      /***
        max_points = l_res.max_points;
        diff = l_res.diff;
        empty_diff = l_res.empty_diff;
      ***/

        if (diff == 0)
          {
           max_points = points;
          }
       }
     else
       {
        unknown = 0;
       }
    }

  if (run_mode & (TRACE_ADDR | TRACE_LEV))
    {
     printf ("%s", desc);
     print_number (":  points", points, -1);
     print_number (",  max_points", max_points, -1);
     print_number (",  diff", diff, -1);
     print_number (",  empty_diff", empty_diff, -1);
     printf ("\n");
    }

  l_res.points = points;
  l_res.max_points = max_points;
  l_res.diff = diff;
  l_res.empty_diff = empty_diff;

  return (l_res);
}




struct LEV_RESULT lev_zipcode_city (char *zip_code1, char *zip_code2, 
    char *city1, char *city2, char *desc, 
    int max_points, int max_points_city, int run_mode)
{
  struct LEV_RESULT l_res;
  int  i,n,limit,diff;
  int  empty_diff;
  int  points,unknown;
  char *s,temp1[51];
  char temp2[51];

  if (max_points < 100)
    {
     max_points = 100;
    }

  l_res = calculate_limit (zip_code1, zip_code2, run_mode);

/***
  max_points = l_res.max_points;
***/
  limit = l_res.points;
  empty_diff = l_res.empty_diff;
  unknown = l_res.diff;

  if (unknown == 0)
    {
     limit = 300;
    }

  if (empty_diff != 0)
    {
  /****
     l_res = calculate_points (diff, limit, unknown, empty_diff, max_points);
  ***/

     /****  compare cities instead of ZIP codes  ****/
     if (max_points < max_points_city)
       {
         max_points_city = max_points;
       }
     l_res = lev_x (city1, city2, desc, max_points_city, run_mode);

     points = l_res.points;
     max_points = l_res.max_points;
     diff = l_res.diff;
     empty_diff = l_res.empty_diff;

     return (l_res);
    }
  else
    {
     /****  compare "zip_code1" with "zip_code2"  ****/
     diff = 0;
     if (strcmp (zip_code2,zip_code1) != 0)
       {
        diff = lev_diff (zip_code1, zip_code2, 
                     limit + 100, run_mode);
       }

     if (diff > 100  &&  zip_code1[0] == zip_code2[0]
     &&  IS_DIGIT (zip_code1[0])
     &&  IS_DIGIT (zip_code1[1]))
       {
        i = (unsigned char) (zip_code1[1]) - (unsigned char) (zip_code2[1]);

        if (diff >= 400  &&  i >= -1  &&  i <= 1)
          {
           /****  ZIP codes are (e.g.) "56xxxx" and "57xxxx"  ****/
           diff -= 100;
          }

        if (diff >= 300  &&  (i < -1  ||  i > 1))
          {
           /****  ZIP codes are (e.g.) "50xxxx" and "57xxxx"  ****/
           diff += 100;
          }

        if (i == 0)
          {
           /****  first two digits are matching  ****/
           diff = 200;
           if (zip_code1[2] == zip_code2[2])
             {
              /****  first three digits are matching  ****/
              diff = 100;
             }
          }

        if ((run_mode & LEV_COMPARE_GERMAN)
        &&  diff > 200)
          {
           /****  check and compare cities  ****/
           up_expand (city1, temp1, 51,0);
           up_expand (city2, temp2, 51,0);

           n = 0;
           if (strcmp (temp1, temp2) == 0)
             {
              for (i=0; (s=zip_german_cities[i]) != NULL; i++)
                {
                 if (s[0] == zip_code1[0]  &&  s[1] == zip_code1[1]
                 &&  strcmp (s+3,temp1) == 0)
                   {
                    n += 1;
                   }

                 if (s[0] == zip_code2[0]  &&  s[1] == zip_code2[1]
                 &&  strcmp (s+3,temp2) == 0)
                   {
                    n += 2;
                   }
                }
             }

           if (n == 3)
             {
              diff = 200;
             }
          }
       }

     points = 0;
     if (diff <= limit + 200)
       {
        l_res = calculate_points (diff, limit, unknown, 0, max_points);

        points = l_res.points;

      /***
        max_points = l_res.max_points;
        diff = l_res.diff;
        empty_diff = l_res.empty_diff;
      ***/

        if (diff == 0)
          {
           max_points = points;
          }
       }
     else
       {
        unknown = 0;
       }
    }

  if (run_mode & (TRACE_ADDR | TRACE_LEV))
    {
     printf ("%s", desc);
     print_number (":  points", points, -1);
     print_number (",  max_points", max_points, -1);
     print_number (",  diff", diff, -1);
     print_number (",  empty_diff", empty_diff, -1);
     printf ("\n");
    }

  l_res.points = points;
  l_res.max_points = max_points;
  l_res.diff = diff;
  l_res.empty_diff = empty_diff;

  return (l_res);
}




struct LEV_RESULT lev_ph (char *a_text, char *a2_text,
    char *a_text_ph, char *a2_text_ph,
    char *desc, int max_points, int run_mode)
{
  struct LEV_RESULT l_res;
  int  n,limit;
  int  diff,points;
  int  empty_diff;
  int  unknown;
  double shrink_factor_ph = 0.1;
#ifdef USE_PHONET
  char a_temp[LENGTH_INTERNAL_VAR+1];
  char a2_temp[LENGTH_INTERNAL_VAR+1];
#endif

  if (run_mode & TRACE_LEV)
    {
     printf ("Strings before \"lev_ph\": '%s'%s'\n", a_text, a2_text);
    }

  l_res = calculate_limit (a_text, a2_text, run_mode);

/***
  max_points = l_res.max_points;
***/
  limit = l_res.points;
  empty_diff = l_res.empty_diff;
  unknown = l_res.diff;

#ifdef SHRINK_FACTOR_PH
    shrink_factor_ph = SHRINK_FACTOR_PH;
#endif

  if (empty_diff != 0)
    {
     l_res = calculate_points
                  (100, limit, unknown, empty_diff, max_points);

     points = l_res.points;
     max_points = l_res.max_points;
     diff = l_res.diff;
     empty_diff = l_res.empty_diff;
    }
  else
    {
     /****  compare "a_text" with "a2_text"  ****/
     diff = 0;
     if (strcmp (a_text,a2_text) != 0)
       {
        diff = lev_diff (a_text, a2_text, limit+50, run_mode);
       }

     if (diff > 100)
       {
   #ifdef USE_PHONET
        shrink_factor_ph = PHONET_SHRINK_FACTOR;

        if (run_mode & LEV_COMPARE_GERMAN)
          {
           /****  do phonetic conversion with "phonet"  ****/
           if (a_text_ph == NULL  ||  a_text_ph[0] == '\0')
             {
              (void) phonet (a_text, a_temp, LENGTH_WHOLE_NAME+1, PHONET_MODE);
              a_text_ph = a_temp;
             }

           if (a2_text_ph == NULL  ||  a2_text_ph[0] == '\0')
             {
              (void) phonet (a2_text, a2_temp, LENGTH_WHOLE_NAME+1, PHONET_MODE);
              a2_text_ph = a2_temp;
             }
          }
   #endif

        if (a_text_ph != NULL   &&  a_text_ph[0] != '\0'
        &&  a2_text_ph != NULL  &&  a2_text_ph[0] != '\0')
          {
           /****  compare phonetic variables  ****/
           n = lev_diff (a_text_ph, a2_text_ph, limit, run_mode);
           n = (int) ((4L * (long)(n+100)) / 3L);

           if (n < diff)
             {
              diff = n;
              max_points = (int) ((double)max_points * shrink_factor_ph);
             }
          }
       }

     points = 0;
     if (diff <= limit + 200)
       {
        l_res = calculate_points (diff, limit, unknown, 0, max_points);

        points = l_res.points;

      /***
        max_points = l_res.max_points;
        diff = l_res.diff;
        empty_diff = l_res.empty_diff;
      ***/

        if (diff == 0)
          {
           max_points = points;
          }
       }
     else
       {
        unknown = 0;
       }

     if (diff > limit + 200  &&  limit > 100)
       {
        /****  Look for names like "Miller" vs. "Miller-Smith"  ****/
        n = lev_2_name (a_text, a2_text,
                 (run_mode & ~TRACE_ADDR & ~TRACE_LEV));

        if (n+n <= limit)
          {
           diff = n + 100;
           l_res = calculate_points (n, limit, unknown, 0, max_points);

           points = (int) (3 * l_res.points / 4);
          }
       }
    }

  if (run_mode & (TRACE_ADDR | TRACE_LEV))
    {
     printf ("%s", desc);
     print_number (":  points", points, -1);
     print_number (",  max_points", max_points, -1);
     print_number (",  diff", diff, -1);
     print_number (",  empty_diff", empty_diff, -1);
     printf ("\n");
    }

  l_res.points = points;
  l_res.max_points = max_points;
  l_res.diff = diff;
  l_res.empty_diff = empty_diff;

  return (l_res);
}

/************************************************************/
/****  end of file "lev100.c"  ******************************/
/************************************************************/
