/*
 * demo.c
 * ------
 *
 * Demo program for address comparisons.
 *
 * Copyright (c):
 * 2007-2008:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *
 * SCCS: @(#) demo.c  1.2  2008-11-30
 *
 * This file is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this file; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLA.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. modify the underlying logic or translate this program into another
 * programming language), you must also release the changes under the terms
 * of the LGPL.
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "addr_ext.h"



/****  TO-DO:  If you want to use this program     ****/
/****  in a library, delete the following macro:   ****/
  #define DEMO_PROG_EXECUTABLE
/****  (and do not forget to call the function     ****/
/****  "cleanup_addr()" when you exit the program) ****/


/****  TO-DO:  In a non-German environment, delete        ****/
/****         "| COMPARE_LANGUAGE_GERMAN"  from RUN_MODE  ****/
#define RUN_MODE   (COMPARE_NORMAL | COMPARE_LANGUAGE_GERMAN)

#define SAMPLE_FILE    "samples.txt"

/****  test strings for "check_consistency"  ****/
#define  TEST_GENDER         "M"
#define  TEST_FIRST_NAME     "Fred"
#define  TEST_FAM_NAME       "Miller"
#define  TEST_STREET         "Main Street 789"
#define  TEST_ZIP_CODE       "12345"
#define  TEST_FULL_BIRTHDAY  "1978-09-25"



/****  variables for address comparison.          ****/
/****  (Note: We do not need to define all field  ****/
/****  since undefined variables (e.g. customer   ****/
/****  number) are treated as being empty).       ****/

static char first_gender [LENGTH_GENDER +1];
static char first_first_name [LENGTH_FIRST_NAME +1];
static char first_fam_name [LENGTH_FAM_NAME +1];
static char first_street [LENGTH_STREET +1];
static char first_zip_code [LENGTH_ZIP_CODE +1];
static char first_city [LENGTH_CITY +1];
static char first_country [LENGTH_COUNTRY +1];
static char first_full_birthday [LENGTH_FULL_BIRTHDAY +1];
static char first_birth_year [LENGTH_BIRTH_YEAR +1];
static char first_birth_month [LENGTH_BIRTH_MONTH +1];
static char first_birth_day [LENGTH_BIRTH_DAY +1];

static char second_gender [LENGTH_GENDER +1];
static char second_first_name [LENGTH_FIRST_NAME +1];
static char second_fam_name [LENGTH_FAM_NAME +1];
static char second_street [LENGTH_STREET +1];
static char second_zip_code [LENGTH_ZIP_CODE +1];
static char second_city [LENGTH_CITY +1];
static char second_country [LENGTH_COUNTRY +1];
static char second_full_birthday [LENGTH_FULL_BIRTHDAY +1];
static char second_birth_year [LENGTH_BIRTH_YEAR +1];
static char second_birth_month [LENGTH_BIRTH_MONTH +1];
static char second_birth_day [LENGTH_BIRTH_DAY +1];

static struct MAIL_ADDR first_addr[] =
   { { first_gender,        IS_GENDER     },
     { first_first_name,    IS_FIRST_NAME },
     { first_fam_name,      IS_FAM_NAME   },
     { first_street,        IS_STREET     },
     { first_zip_code,      IS_ZIP_CODE   },
     { first_full_birthday, IS_FULL_BIRTHDAY },  /***  do not change  ***/
     { first_city,          IS_CITY       },     /***  do not change  ***/
     { first_country,       IS_COUNTRY    },
     {   NULL,                 0          }
   };

static struct MAIL_ADDR second_addr[] =
   { { second_gender,       IS_GENDER     },
     { second_first_name,   IS_FIRST_NAME },
     { second_fam_name,     IS_FAM_NAME   },
     { second_street,       IS_STREET     },
     { second_zip_code,     IS_ZIP_CODE   },
     { second_full_birthday,IS_FULL_BIRTHDAY },  /***  do not change  ***/
     { second_city,         IS_CITY       },     /***  do not change  ***/
     { second_country,      IS_COUNTRY    },
     {   NULL,                 0          }
   };




/************************************************************/
/****  private (static) functions  **************************/
/************************************************************/

static void check_consistency (int run_mode)
{
 int i,k;
 int errors = 0;

 /****  generate second address  ****/
 StrNCpy (second_gender, TEST_GENDER, LENGTH_GENDER +1);
 StrNCpy (second_first_name, TEST_FIRST_NAME, LENGTH_FIRST_NAME+1);
 StrNCpy (second_fam_name, TEST_FAM_NAME, LENGTH_FAM_NAME+1);
 StrNCpy (second_street,   TEST_STREET,   LENGTH_STREET+1);
 StrNCpy (second_zip_code, TEST_ZIP_CODE, LENGTH_ZIP_CODE+1);
 strcpy (second_city,"");
 StrNCpy (second_full_birthday, TEST_FULL_BIRTHDAY, LENGTH_FULL_BIRTHDAY +1);
 strcpy (second_country,"");

 for (i=0; i<64; i++)
   {
    /****  generate first address  ****/
    for (k=0; k<6; k++)
      {
       if (i & (1<<k))
         {
          strcpy (first_addr[k].text, second_addr[k].text);
         }
       else
         {
          strcpy (first_addr[k].text, "");
         }
      }
    strcpy (first_country,"");

    /****  compare addresses in SELECT mode  ****/
    /****  (result must be "MAX_POINTS")     ****/
    k = compare_addr (first_addr, second_addr, 0, run_mode | DATABASE_SELECT);

    if (k != MAX_POINTS)
      {
       errors++;
       printf ("i = %d:\n",i);

       printf ("first address :  ");
       for (k=0; k<6; k++)
         {
          if (first_addr[k].text[0] != '\0')
            {
             printf ("  %s", first_addr[k].text);
            }
          else
            {
             printf ("  ''");
            }
         }
       printf ("\n");

       printf ("second address:  ");
       for (k=0; k<6; k++)
         {
          if (second_addr[k].text[0] != '\0')
            {
             printf ("  %s", second_addr[k].text);
            }
          else
            {
             printf ("  ''");
            }
         }
       printf ("\n");
       printf ("\n");

       k = compare_addr (first_addr, second_addr, 0, run_mode | DATABASE_SELECT | TRACE_ADDR);
       printf ("--> Error in SELECT mode:  points(=%d) != MAX_POINTS(=%d)\n",
                k, MAX_POINTS);

       printf ("\n");
       printf ("--------------------------------------------------------------------\n");
       printf ("\n");
      }

    /****  compare one address in "normal" mode  ****/
    /****  (result must be "MAX_POINTS")         ****/
    k = compare_addr (first_addr, first_addr, 0, run_mode & ~DATABASE_SELECT);

    if (k != MAX_POINTS)
      {
       errors++;
       printf ("i = %d:\n",i);

       printf ("first address :  ");
       for (k=0; k<6; k++)
         {
          if (first_addr[k].text[0] != '\0')
            {
             printf ("  %s", first_addr[k].text);
            }
          else
            {
             printf ("  ''");
            }
         }
       printf ("\n");

       printf ("second address:  ");
       for (k=0; k<6; k++)
         {
          if (second_addr[k].text[0] != '\0')
            {
             printf ("  %s", second_addr[k].text);
            }
          else
            {
             printf ("  ''");
            }
         }
       printf ("\n");
       printf ("\n");

       k = compare_addr (first_addr, first_addr, 0, (run_mode & ~DATABASE_SELECT) | TRACE_ADDR);
       printf ("--> Error in 'normal' mode:  points(=%d) != MAX_POINTS(=%d)\n",
                k, MAX_POINTS);

       printf ("\n");
       printf ("--------------------------------------------------------------------\n");
       printf ("\n");
      }

    /****  compare addresses in "normal" mode      ****/
    /****  (result must be less than "MAX_POINTS", ****/
    /****   if the two addresses differ)           ****/
    k = compare_addr (first_addr, second_addr, 0, run_mode & ~DATABASE_SELECT);

    if (k >= MAX_POINTS  &&  i != 46  &&  i != 47
    &&  i != 53  &&  i != 54  &&  i != 55  &&  i < 58)
      {
       errors++;
       printf ("i = %d:\n",i);

       printf ("first address :  ");
       for (k=0; k<6; k++)
         {
          if (first_addr[k].text[0] != '\0')
            {
             printf ("  %s", first_addr[k].text);
            }
          else
            {
             printf ("  ''");
            }
         }
       printf ("\n");

       printf ("second address:  ");
       for (k=0; k<6; k++)
         {
          if (second_addr[k].text[0] != '\0')
            {
             printf ("  %s", second_addr[k].text);
            }
          else
            {
             printf ("  ''");
            }
         }
       printf ("\n");
       printf ("\n");

       k = compare_addr (first_addr, second_addr, 0, (run_mode & ~DATABASE_SELECT) | TRACE_ADDR);
       printf ("--> Error 2 in 'normal' mode:  points(=%d) >= MAX_POINTS(=%d)\n",
                k, MAX_POINTS);

       printf ("\n");
       printf ("--------------------------------------------------------------------\n");
       printf ("\n");
      }
   }

 printf ("Consistency check:  ");
 if (errors == 0)
   {
    printf ("No inconsistencies found\n");
   }
 else
   {
    printf ("%d errors found\n", errors);
   }
}




static void compare_addr_from_sample_file (int run_mode, char *file)
{
  FILE *fr;
  int  i,k,n;
  char *s,text[201];

  fr = fopen (file,"r");
  if (fr == NULL)
    {
     printf ("Error: could not open file '%s'\n", file);
     return;
    }

  for (i=0; first_addr[i].text != NULL; i++)
    {
     strcpy (first_addr[i].text, "");
    }
  for (i=0; second_addr[i].text != NULL; i++)
    {
     strcpy (second_addr[i].text, "");
    }

  n = 0;
  while (! feof (fr))
    {
     /****  read data  ****/
     if (fgets (text,200,fr) != NULL)
       {
        i = (int) strlen (text);

        if (i > 0  &&  text[i-1] == '\n')
          {
           i--;
           if (i > 0  &&  text[i-1] == '\r')
             {
              i--;
             }
           text[i] = '\0';
          }

        if (i == 0  ||  text[0] == '#')
          {
           continue;
          }

        printf ("%s\n", text);

        s = text;
        k = 0;
        n += 10;

        if (n <= 10)
          {
           for (i=0; first_addr[i].text != NULL; i++)
             {
              strcpy (first_addr[i].text, "");
             }
          }
        else
          {
           for (i=0; second_addr[i].text != NULL; i++)
             {
              strcpy (second_addr[i].text, "");
             }
          }

        for (i=0; text[i] != '\0'; i++)
          {
           if (text[i] == ';'  &&  k < 8)
             {
              text[i] = '\0';

              switch (k+n)
                {
                 case 10:  StrNCpy (first_gender, s, LENGTH_GENDER +1);
                     break;
                 case 11:  StrNCpy (first_first_name, s, LENGTH_FIRST_NAME +1);
                     break;
                 case 12:  StrNCpy (first_fam_name, s, LENGTH_FAM_NAME +1);
                     break;

                 case 13:  StrNCpy (first_street, s, LENGTH_STREET +1);
                     break;
                 case 14:  StrNCpy (first_zip_code, s, LENGTH_ZIP_CODE +1);
                     break;
                 case 15:  StrNCpy (first_city, s, LENGTH_CITY +1);
                     break;
                 case 16:  StrNCpy (first_country, s, LENGTH_COUNTRY +1);
                     break;

                 case 17:  StrNCpy (first_full_birthday, s, LENGTH_FULL_BIRTHDAY +1);
                     break;

                 case 20:  StrNCpy (second_gender, s, LENGTH_GENDER +1);
                     break;
                 case 21:  StrNCpy (second_first_name, s, LENGTH_FIRST_NAME +1);
                     break;
                 case 22:  StrNCpy (second_fam_name, s, LENGTH_FAM_NAME +1);
                     break;

                 case 23:  StrNCpy (second_street, s, LENGTH_STREET +1);
                     break;
                 case 24:  StrNCpy (second_zip_code, s, LENGTH_ZIP_CODE +1);
                     break;
                 case 25:  StrNCpy (second_city, s, LENGTH_CITY +1);
                     break;
                 case 26:  StrNCpy (second_country, s, LENGTH_COUNTRY +1);
                     break;

                 case 27:  StrNCpy (second_full_birthday, s, LENGTH_FULL_BIRTHDAY +1);
                     break;
                }

              text[i] = ';';
              s = text + i+1;
              k++;
             }
          }

        if (n >= 20)
          {
           printf ("\n");
           n = compare_addr (first_addr, second_addr, 0,run_mode);

           printf ("Result:  %d points\n", n);

           printf ("\n");
           printf ("\n");
           printf ("============================================\n");
           printf ("\n");
           printf ("\n");

           for (i=0; first_addr[i].text != NULL; i++)
             {
              strcpy (first_addr[i].text, "");
             }
           for (i=0; second_addr[i].text != NULL; i++)
             {
              strcpy (second_addr[i].text, "");
             }

           n = 0;
          }
       }
    }

  fclose (fr);
}




static void unload_field_from_sample_file (int field_no, char *file)
{
  FILE *fr;
  int  i,k,n;
  char *s,text[201];
  char field[81];

  if (field_no < 0  ||  field_no > 10)
    {
     printf ("Error: field number %d is invalid (range: 1 - 10).\n", field_no);
     return;
    }

  fr = fopen (file,"r");
  if (fr == NULL)
    {
     printf ("Error: could not open file '%s'\n", file);
     return;
    }

  n = 0;
  while (! feof (fr))
    {
     /****  read data  ****/
     if (fgets (text,200,fr) != NULL)
       {
        i = (int) strlen (text);

        if (i > 0  &&  text[i-1] == '\n')
          {
           i--;
           if (i > 0  &&  text[i-1] == '\r')
             {
              i--;
             }
           text[i] = '\0';
          }

        if (i == 0  ||  text[0] == '#')
          {
           continue;
          }

        for (i=0; first_addr[i].text != NULL; i++)
          {
           strcpy (first_addr[i].text, "");
          }

        s = text; 
        k = 0;

        for (i=0; text[i] != '\0'; i++)
          {
           if (text[i] == ';'  &&  k < 8)
             {
              text[i] = '\0';
              StrNCpy (field, s,81);

              text[i] = ';';
              s = text + i+1;
              k++;

              if (k == 8)
                {
                 split_birthday (field, NULL, text, text+10, text+20);
                 k += 2;
                 switch (field_no)
                   {
                    case  8:  printf ("%s\n", text);     break;
                    case  9:  printf ("%s\n", text+10);  break;
                    case 10:  printf ("%s\n", text+20);  break;
                   }
                }
              else
                {
                 if (k == field_no)
                   {
                    printf ("%s\n", field);
                   }
                }
             }
          }
       }
    }

  fclose (fr);
}




void test_addresses (int argc, char *argv[], int run_mode)
{
 if (argc > 6)
   {
    /****  read first address  ****/
    StrNCpy (first_gender,   argv[1], LENGTH_GENDER+1);
    StrNCpy (first_first_name, argv[2], LENGTH_FIRST_NAME+1);
    StrNCpy (first_fam_name, argv[3], LENGTH_FAM_NAME+1);
    StrNCpy (first_street,   argv[4], LENGTH_STREET+1);
    StrNCpy (first_zip_code, argv[5], LENGTH_ZIP_CODE+1);
    StrNCpy (first_full_birthday, argv[6], LENGTH_FULL_BIRTHDAY+1);

    if (argc > 12)
      {
       /****  read second address  ****/
       StrNCpy (second_gender,   argv[7], LENGTH_GENDER+1);
       StrNCpy (second_first_name, argv[8], LENGTH_FIRST_NAME+1);
       StrNCpy (second_fam_name, argv[9],  LENGTH_FAM_NAME+1);
       StrNCpy (second_street,   argv[10], LENGTH_STREET+1);
       StrNCpy (second_zip_code, argv[11], LENGTH_ZIP_CODE+1);
       StrNCpy (second_full_birthday, argv[12], LENGTH_FULL_BIRTHDAY+1);
      }
    else
      {
       /****  second address = first address  ****/
       strcpy (second_gender, first_gender);
       strcpy (second_first_name, first_first_name);
       strcpy (second_fam_name, first_fam_name);
       strcpy (second_street,   first_street);
       strcpy (second_zip_code, first_zip_code);
       strcpy (second_full_birthday, first_full_birthday);
      }

    split_birthday (first_full_birthday, NULL,
            first_birth_day, first_birth_month, first_birth_year);
    split_birthday (second_full_birthday, NULL,
            second_birth_day, second_birth_month, second_birth_year);

    (void) compare_addr (first_addr, second_addr, 0, (run_mode | TRACE_ADDR));

    printf ("\n");
    printf ("--------------------\n");
    printf ("\n");
   }
}




static void calculate_weight_from_file (char *file)
{
  FILE *fr;
  int  i,k,n;
  long count,lines;
  long abbrev,max_count;
  double d;
  char last[201];
  char text[201];
  char max_text[201];

  if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
    {
     initialize_conv_strings (TRACE_ADDR);

     if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
       {
        /****  internal error  ****/
        printf ("Internal error: could not initialize conv strings.\n");
        return;
       }
    }

  fr = fopen (file,"r");
  if (fr == NULL)
    {
     printf ("Error: could not open file '%s'\n", file);
     return;
    }

  abbrev = 0;
  count = 0L;
  max_count = 0L;
  lines = 0L;
  strcpy (last,"");
  strcpy (max_text,"");

  while (! feof (fr))
    {
     /****  read data  ****/
     if (fgets (text,200,fr) != NULL)
       {
        i = (int) strlen (text);

        if (i > 0  &&  text[i-1] == '\n')
          {
           i--;
           if (i > 0  &&  text[i-1] == '\r')
             {
              i--;
             }
           text[i] = '\0';
          }

        /****  delete trailing blanks  ****/
        while (i > 0  &&  text[i-1] == ' ')
          {
           i--;
          }
        text[i] = '\0';

        if (i == 0)
          {
           continue;
          }
        if (i >= 150)
          {
           printf ("This function treats the whole line as one field.\n");
           printf ("Error: lines in file '%s' seem to contain more than one field.\n",
               file);
           fclose(fr);
           return;
          }

        /****  compare strings  ****/
        i = 0;
        while (text[i] == last[i]  &&  text[i] != '\0')
          {
           i++;
          }

        k = (unsigned char) text[i];
        n = (unsigned char) last[i];

        if (k != n  &&  ! IS_UMLAUT(k)  &&  ! IS_UMLAUT(n))
          {
           i = 0;

           if (IS_LOWER(k))  i += 1;
           else if (IS_UPPER(k)) i += 2;
           else if (IS_DIGIT(k)) i += 4;
           else  i += 100;    /***  100 !!  ***/

           if (IS_LOWER(n))  i -= 1;
           else if (IS_UPPER(n)) i -= 2;
           else if (IS_DIGIT(n)) i -= 4;
           else  i -= 200;    /***  200 !!  ***/

           if (i == 0  &&  k < n)
             {
              printf ("Error: wrong sorting order in line %ld:\n", lines);
              printf ("--> %d %s\n", n,last);
              printf ("--> %d %s\n", k,text);

              fclose (fr);
              return;
             }
          }

        if (k != n)
          {
           /****  new name found  ****/
           if (count >= max_count)
             {
              if (count > max_count
              ||  strcmp (max_text,"") == 0
              ||  (int)strlen (last) < (int)strlen (max_text))
                {
                 strcpy (max_text, last);
                }
              max_count = count;
             }
           count = 0L;
          }

        count++;
        lines++;
        if (strchr (text,'.') != NULL  &&  (int)strlen (text) < 7)
          {
           abbrev = 1;
          }
        strcpy (last,text);
       }
    }

  if (count > max_count)
    {
     max_count = count;
    }
  fclose (fr);

  if (max_count < 1L)
    {
      printf ("Error:  file doesn't contain any data.\n");
      return;
    }

  /****  evaluate results  ****/
  d = (double) lines / (double) max_count;

  printf ("Number of non-empty lines       :  %5ld\n", lines);
  printf ("Most frequent line is           :  '%s'\n", max_text);
  printf ("Occurences of most frequent line:  %5ld\n", max_count);

  sprintf (text,"%.2f", d);
  printf ("Filter factor :    1 : %s\n", text);
  printf ("\n");

  d = 1000.0 * calculate_ln (d) / calculate_ln (10.0);
  i = (int) (d);

  if (i < 250)
    {
      printf ("Weight is too small and will be enlarged.\n");
      i = 250;
    }
  if (i > 5000)
    {
      printf ("Weight is too large and will be lowered.\n");
      i = 5000;
    }

  printf ("Weight (in hundredths of points):   %d\n", i);

  if (abbrev != 0)
    {
      printf ("\n");
      printf ("Warning:  File seems to contain abbreviations (e.g. \"Chr.\").\n");
    }

  if (lines < 200L)
    {
      printf ("\n");
      printf ("Warning:  Number of lines is MUCH too small to get valid results.\n");
    }
  else if (lines < 1000L)
    {
      printf ("\n");
      printf ("Warning:  Number of lines may be too small to get valid results.\n");
    }
}





#ifdef DEMO_PROG_EXECUTABLE

int main (int argc, char *argv[])
{
 char *s;
 int  i;
 struct LEV_RESULT l_res;

 if (argc < 2
 ||  strcmp (argv[1], "-?") == 0
 ||  strcmp (argv[1], "-h") == 0
 ||  strcmp (argv[1], "-help") == 0)
   {
    printf ("Usage:  %s  -calculate_limit  <name>\n", argv[0]);
    printf (" or  :  %s  -compare_field  <name_1>  <name_2>\n", argv[0]);
    printf (" or  :  %s  -lev            <name_1>  <name_2>\n", argv[0]);
    printf ("\n");
    printf (" or  :  %s  [-compare_db_select]  <address_1>  [ <address_2> ]\n",
        argv[0]);
    printf ("\n");
    printf (" or  :  %s  -calculate_weight  <sorted_unload_file>\n", argv[0]);
    printf (" or  :  %s  -check_consistency\n", argv[0]);
    printf (" or  :  %s  -compare_addr_from_sample_file  [-compare_db_select]  [-trace]  [ <sample_file> ]\n",
                 argv[0]);
    printf ("        (If no filename is given, the file \"samples.txt\" will be read.)\n");
    printf (" or  :  %s  -unload_field_from_sample_file  <field_no>  [ <sample_file> ]\n", argv[0]);
    printf ("        (Range for <field_no>: 1 - 10  (1 = gender;  10 = year_of_birth)\n");
    printf ("        (If no filename is given, the file \"samples.txt\" will be read.)\n");
    printf ("\n");

    printf ("Demo program for address comparisons.\n");
    printf ("\n");
    printf ("<address> must be given as:\n");
    printf ("   <gender> <first_name> <fam_name> <street> <zip_code> <birthday>\n");
    printf ("Supported date formats for birthday:\n");
    printf ("   <year>-<month>-<day>,  <day>.<month>.<year>  and  <month>/<day>/<year>\n");
    printf ("\n");
    printf ("The option \"-compare_db_select\" does an \"asymmetrical\" comparison\n");
    printf ("which is needed (e.g.) in a database select.\n");
    return (1);
   }

 if (strncmp (argv[1],"-calculate_weight",17) == 0)
   {
    if (argc <= 2)
      {
        printf ("Error: wrong # of arguments.\n");
        return (1);
      }

    calculate_weight_from_file (argv[2]);
   }

 if (strncmp (argv[1],"-check_consistency",8) == 0)
   {
    check_consistency (RUN_MODE | DATABASE_SELECT);
   }

 if (strcmp (argv[1],"-calculate_limit") == 0)
   {
    if (argc <= 2)
      {
        printf ("Error: wrong # of arguments.\n");
        return (1);
      }

    l_res = calculate_limit (argv[2],argv[2],0);

    printf ("The following values are internal results\n");
    printf ("for use with the Levenshtein function (in file \"lev100ph.c\"):\n");
    printf ("\n");

    print_number ("limit     ", l_res.points,0);
    printf ("\n");
    print_number ("max_points", l_res.max_points,0);
    printf ("\n");
    print_number ("empty_diff", l_res.empty_diff,0);
    printf ("\n");
    print_number ("unknown   ", l_res.diff,0);
    printf ("\n");
   }

 if (strcmp (argv[1],"-compare_field") == 0)
   {
    if (argc != 4)
      {
        printf ("Error: wrong # of arguments.\n");
        return (1);
      }

    printf ("Comparing '%s' and '%s' (max_points = 10.00):\n",
                 argv[2],argv[3]);
    lev_2_ph (argv[2],argv[3], NULL, NULL, "",1000, (RUN_MODE | TRACE_LEV));
   }

 if (strcmp (argv[1],"-lev") == 0)
   {
    if (argc != 4)
      {
        printf ("Error: wrong # of arguments.\n");
        return (1);
      }

    printf ("Comparing '%s' and '%s' (max_points = 10.00):\n",
                 argv[2],argv[3]);
    lev_2_ph (argv[2],argv[3], NULL, NULL, "",1000, (RUN_MODE | TRACE_LEV));
   }

 if (strcmp (argv[1],"-compare_addr_from_sample_file") == 0)
   {
    /****  compare addresses from sample file  ****/
    i = RUN_MODE;
    if (argc >= 3  &&  strcmp (argv[2],"-trace") == 0)
      {
       i |= TRACE_ADDR;
       argc--;
       argv++;
      }
    if (argc >= 3  &&  strcmp (argv[2],"-compare_db_select") == 0)
      {
       i |= DATABASE_SELECT;
       argc--;
       argv++;
      }
    if (argc >= 3  &&  strcmp (argv[2],"-trace") == 0)
      {
       i |= TRACE_ADDR;
       argc--;
       argv++;
      }

    s = argv[2];
    if (s == NULL)
      {
       s = SAMPLE_FILE;
      }
    compare_addr_from_sample_file (i, s);
   }

 if (argc >= 3  &&  strcmp (argv[1],"-unload_field_from_sample_file") == 0)
   {
    /****  unload field from sample file  ****/
    i = atoi (argv[2]);
    s = argv[3];
    if (s == NULL)
      {
       s = SAMPLE_FILE;
      }
    unload_field_from_sample_file (i, s);
   }

 if (argc > 6)
   {
    if (argc > 7  &&  argv[1][0] == '-'
    &&  strcmp (argv[1],"-compare_db_select") == 0)
      {
       test_addresses (argc-1, argv+1, RUN_MODE | DATABASE_SELECT);
      }
    else
      {
       test_addresses (argc, argv, RUN_MODE);
      }
   }

 cleanup_addr();
 return (0);
}

#endif

/************************************************************/
/****  end of file "demo.c"  ********************************/
/************************************************************/
