// $Id: ThreadPool.h 302 2007-05-03 15:20:29Z olau $

#pragma once

#include "stdafx.h"

class ThreadPool;
class Job;
class Mutex;

///////////////////////////////////////////////////////////////////
// class WorkerThread
///////////////////////////////////////////////////////////////////

class WorkerThread : public Thread
{
public:
    WorkerThread(ThreadPool *pool, HANDLE hTriggerEvent, HANDLE hReadyEvent, LPCSTR name = NULL)
        : pool(pool), hTriggerEvent(hTriggerEvent), hReadyEvent(hReadyEvent), name(name), doQuit(false)
    { }

    /// The worker thread's main loop processes the thread pool's job queue
    /// and calls signalReady() when there are no more jobs left.
    THREADRESULT run();

protected:
    bool doQuit;
    friend class ThreadPool;

private:
    HANDLE hTriggerEvent;
    HANDLE hReadyEvent;
    ThreadPool *pool;
    LPCSTR name;

    void waitForTrigger();
    int signalReady();
};


///////////////////////////////////////////////////////////////////
// class ThreadPool
///////////////////////////////////////////////////////////////////

class ThreadPool
{
public:
    /// ThreadPool constructor.
    /// Create a thread pool with the given number of worker threads.
    ThreadPool(int numThreads);

    /// ThreadPool destructor.
    /// Wait for all worker threads to return, clear job queue.
    /// This method is thread-safe.
    ~ThreadPool();

    void setMaximumNumberOfThreads(int);

    /// Add a job to the job queue.
    /// This method is thread-safe.
    void enqueue(Job *);

    /// Wait for worker threads to send ready event.
    void waitForAll();

    /// Start worker threads unless already running (\see start()).
    /// Then signal worker threads to take up their jobs (\see signalAll()).
    int run();

    /// Signal worker threads to quit (\see quitAll()),
    /// then trigger worker threads to process the job queue (\see signalAll()).
    void stop();

    /// Return next job in queue (returns NULL if queue is empty).
    Job *nextJob();

    static const int ThreadHardLimit = 128;

    enum ErrorCode 
    {
        NoError = 0,
        ThreadHardLimitReached,
        CreateEventFailed,
        SetEventFailed,
        InvalidHandle,
    };

private:
    WorkerThread *pool[ThreadHardLimit];
    HANDLE hReadyEvent[ThreadHardLimit];
    HANDLE hTriggerEvent[ThreadHardLimit];
    int numberOfThreads;
    int maximumNumberOfThreads;
    Mutex *m_mutex;
    std::queue <Job *, std::deque<Job *> > jobs;

    /// Trigger worker threads to process the job queue.
    int signalAll();

    /// Signal worker threads to quit by setting the doQuit flag.
    void quitAll();
};
