/* $Id: TrieMap.h 1211 2006-10-16 16:06:27Z olau $ */

/* Copyright (C) 2006 Oliver Lau <ola@ctmagazin.de>,
                      Heise Zeitschriften Verlag.
   Alle Rechte vorbehalten.
*/

#ifndef __TRIEMAP_H_
#define __TRIEMAP_H_


/////////////////////////////////////////////////////
///
/// TrieNode-Klasse
///
/////////////////////////////////////////////////////

template <class T>
class TrieNode {
  static const int num_children = 256; // pow(2, sizeof(char))

public: // variables
  bool m_end_of_word;
  T m_val;
  TrieNode *m_child[num_children];

public: // methods

  /// Konstruktor.
  TrieNode() : m_end_of_word(false)
  {
    clear();
  }

  /// Copy-Konstruktor.
  TrieNode(T &val) : m_end_of_word(false), m_val(val)
  {
    clear();
  }

  /// Destruktor.
  ~TrieNode()
  {
    for (int i = 0; i < num_children; ++i)
    {
      if (m_child[i] != NULL)
        delete m_child[i];
      m_child[i] = NULL;
    }
  }

private: // methods

  /// Smtliche Kinderknoten auf Null setzen.
  void clear(void)
  {
    for (int i = 0; i < num_children; ++i)
      m_child[i] = NULL;
  }
};


/////////////////////////////////////////////////////
///
/// TrieMap-Klasse
///
/////////////////////////////////////////////////////

template <class T>
class TrieMap {
private: // variables

  /// Maximal erlaubte Schlssellnge.
  static const int MAX_WORD_LENGTH = 100;

  /// Zeiger auf die Trie-Wurzel.
  TrieNode<T> *m_root;

  /// Zhler fr Schlssel im Trie.
  int m_size;

public: // methods

  /// Konstruktor.
  TrieMap(void) : m_size(0)
  {
    m_root = new TrieNode<T>;
  }

  /// Destruktor.
  ~TrieMap(void)
  {
    delete m_root;
  }

  /// Anzahl der Elemente im Trie zurckgeben.
  inline int size(void)
  {
    return m_size;
  }

  const T value(const char *key) const;

  /// Subscript-Operator fr Lesezugriff auf das assoziative Array.
  inline const T operator[] (const char *key) const
  {
    return value(key);
  }

  T &operator[] (const char *key);

};


/// Den einem Schlssel zugeordneten Wert zurckgegeben.
/// <param name="key">Schlssel</param>
/// <return>Wert</return>
template<typename T>
const T TrieMap<T>::value(const char *key) const
{
  int ch;
  // Suche an der Wurzel des Trie beginnen
  TrieNode<T> *node = m_root;
  int i = 0;
  // Schlssel zeichenweise bis zum Ende abarbeiten
  while ((ch = (int) key[i++]) != 0)
  {
    // zum Kindknoten weiterhangeln, wenn vorhanden
    if (node->m_child[ch] != NULL)
      node = node->m_child[ch];
  }
  return (node->m_end_of_word)? node->m_val : T();
}


/// Subscript-Operator fr Lese- und Schreibzugriff auf
/// das assoziative Array.
/// <param name="key">Schlssel<param>
/// <return>Referenz auf den dem Schlssel zugeordneten Wert</return>
template<typename T>
T & TrieMap<T>::operator[](const char *key)
{
  int ch;
  // Suche an der Wurzel des Trie beginnen
  TrieNode<T> *node = m_root;
  int i = 0;
  // Schlssel zeichenweise bis zum Ende abarbeiten
  while ((ch = (int) key[i++]) != 0)
  {
    if (node->m_child[ch] == 0)
    {
      // wenn kein Kindknoten vorhanden, einen neuen erzeugen
      node->m_child[ch] = new TrieNode<T>;
      ++m_size;
    }
    node = node->m_child[ch];
  }
  node->m_end_of_word = true;
  return node->m_val;
}


#endif /* __TRIEMAP_H_ */
