using System;
using System.IO;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;

namespace Sudoku
{
    public partial class SudokuForm : Form
    {
        TextBox ActiveTextBox;
        SudokuField Field, StartField, MemField;
        SudokuList[,] OwnLists, MemLists;
        bool EmptyMode;
        Point MouseLocation;
        Stopwatch ElapsedTime;

        public SudokuForm()
        {
            InitializeComponent();
            Field = new SudokuField();
            StartField = new SudokuField();
            MemField = new SudokuField();
            OwnLists = new SudokuList[SudokuField.nn, SudokuField.nn];
            MemLists = new SudokuList[SudokuField.nn, SudokuField.nn];
            ClearAll(true);
        }

        private void ClearAll(bool Empty)
        {
            EmptyMode = Empty;
            Field.Clear();
            MemField.Clear();
            for (int i = 0; i < SudokuField.nn; i++)
                for (int j = 0; j < SudokuField.nn; j++)
                {
                    OwnLists[i, j] = new SudokuList("");
                    MemLists[i, j] = new SudokuList("");
                }
            OutputField(true, false);
        }

        private void EmptyButton_Click(object sender, EventArgs e)
        {
            ClearAll(true);
            Timer.Enabled = false;
            if (ElapsedTime != null)
                ElapsedTime.Stop();
            StatusLabel.Text = "";
        }

        private void NewButton_Click(object sender, EventArgs e)
        {
            ClearAll(false);
            Timer.Enabled = false;
            Cursor.Current = Cursors.WaitCursor;
            int FieldCount = 0;
            foreach (Control c in SudokuPanel.Controls)
                if (c is RadioButton && ((RadioButton)c).Checked)
                    FieldCount = int.Parse(c.Tag.ToString());
            for (int i = 0; i < 10; i++)
            {
                SudokuField NewField = new SudokuField(FieldCount);
                if (Field.DigitCount == 0 || NewField.DigitCount < Field.DigitCount)
                    Field = NewField;
                if (Field.DigitCount == FieldCount)
                    break;
            }
            OutputField(true, false);
            StartField.Copy(Field);
            MemField.Copy(Field);
            Cursor.Current = Cursors.Default;
            if (ElapsedTime != null)
                ElapsedTime.Stop();
            Timer.Enabled = true;
        }

        private void Timer_Tick(object sender, EventArgs e)
        {
            if (ElapsedTime == null || !ElapsedTime.IsRunning)
            {
                ElapsedTime = new Stopwatch();
                ElapsedTime.Start();
            }
            if (Field.DigitCount == SudokuField.nn * SudokuField.nn)
            {
                Timer.Enabled = false;
                ElapsedTime.Stop();
            }
            int Time = (int)ElapsedTime.Elapsed.TotalSeconds;
            StatusLabel.Text = String.Format("Zeit = {0:D}:{1:D2} min  ({2:D}/{3:D} Felder)", 
                                             Time / 60, Time % 60, Field.DigitCount,
                                             SudokuField.nn * SudokuField.nn);
        }

        private void RestartButton_Click(object sender, EventArgs e)
        {
            if (EmptyMode)
            {
                EmptyMode = false;
                Field.InitLists();
                StartField.Copy(Field);
            }
            else
            {
                ClearAll(false);
                Field.Copy(StartField);
            }
            MemField.Copy(Field);
            OutputField(true, false);
            Timer.Enabled = true;
        }

        private void HintCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            Field.InitLists();
            OutputField(false, false);
        }

        private void SaveButton_Click(object sender, EventArgs e)
        {
            MemField.Copy(Field);
            MemLists = (SudokuList[,])OwnLists.Clone();
            OutputField(false, true);
        }

        private void RestoreButton_Click(object sender, EventArgs e)
        {
            Field.Copy(MemField);
            OwnLists = (SudokuList[,])MemLists.Clone();
            OutputField(false, true);
            Timer.Enabled = true;
        }

        private string TextBoxString(SudokuList List)      
        {
            string ListString = List.ToString(), Result = "";
            for (int Digit = 1; Digit <= SudokuField.nn; Digit++)
            {
                string S = SudokuField.DigitToString(Digit);
                Result += ListString.Contains(S) ? S : " ";
                Result += Digit % SudokuField.n == 0 ? "" : " ";
                Result += Digit % SudokuField.n == 0 && Digit != SudokuField.nn ? "\r\n" : "";
            }
            return Result;
        }

        private void OutputField(bool ForceLock, bool ForceReadOnly)
        {
            HintCheckBox.Enabled = !EmptyMode;
            LockUnlockButton.Enabled = !EmptyMode;
            foreach (Control c in Controls)
                if (c is TextBox && c.Tag != null)
                {
                    TextBox tb = (TextBox)c;
                    int Tag = int.Parse(tb.Tag.ToString()),
                        i = Tag / 9, j = Tag % 9;
                    if (i >= SudokuField.nn || j >= SudokuField.nn)
                        continue;                        // for testing order 2 Sudokus
                    int Digit = Field.Digits[i, j];
                    tb.TextChanged -= new System.EventHandler(this.SudokuBox_TextChanged);
                    if (Digit > 0 && Digit <= SudokuField.nn || EmptyMode)
                    {
                        if (EmptyMode && Digit == 0)
                            tb.ReadOnly = false;
                        if (ForceReadOnly && Digit > 0)
                            tb.ReadOnly = true;
                        if (ForceLock && !EmptyMode)
                        {
                            tb.ReadOnly = true;
                            Field.Lock(i, j, Digit);
                            tb.BackColor = Color.Lavender;
                        }
                        else
                            if (!tb.ReadOnly)
                                tb.BackColor = Color.Empty;
                        tb.Font = new Font("Verdana", 28f, tb.ReadOnly ? FontStyle.Bold : 0, GraphicsUnit.Point);
                        tb.Text = Digit != 0 ? SudokuField.DigitToString(Digit) : "";
                    }
                    else
                    {
                        tb.ReadOnly = false;
                        tb.BackColor = Color.Empty;
                        tb.Font = new Font("Courier New", 9f, 0, GraphicsUnit.Point);
                        tb.Text = TextBoxString(HintCheckBox.Enabled && HintCheckBox.Checked ? 
                                                Field.Lists[Tag / 9, Tag % 9] :
                                                OwnLists[Tag / 9, Tag % 9]);
                    }
                    tb.TextChanged += new System.EventHandler(this.SudokuBox_TextChanged);
                }
        }

        private void SudokuBox_TextChanged(object sender, EventArgs e)
        {
            TextBox tb = sender as TextBox;
            int Tag = int.Parse(tb.Tag.ToString()),
                i = Tag / 9, j = Tag % 9;
            if (i >= SudokuField.nn || j >= SudokuField.nn) // for testing order 2 Sudokus
                return;
            tb.TextChanged -= new System.EventHandler(this.SudokuBox_TextChanged);
            if (EmptyMode)
            {
                if (tb.Focused)
                {
                    tb.SelectionStart = 0; 
                    tb.SelectionLength = 1;
                }
                int Digit = 0;
                try
                {
                    Digit = SudokuField.SubstringToDigit(tb.Text);
                }
                catch
                {
                }
                if (Field.TestRules(i, j, Digit))
                    Field.Digits[i, j] = Digit;
                else
                {
                    Field.Digits[i, j] = 0;
                    tb.Text = "";
                    tb.SelectedText = "";
                    if (Digit != 0)
                        Console.Beep();
                }
            }
            else
            {
                SudokuList List = new SudokuList(tb.Text);
                foreach (int Digit in List)
                    if (!(Field.TestRules(i, j, Digit)))
                    {
                        List -= Digit;
                        Console.Beep();
                    }
                int Cursor = tb.SelectionStart;
                tb.Text = TextBoxString(List);
                tb.SelectionStart = Cursor < 18 ? Cursor : 18;
                if (HintCheckBox.Enabled && HintCheckBox.Checked)
                    Field.Lists[i, j] = List;
                else
                    OwnLists[i, j] = List;
            }
            tb.TextChanged += new System.EventHandler(this.SudokuBox_TextChanged);
        }

        private void SudokuBox_Enter(object sender, EventArgs e)
        {
            ActiveTextBox = sender as TextBox;
        }

        private void SudokuBox_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (LockUnlockButton.Enabled)
            {
                TextBox tb = sender as TextBox;
                LockUnlockButton_Click(null, null);
                tb.SelectionLength = 0;
            }
        }

        private void LockUnlockButton_Click(object sender, EventArgs e)
        {
            int Tag = int.Parse(ActiveTextBox.Tag.ToString()),
                i = Tag / 9, j = Tag % 9;
            if (i >= SudokuField.nn || j >= SudokuField.nn) // for testing order 2 Sudokus
                return;
            SudokuList List = new SudokuList(ActiveTextBox.Text);
            if (ActiveTextBox.ReadOnly)
            {
                ActiveTextBox.ReadOnly = false;
                Field.Digits[i, j] = 0;
            }
            else
            {
                int Count = 0;
                foreach (int Dig in List)
                    Count++;
                if (Count != 1)
                {
                    Console.Beep();
                    return;
                }
                int Digit = 1;
                while (!(List * Digit))
                    Digit++;
                if (Field.TestRules(i, j, Digit))
                {
                    Field.Lock(i, j, Digit);
                    ActiveTextBox.ReadOnly = true;
                }
                else
                    Console.Beep();
            }
            OutputField(false, false);
        }

        private void SolveButton_Click(object sender, EventArgs e)
        {
            if (EmptyMode)
                RestartButton_Click(null, null);
            Timer.Enabled = false;
            Cursor.Current = Cursors.WaitCursor;
            SudokuStatus Status = Field.Solve();
            StatusLabel.Text = String.Format("Status = {0:S} ({1:D} Iterationen)", Status.ToString(), Field.Iterations);
            if (Status != SudokuStatus.OneSolution)
                Console.Beep();
            OutputField(false, false);
            Cursor.Current = Cursors.Default;
        }

        private void ToolStrip_MouseMove(object sender, MouseEventArgs e)
        {
            if (ToolStrip.Capture)
            {
                Point MousePos = MousePosition;
                SetDesktopLocation(Location.X + MousePos.X - MouseLocation.X,
                                   Location.Y + MousePos.Y - MouseLocation.Y);
                MouseLocation = MousePos;            
            }
        }

        private void ToolStrip_MouseUp(object sender, MouseEventArgs e)
        {
            ToolStrip.Capture = false;
        }

        private void ToolStrip_MouseDown(object sender, MouseEventArgs e)
        {
            MouseLocation = MousePosition;
            ToolStrip.Capture = true;
        }

        private void ToolStripExitButton_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void ToolStrip_MouseHover(object sender, EventArgs e)
        {
            Activate();
        }
    }
}

