using System;
using System.Collections;

namespace Sudoku
{
    public partial class SudokuField
    {
        public const int n = 3;        // Sudoku n * n blocks of size n * n (n = order of Sudoku)
        public const int nn = n * n;   // Sudoku has nn * nn cells

        //private const string nnDigits = "1234";
        private const string nnDigits = "123456789";
        //private const string nnDigits = "0123456789ABCDEF";
        //private const string nnDigits = "ABCDEFGHIJKLMNOPQRSTUVWXY";

        public int[,] Digits;
        public SudokuList[,] Lists;

        /// <summary>
        /// Creates a new Sudoku field
        /// </summary>
        public SudokuField()
        {
            Digits = new int[nn, nn];
            Lists = new SudokuList[nn, nn];
            Clear();
        }

        public SudokuField(SudokuField ToCopy)
        {
            Copy(ToCopy);
        }

        public SudokuField(string Init)
        {
            Digits = new int[nn, nn];
            Lists = new SudokuList[nn, nn];
            Clear();
            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                    Digits[i, j] = SubstringToDigit(Init.Substring(nn * i + j, 1));
        }

        static public int SubstringToDigit(string Substring)
        {
            return nnDigits.IndexOf(Substring) + 1;
        }

        /// <summary>
        /// Clears the Sudoku field.
        /// </summary>
        public void Clear()
        {
            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                {
                    Digits[i, j] = 0;
                    Lists[i, j] = new SudokuList();
                }
        }

        /// <summary>
        /// Copies a Sudoku field.
        /// </summary>
        /// <param name="ToCopy">SudokuField to copy</param>
        public void Copy(SudokuField ToCopy)
        {
            Digits = (int[,])ToCopy.Digits.Clone();      // Clone's flat copy is enough, 
            Lists = (SudokuList[,])ToCopy.Lists.Clone(); // because every alteration of a
        }                                                // SudokuList creates a new object

        /// <summary>
        /// Returns a readable string representing SudokuField.
        /// </summary>
        public override string ToString()
        {
            string DigitString = "", ListString = "";
            foreach (int Digit in Digits)
                DigitString += DigitToString(Digit);
            foreach (SudokuList List in Lists)
                ListString += " " + List.ToString();
            return DigitString + ListString;
        }

        public static string DigitToString(int Digit)
        {
            return Digit > 0 ? nnDigits.Substring(Digit - 1, 1) : "_";
        }

        /// <summary>
        /// Tests if Digit fits into the given Sudoku cell (using Sudoku rules only).
        /// </summary>
        /// <param name="i">Column of cell</param>
        /// <param name="j">Row of cell</param>
        /// <param name="Digit">Digit of cell</param>
        /// <returns>true, if Digit fits into Sudoku cell</returns>
        public bool TestRules(int i, int j, int Digit)
        {
            if (Digit <= 0 || Digit > nn)
                return false;
            for (int ii = 0; ii < nn; ii++)
                if (ii != i && Digits[ii, j] == Digit)
                    return false;
            for (int jj = 0; jj < nn; jj++)
                if (jj != j && Digits[i, jj] == Digit)
                    return false;
            for (int ii = i / n * n; ii < (i + n) / n * n; ii++)
                for (int jj = j / n * n; jj < (j + n) / n * n; jj++)
                    if (ii != i && jj != j && Digits[ii, jj] == Digit)
                        return false;
            return true;
        }

        /// <summary>
        /// Sets the Digit at the given Sudoku cell and deletes all 
        /// possibilities which are not allowed anymore due to this Digit.
        /// </summary>
        /// <param name="i">Column of cell</param>
        /// <param name="j">Row of cell</param>
        /// <param name="Digit">Digit of cell</param>
        /// <returns>this</returns>
        public SudokuField Lock(int i, int j, int Digit)
        {
            Digits[i, j] = Digit;
            Lists[i, j] = Digit;             // this single possibility left
            for (int ii = 0; ii < nn; ii++)
                if (ii != i)
                    Lists[ii, j] -= Digit;
            for (int jj = 0; jj < nn; jj++)
                if (jj != j)
                    Lists[i, jj] -= Digit;
            for (int ii = i / n * n; ii < (i + n) / n * n; ii++)
                for (int jj = j / n * n; jj < (j + n) / n * n; jj++)
                    if (ii != i && jj != j)
                        Lists[ii, jj] -= Digit;
            return this;
        }

        /// <summary>
        /// Initializes possibility lists by calling Lock(i,j) for every given.
        /// </summary>
        /// <returns>this</returns>
        public SudokuField InitLists()
        {
            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                    Lists[i, j] = new SudokuList();
            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                    if (Digits[i, j] != 0)
                        Lock(i, j, Digits[i, j]);
            return this;
        }

        /// <summary>
        /// Counts the number of Digits in Digits.
        /// </summary>
        public int DigitCount
        {
            get { return GetDigitCount(); }
        }

        private int GetDigitCount()
        {
            int Result = 0;
            foreach (int Digit in Digits)
                if (Digit != 0)
                    Result++;
            return Result;
        }
    }

    public class SudokuList
    {
        private int Flags;

        public SudokuList()
        {
            Flags = ((1 << SudokuField.nn) - 1) << 1;  // all nn Digits allowed (Bit 0 not used)
        }

        public SudokuList(string Init)
        {
            for (int i = 0; i < Init.Length; i++)
                Flags = (this + SudokuField.SubstringToDigit(Init.Substring(i, 1))).Flags;
        }

        private SudokuList(int Flags)
        {
            this.Flags = Flags;
        }

        public static implicit operator SudokuList(int Digit)
        {
            return new SudokuList(Digit > 0 ? 1 << Digit : 0);
        }

        public static SudokuList operator +(SudokuList List, int Digit)
        {
            int Flags = List.Flags | (Digit > 0 ? 1 << Digit : 0);
            return Flags != List.Flags ? new SudokuList(Flags) : List;
        }

        public static SudokuList operator -(SudokuList List, int Digit)
        {
            int Flags = List.Flags & ~(Digit > 0 ? 1 << Digit : 0);
            return Flags != List.Flags ? new SudokuList(Flags) : List;
        }

        public static bool operator *(SudokuList List, int Digit)
        {
            return (List.Flags & (1 << Digit)) != 0;
        }
        
        public IEnumerator GetEnumerator()
        {
            for (int Digit = 1; Digit <= SudokuField.nn; Digit++)
                if ((Flags & (1 << Digit)) != 0)
                    yield return Digit;
        }

        public override string ToString()
        {
            string Result = "";
            foreach (int Digit in this)
                Result += SudokuField.DigitToString(Digit);
            return Result;
        }
    }
}
