/* $Id: CoStreamColumns.cpp 786 2006-05-05 13:48:02Z olau $ */

// Copyright (c) 2006 Caspar Gorvin, http://www.gorvin.de/
// Copyright (c) 2006 Heise Zeitschriften Verlag, http://www.heise.de/
// All rights reserved.

#include "stdafx.h"
#include "CoStreamColumns.h"
#include "ObjectCounter.h"
#include "ctShellExtensionGUID.h"	
#include "CFileNTFS.h"


CoStreamColumns::CoStreamColumns(void) 
:	m_nRefCount(0)
{
	IncrementObjectCounter();
}

CoStreamColumns::~CoStreamColumns(void)
{
	DecrementObjectCounter();
}

// *****************************************************************
// Implementierung des Interface IUnknown
// *****************************************************************

STDMETHODIMP CoStreamColumns::QueryInterface(
	REFIID riid, void** ppvObject)
{
	if (riid == IID_IUnknown)
		*ppvObject = (IUnknown*)this;
	else if (riid == IID_IColumnProvider)
		*ppvObject = (IColumnProvider*)this;
	else 
	{
		*ppvObject = NULL;
		return E_NOINTERFACE;
	}
	AddRef();
	return S_OK;
}

STDMETHODIMP_(ULONG) CoStreamColumns::AddRef()
{
	return ++m_nRefCount;
}

STDMETHODIMP_(ULONG) CoStreamColumns::Release()
{
	if (--m_nRefCount)
		return m_nRefCount;
	delete this;
	return 0;
}


//******************************************************************
// Implementierung des Interface IColumnProvider
//
// Das Interface IColumnProvider implementiert drei Funktionen,
// die nacheinander vom Windows Explorer aufgerufen werden:
// 1) IColumnProvider::Initialize
//		wird fr jedes Verzeichnis, das geffnet wird, einmal auf-
//		gerufen und dient der Initialisierung des Column-Handlers.
// 2) IColumnProvider::GetColumnInfo
//		wird nach Initialize solange aufgerufen, bis die be-
//		schreibenden Angaben zu allen Spalten, die angezeigt werden
//		sollen, abgefragt sind.
// 3) IColumnProvider::GetItemData
//		bergibt dem Windows Explorer die anzuzeigenden Daten,
//		wird fr jede Datei und jeden Ordner aufgerufen, die
//		im rechten Explorerfenster angezeigt werden sollen, und
//		zwar pro anzuzeigende Spalte einmal.

STDMETHODIMP CoStreamColumns::Initialize(LPCSHCOLUMNINIT psci)
{
	/* Parameterbeschreibung:
	[in] psci - LPSHCOLUMNNINIT Struktur mit folgenden Elementen:

	typedef struct {
	ULONG dwFlags;						// immer NULL
	ULONG dwReserved;					// immer NULL
	WCHAR wszFolder[MAX_PATH];			// Name des Verzeichnisses
	} SHCOLUMNNINIT, *LPSHCOLUMNNINIT;	// das geffnet wurde
	*/

	// Spaltenzhler und Dateinamen initialisieren
	m_dwColumnCount = 0;
	m_wszFile[0] = (WCHAR) 0; // Name der zuletzt bearbeiteten Datei (fr Optimierungszwecke)
	return S_OK;
}

STDMETHODIMP CoStreamColumns::GetColumnInfo(
	DWORD dwIndex,        // [in] Explorer-Spaltenindex
	SHCOLUMNINFO* psci)   // [out] Spalteninformationen
{
	//
	//  Liefert Informationen zu jeder Spalte, die dieser Column-
	//  Handler untersttzt. Die Funktion wird vom Windows Explorer
	//  solange wiederholt aufgerufen, bis sie durch die Rckgabe 
	//  von S_FALSE anzeigt, da alle Spalten abgefragt wurden oder
	//  bis ein Fehler-Code zurckgegeben wird.
	//  

	/* Parameterbeschreibung:
	[in] dwIndex - der interne Spaltenindex des Explorers.
	[out] psci - Zeiger auf eine SHCOLUMNINFO Struktur,
	die die Spalteninformationen erhlt 
	typedef struct {
	SHCOLUMNID scid;					// Spalten-ID
	VARTYPE vt;				  			// Datentyp der Spalte
	DWORD fmt;					  		// Spaltenformat
	UINT cChars;					  	// Spaltenbreite
	DWORD csFlags;						// Spaltenflags
	WCHAR wszTitle[MAX_COLUMN_NAME_LEN];// Spaltenberschrift
	WCHAR wszDescription[MAX_COLUMN_DESC_LEN];	// Beschreibung
	} SHCOLUMNINFO, *LPSHCOLUMNINFO;
	*/

	// Spalteninformationen zusammenstellen
	psci->scid.fmtid = CLSID_CoStreamColumns; // ID der Klasse ist fr alle Spalten identisch

	switch (m_dwColumnCount) 
	{	
		case PID_NUMBER_OF_STREAMS:
			StringCchCopy(psci->wszTitle, MAX_COLUMN_NAME_LEN, L"Streams");
			StringCchCopy(psci->wszDescription, MAX_COLUMN_DESC_LEN, L"Anzahl der Streams ohne Default Data Stream");
			psci->scid.pid = PID_NUMBER_OF_STREAMS;
  			psci->csFlags = SHCOLSTATE_TYPE_INT;// Spalte enthlt Zahlen
			psci->vt = VT_UINT;					// Datentyp des Rckgabewerts
			psci->cChars = 10;					// Spaltenbreite
			psci->fmt = LVCFMT_RIGHT;			// Rechtsbndige Textausrichtung
			break;

		case PID_STREAM_SIZE:
			StringCchCopy(psci->wszTitle, MAX_COLUMN_NAME_LEN, L"Streamgre");
			StringCchCopy(psci->wszDescription, MAX_COLUMN_DESC_LEN, L"Gesamtgre der Streams ohne Default Data Stream");
			psci->scid.pid = PID_STREAM_SIZE;
  			psci->csFlags = SHCOLSTATE_TYPE_STR;// Spalte enthlt Text
			psci->vt = VT_BSTR;					// Datentyp des Rckgabewerts
			psci->cChars = 14;					// Spaltenbreite
			psci->fmt = LVCFMT_RIGHT;			// Rechtsbndige Textausrichtung
			break;

		default:
			return S_FALSE; // keine weiteren Spalten zu bearbeiten
	}	

	m_dwColumnCount++;
	return S_OK;
}

STDMETHODIMP CoStreamColumns::GetItemData(
	LPCSHCOLUMNID pscid,			// [in] identifiziert die Spalte
	LPCSHCOLUMNDATA pscd,			// [in] liefert den Dateinamen
	VARIANT* pvarData)				// [out] erhlt die Spaltendaten
{
	// Der Windows Explorer fordert Daten fr eine unserer
	// Spalten an. Prfe die Eingabeparameter und gib die
	// angeforderten Daten zurck.

	/* Parameterbeschreibung:
	[in] pscid - SHCOLUMNID Struktur identifiziert die Spalte
	[in] pscd - SHCOLUMNDATA Struktur mit dem Dateinamen
	[out] pvarData - Zeiger auf eine VARIANT-Struktur, 
	die unsere Daten erhlt

	typedef struct {
	ULONG dwFlags;				// Datei-gendert-Flag
	DWORD dwFileAttributes;		// Dateiattribute
	ULONG dwReserved;
	WCHAR* pwszExt;				// Dateierweiterung
	WCHAR wszFile[MAX_PATH];	// vollstndiger Dateipfad
	} SHCOLUMNDATA, *LPSHCOLUMNDATA;
	*/

	// Prfen, ob die Abfrage einer unserer Spalten gilt
	if (pscid->fmtid != CLSID_CoStreamColumns
		|| pscid->pid >= PID_MAX)
			return S_FALSE;							

	// Prfen, ob die Datei auf ein Offline-Speichermedium, z. B.
	// Datensicherungsband, ausgelagert ist, dann nicht ffnen
	// (wrde den Benutzer zwingen, das Band einzulegen)
	if (pscd->dwFileAttributes & FILE_ATTRIBUTE_OFFLINE)
		return S_FALSE;

	// Die Lnge des Dateipfades prfen
	size_t stPathLength = 0;
	HRESULT hr;
	hr = StringCchLength(pscd->wszFile, MAX_PATH, &stPathLength);
	if (FAILED(hr) || !stPathLength)
		return E_INVALIDARG;

#ifdef _DEBUG
	//	Test der Implementierung des IColumnProvider-Interface:
	//	Die Message-Box gibt den Namen der Datei aus, fr die der
	//	Windows Explorer Informationen angefordert hat.
	const unsigned int MAX_TEXT = 1000;
	WCHAR wszMsg [MAX_TEXT];
	StringCchPrintf(wszMsg, MAX_TEXT, 
		L"Aufruf der Funktion: %s\n\nDateiname: \"%s\"",
		_T(__FUNCTION__), pscd->wszFile);
	MessageBox(NULL, wszMsg, L"Projekt ctShellExtension", 
		MB_OK | MB_ICONINFORMATION);
#endif

	// Die angeforderten Stream-Informationen zusammenstellen
	if (FAILED(GetStreamData(pscd->wszFile)))
		return E_FAIL;

	// Den Ausgabeparameter mit den angeforderten Daten fllen
	if (!m_dwStreamCount)	// mittels S_FALSE bleibt die Spalte
		return S_FALSE;		// leer, statt sie mit Nullen zu fllen (sieht ansprechender aus)

	switch (pscid->pid) 
	{
		case PID_NUMBER_OF_STREAMS:	// Spalte "Streams"
			pvarData->vt = VT_UINT;	// Rckgabedatentyp ist UINT
			pvarData->uintVal = m_dwStreamCount; 
			break;

		case PID_STREAM_SIZE:		// Spalte "Streamgre"
			pvarData->vt = VT_BSTR;	// Datentyp des Rckgabeparm ist COM-spezifischer String

			// Ganzzahlenwert in String umwandeln
			if (FAILED(StringCchPrintf(m_wszDigits, 
					sizeof m_wszDigits / sizeof(WCHAR), L"%I64u", 
					m_lnStreamSize.QuadPart)))
				return S_FALSE;

			// Zahlenformatdarstellung festlegen
			m_stFormat.NumDigits = 0;		// keine Dezimalstellen	
			m_stFormat.LeadingZero = 0;		// keine fhrenden Nullen
			m_stFormat.Grouping = 3;		// Tausendergruppierung
			m_stFormat.lpDecimalSep = L",";	// Komma
			m_stFormat.lpThousandSep = L".";// Tausendertrennzeichen
			m_stFormat.NegativeOrder = 1;	// Darstellung als "-1,1"

			// Zahlenstring (Streamgre) formatieren
			if (!GetNumberFormat(
				LOCALE_USER_DEFAULT,	// Fr Defaultformate Benutzer-
				0,						// einstellungen verwenden
				m_wszDigits,			// unformatierter Eingabestring
				&m_stFormat,			// gewnschte Formatangaben
				m_wszFormatedDigits,	// formatierter Ausgabestring
				sizeof(m_wszFormatedDigits) / sizeof(WCHAR))) // Lnge
					return S_FALSE;

			// In einen COM-spezifischen String (Typ BSTR) umwandeln
			m_bstrVal = SysAllocString(m_wszFormatedDigits);
			if (m_bstrVal)
				pvarData->bstrVal = m_bstrVal; 
			else
				return S_FALSE;
			break;

		default:
			return S_FALSE;
	}
	return S_OK;
}


// Zhlt die Anzahl der Streams in einer Datei und berechnet die 
// Gesamtgre der Streams. Der Default Data Stream wird nicht mitgezhlt.
// Diese Methode gehrt nicht zur Implementierung des Interface IColumnProvider.
HRESULT CoStreamColumns::GetStreamData(LPCWSTR wszFile)
{
	CFileNTFS cFile;
	CFileNTFS::FILE_STREAM_ID wStreamID;

	// Falls die gleiche Datei schon gerade bearbeitet wurde,
	// sind die Stream-Daten bereits vorhanden und knnen
	// ohne erneutes Lesen der Datei geliefert werden
	// (verhindert doppeltes Lesen jeder Datei)
	if (!lstrcmpi(m_wszFile, wszFile))
		return S_OK;

	// Stream-Informationen zurcksetzen
	m_dwStreamCount = 0;
	m_lnStreamSize.QuadPart = 0;
	m_wszFile[0] = 0;

	// Datei fr das Lesen von Streams ffnen
	if (FAILED(cFile.CreateFile(
		wszFile,					// Dateiname
		READ_CONTROL, 				// Lesen von Kontrollinfo
		FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,		
		OPEN_EXISTING,					// bestehende Datei ffnen
		FILE_FLAG_SEQUENTIAL_SCAN |		// schnelles Durchsuchen
		FILE_FLAG_BACKUP_SEMANTICS |	// Streams lesen
		FILE_FLAG_OPEN_REPARSE_POINT)))	// Reparse Points lesen
			return S_FALSE;

	// Alle Stream-ID-Strukturen der Datei lesen und die
	// Anzahl und Gre der Streams zhlen
	// (der Default Data Stream wird nicht mitgezhlt)
	HRESULT hr;
	while (S_OK == (hr = cFile.ReadNextStreamInfo(&wStreamID)))
		switch (wStreamID.dwStreamId)
		{
		case BACKUP_DATA:			// 1: Standard (Default) Data 
			break;					// nicht mitzhlen
		case BACKUP_EA_DATA:		// 2: Extended Attribute Data
		case BACKUP_SECURITY_DATA:	// 3: Security Descriptor Data		
		case BACKUP_ALTERNATE_DATA:	// 4: Alternate Data Streams
		case BACKUP_LINK:			// 5: Hard Link Information 
		case BACKUP_PROPERTY_DATA:	// 6: Property Data
		case BACKUP_OBJECT_ID:		// 7: Object Identifiers
		case BACKUP_REPARSE_DATA:	// 8: Reparse Points
		case BACKUP_SPARSE_BLOCK:	// 9: Sparse File
		default:					// Unknown Stream ID
				m_dwStreamCount++;	
				m_lnStreamSize.QuadPart += wStreamID.Size.QuadPart;
			break;
		}

	// Datei schlieen 
	cFile.CloseFile();

	// Prfen, ob beim Lesen Fehler aufgetreten sind
	// (Fehler sind in aller Regel kein I/O-Fehler, 
	// sondern fehlende Zugriffsrechte.
	// Die Spalte bleibt dann einfach leer.)
	if (hr == E_FAIL)
		return S_FALSE;

	// Dateinamen fr die Optimierung merken
	if (FAILED(StringCchCopy(m_wszFile, MAX_PATH, wszFile)))
		m_wszFile[0] = 0;

	return S_OK;
}
