/*
 * Copyright 1998-2004 VIA Technologies, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * VIA, AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "padlock.h"
#include "via_ace.h"

#include <stdio.h>
#include <memory.h>

// VIA ACE hardware control words constant
static int NEH_GENKEY_ENCRYPT_128 = NEH_ENCRYPT + NEH_GEN_KEY + NEH_KEY128;
static int NEH_LDKEY_ENCRYPT_192 = NEH_ENCRYPT + NEH_LOAD_KEY + NEH_KEY192;
static int NEH_LDKEY_ENCRYPT_256 = NEH_ENCRYPT + NEH_LOAD_KEY + NEH_KEY256;

static int NEH_GENKEY_DECRYPT_128 = NEH_DECRYPT + NEH_GEN_KEY + NEH_KEY128;
static int NEH_LDKEY_DECRYPT_192 = NEH_DECRYPT + NEH_LOAD_KEY + NEH_KEY192;
static int NEH_LDKEY_DECRYPT_256 = NEH_DECRYPT + NEH_LOAD_KEY + NEH_KEY256;

// tables for key manipulation
static unsigned char pow_tab[256]= {
 0x1,   0x3,   0x5,   0xf,   0x11,  0x33,  0x55,  0xff, 
 0x1a,  0x2e,  0x72,  0x96,  0xa1,  0xf8,  0x13,  0x35, 
 0x5f,  0xe1,  0x38,  0x48,  0xd8,  0x73,  0x95,  0xa4, 
 0xf7,  0x2,   0x6,   0xa,   0x1e,  0x22,  0x66,  0xaa, 
 0xe5,  0x34,  0x5c,  0xe4,  0x37,  0x59,  0xeb,  0x26, 
 0x6a,  0xbe,  0xd9,  0x70,  0x90,  0xab,  0xe6,  0x31, 
 0x53,  0xf5,  0x4,   0xc,   0x14,  0x3c,  0x44,  0xcc, 
 0x4f,  0xd1,  0x68,  0xb8,  0xd3,  0x6e,  0xb2,  0xcd, 
 0x4c,  0xd4,  0x67,  0xa9,  0xe0,  0x3b,  0x4d,  0xd7, 
 0x62,  0xa6,  0xf1,  0x8,   0x18,  0x28,  0x78,  0x88, 
 0x83,  0x9e,  0xb9,  0xd0,  0x6b,  0xbd,  0xdc,  0x7f, 
 0x81,  0x98,  0xb3,  0xce,  0x49,  0xdb,  0x76,  0x9a, 
 0xb5,  0xc4,  0x57,  0xf9,  0x10,  0x30,  0x50,  0xf0, 
 0xb,   0x1d,  0x27,  0x69,  0xbb,  0xd6,  0x61,  0xa3, 
 0xfe,  0x19,  0x2b,  0x7d,  0x87,  0x92,  0xad,  0xec, 
 0x2f,  0x71,  0x93,  0xae,  0xe9,  0x20,  0x60,  0xa0, 
 0xfb,  0x16,  0x3a,  0x4e,  0xd2,  0x6d,  0xb7,  0xc2, 
 0x5d,  0xe7,  0x32,  0x56,  0xfa,  0x15,  0x3f,  0x41, 
 0xc3,  0x5e,  0xe2,  0x3d,  0x47,  0xc9,  0x40,  0xc0, 
 0x5b,  0xed,  0x2c,  0x74,  0x9c,  0xbf,  0xda,  0x75, 
 0x9f,  0xba,  0xd5,  0x64,  0xac,  0xef,  0x2a,  0x7e, 
 0x82,  0x9d,  0xbc,  0xdf,  0x7a,  0x8e,  0x89,  0x80, 
 0x9b,  0xb6,  0xc1,  0x58,  0xe8,  0x23,  0x65,  0xaf, 
 0xea,  0x25,  0x6f,  0xb1,  0xc8,  0x43,  0xc5,  0x54, 
 0xfc,  0x1f,  0x21,  0x63,  0xa5,  0xf4,  0x7,   0x9, 
 0x1b,  0x2d,  0x77,  0x99,  0xb0,  0xcb,  0x46,  0xca, 
 0x45,  0xcf,  0x4a,  0xde,  0x79,  0x8b,  0x86,  0x91, 
 0xa8,  0xe3,  0x3e,  0x42,  0xc6,  0x51,  0xf3,  0xe, 
 0x12,  0x36,  0x5a,  0xee,  0x29,  0x7b,  0x8d,  0x8c, 
 0x8f,  0x8a,  0x85,  0x94,  0xa7,  0xf2,  0xd,   0x17, 
 0x39,  0x4b,  0xdd,  0x7c,  0x84,  0x97,  0xa2,  0xfd, 
 0x1c,  0x24,  0x6c,  0xb4,  0xc7,  0x52,  0xf6,  0x1 };

static unsigned char log_tab[256] = {
 0x0,   0x0,   0x19,  0x1,   0x32,  0x2,   0x1a,  0xc6, 
 0x4b,  0xc7,  0x1b,  0x68,  0x33,  0xee,  0xdf,  0x3, 
 0x64,  0x4,   0xe0,  0xe,   0x34,  0x8d,  0x81,  0xef, 
 0x4c,  0x71,  0x8,   0xc8,  0xf8,  0x69,  0x1c,  0xc1, 
 0x7d,  0xc2,  0x1d,  0xb5,  0xf9,  0xb9,  0x27,  0x6a, 
 0x4d,  0xe4,  0xa6,  0x72,  0x9a,  0xc9,  0x9,   0x78, 
 0x65,  0x2f,  0x8a,  0x5,   0x21,  0xf,   0xe1,  0x24, 
 0x12,  0xf0,  0x82,  0x45,  0x35,  0x93,  0xda,  0x8e, 
 0x96,  0x8f,  0xdb,  0xbd,  0x36,  0xd0,  0xce,  0x94, 
 0x13,  0x5c,  0xd2,  0xf1,  0x40,  0x46,  0x83,  0x38, 
 0x66,  0xdd,  0xfd,  0x30,  0xbf,  0x6,   0x8b,  0x62, 
 0xb3,  0x25,  0xe2,  0x98,  0x22,  0x88,  0x91,  0x10, 
 0x7e,  0x6e,  0x48,  0xc3,  0xa3,  0xb6,  0x1e,  0x42, 
 0x3a,  0x6b,  0x28,  0x54,  0xfa,  0x85,  0x3d,  0xba, 
 0x2b,  0x79,  0xa,   0x15,  0x9b,  0x9f,  0x5e,  0xca, 
 0x4e,  0xd4,  0xac,  0xe5,  0xf3,  0x73,  0xa7,  0x57, 
 0xaf,  0x58,  0xa8,  0x50,  0xf4,  0xea,  0xd6,  0x74, 
 0x4f,  0xae,  0xe9,  0xd5,  0xe7,  0xe6,  0xad,  0xe8, 
 0x2c,  0xd7,  0x75,  0x7a,  0xeb,  0x16,  0xb,   0xf5, 
 0x59,  0xcb,  0x5f,  0xb0,  0x9c,  0xa9,  0x51,  0xa0, 
 0x7f,  0xc,   0xf6,  0x6f,  0x17,  0xc4,  0x49,  0xec, 
 0xd8,  0x43,  0x1f,  0x2d,  0xa4,  0x76,  0x7b,  0xb7, 
 0xcc,  0xbb,  0x3e,  0x5a,  0xfb,  0x60,  0xb1,  0x86, 
 0x3b,  0x52,  0xa1,  0x6c,  0xaa,  0x55,  0x29,  0x9d, 
 0x97,  0xb2,  0x87,  0x90,  0x61,  0xbe,  0xdc,  0xfc, 
 0xbc,  0x95,  0xcf,  0xcd,  0x37,  0x3f,  0x5b,  0xd1, 
 0x53,  0x39,  0x84,  0x3c,  0x41,  0xa2,  0x6d,  0x47, 
 0x14,  0x2a,  0x9e,  0x5d,  0x56,  0xf2,  0xd3,  0xab, 
 0x44,  0x11,  0x92,  0xd9,  0x23,  0x20,  0x2e,  0x89, 
 0xb4,  0x7c,  0xb8,  0x26,  0x77,  0x99,  0xe3,  0xa5, 
 0x67,  0x4a,  0xed,  0xde,  0xc5,  0x31,  0xfe,  0x18, 
 0xd,   0x63,  0x8c,  0x80,  0xc0,  0xf7,  0x70,  0x7 };

static unsigned int rco_tab[10] = { 0x1,  0x2,  0x4,  0x8,  0x10,  0x20,  0x40,  0x80, 0x1b,  0x36 };

static unsigned int fl_tab[4][256]={
{0x63,  0x7c,  0x77,  0x7b,  0xf2,  0x6b,  0x6f,  0xc5, 
 0x30,  0x1,   0x67,  0x2b,  0xfe,  0xd7,  0xab,  0x76, 
 0xca,  0x82,  0xc9,  0x7d,  0xfa,  0x59,  0x47,  0xf0, 
 0xad,  0xd4,  0xa2,  0xaf,  0x9c,  0xa4,  0x72,  0xc0, 
 0xb7,  0xfd,  0x93,  0x26,  0x36,  0x3f,  0xf7,  0xcc, 
 0x34,  0xa5,  0xe5,  0xf1,  0x71,  0xd8,  0x31,  0x15, 
 0x4,   0xc7,  0x23,  0xc3,  0x18,  0x96,  0x5,   0x9a, 
 0x7,   0x12,  0x80,  0xe2,  0xeb,  0x27,  0xb2,  0x75, 
 0x9,   0x83,  0x2c,  0x1a,  0x1b,  0x6e,  0x5a,  0xa0, 
 0x52,  0x3b,  0xd6,  0xb3,  0x29,  0xe3,  0x2f,  0x84, 
 0x53,  0xd1,  0x0,   0xed,  0x20,  0xfc,  0xb1,  0x5b, 
 0x6a,  0xcb,  0xbe,  0x39,  0x4a,  0x4c,  0x58,  0xcf, 
 0xd0,  0xef,  0xaa,  0xfb,  0x43,  0x4d,  0x33,  0x85, 
 0x45,  0xf9,  0x2,   0x7f,  0x50,  0x3c,  0x9f,  0xa8, 
 0x51,  0xa3,  0x40,  0x8f,  0x92,  0x9d,  0x38,  0xf5, 
 0xbc,  0xb6,  0xda,  0x21,  0x10,  0xff,  0xf3,  0xd2, 
 0xcd,  0xc,   0x13,  0xec,  0x5f,  0x97,  0x44,  0x17, 
 0xc4,  0xa7,  0x7e,  0x3d,  0x64,  0x5d,  0x19,  0x73, 
 0x60,  0x81,  0x4f,  0xdc,  0x22,  0x2a,  0x90,  0x88, 
 0x46,  0xee,  0xb8,  0x14,  0xde,  0x5e,  0xb,   0xdb, 
 0xe0,  0x32,  0x3a,  0xa,   0x49,  0x6,   0x24,  0x5c, 
 0xc2,  0xd3,  0xac,  0x62,  0x91,  0x95,  0xe4,  0x79, 
 0xe7,  0xc8,  0x37,  0x6d,  0x8d,  0xd5,  0x4e,  0xa9, 
 0x6c,  0x56,  0xf4,  0xea,  0x65,  0x7a,  0xae,  0x8, 
 0xba,  0x78,  0x25,  0x2e,  0x1c,  0xa6,  0xb4,  0xc6, 
 0xe8,  0xdd,  0x74,  0x1f,  0x4b,  0xbd,  0x8b,  0x8a, 
 0x70,  0x3e,  0xb5,  0x66,  0x48,  0x3,   0xf6,  0xe, 
 0x61,  0x35,  0x57,  0xb9,  0x86,  0xc1,  0x1d,  0x9e, 
 0xe1,  0xf8,  0x98,  0x11,  0x69,  0xd9,  0x8e,  0x94, 
 0x9b,  0x1e,  0x87,  0xe9,  0xce,  0x55,  0x28,  0xdf, 
 0x8c,  0xa1,  0x89,  0xd,   0xbf,  0xe6,  0x42,  0x68, 
 0x41,  0x99,  0x2d,  0xf,   0xb0,  0x54,  0xbb,  0x16 },

{0x6300,  0x7c00,  0x7700,  0x7b00,  0xf200,  0x6b00,  0x6f00,  0xc500, 
 0x3000,  0x100,   0x6700,  0x2b00,  0xfe00,  0xd700,  0xab00,  0x7600, 
 0xca00,  0x8200,  0xc900,  0x7d00,  0xfa00,  0x5900,  0x4700,  0xf000, 
 0xad00,  0xd400,  0xa200,  0xaf00,  0x9c00,  0xa400,  0x7200,  0xc000, 
 0xb700,  0xfd00,  0x9300,  0x2600,  0x3600,  0x3f00,  0xf700,  0xcc00, 
 0x3400,  0xa500,  0xe500,  0xf100,  0x7100,  0xd800,  0x3100,  0x1500, 
 0x400,   0xc700,  0x2300,  0xc300,  0x1800,  0x9600,  0x500,   0x9a00, 
 0x700,   0x1200,  0x8000,  0xe200,  0xeb00,  0x2700,  0xb200,  0x7500, 
 0x900,   0x8300,  0x2c00,  0x1a00,  0x1b00,  0x6e00,  0x5a00,  0xa000, 
 0x5200,  0x3b00,  0xd600,  0xb300,  0x2900,  0xe300,  0x2f00,  0x8400, 
 0x5300,  0xd100,  0x0,     0xed00,  0x2000,  0xfc00,  0xb100,  0x5b00, 
 0x6a00,  0xcb00,  0xbe00,  0x3900,  0x4a00,  0x4c00,  0x5800,  0xcf00, 
 0xd000,  0xef00,  0xaa00,  0xfb00,  0x4300,  0x4d00,  0x3300,  0x8500, 
 0x4500,  0xf900,  0x200,   0x7f00,  0x5000,  0x3c00,  0x9f00,  0xa800, 
 0x5100,  0xa300,  0x4000,  0x8f00,  0x9200,  0x9d00,  0x3800,  0xf500, 
 0xbc00,  0xb600,  0xda00,  0x2100,  0x1000,  0xff00,  0xf300,  0xd200, 
 0xcd00,  0xc00,   0x1300,  0xec00,  0x5f00,  0x9700,  0x4400,  0x1700, 
 0xc400,  0xa700,  0x7e00,  0x3d00,  0x6400,  0x5d00,  0x1900,  0x7300, 
 0x6000,  0x8100,  0x4f00,  0xdc00,  0x2200,  0x2a00,  0x9000,  0x8800, 
 0x4600,  0xee00,  0xb800,  0x1400,  0xde00,  0x5e00,  0xb00,   0xdb00, 
 0xe000,  0x3200,  0x3a00,  0xa00,   0x4900,  0x600,   0x2400,  0x5c00, 
 0xc200,  0xd300,  0xac00,  0x6200,  0x9100,  0x9500,  0xe400,  0x7900, 
 0xe700,  0xc800,  0x3700,  0x6d00,  0x8d00,  0xd500,  0x4e00,  0xa900, 
 0x6c00,  0x5600,  0xf400,  0xea00,  0x6500,  0x7a00,  0xae00,  0x800, 
 0xba00,  0x7800,  0x2500,  0x2e00,  0x1c00,  0xa600,  0xb400,  0xc600, 
 0xe800,  0xdd00,  0x7400,  0x1f00,  0x4b00,  0xbd00,  0x8b00,  0x8a00, 
 0x7000,  0x3e00,  0xb500,  0x6600,  0x4800,  0x300,   0xf600,  0xe00, 
 0x6100,  0x3500,  0x5700,  0xb900,  0x8600,  0xc100,  0x1d00,  0x9e00, 
 0xe100,  0xf800,  0x9800,  0x1100,  0x6900,  0xd900,  0x8e00,  0x9400, 
 0x9b00,  0x1e00,  0x8700,  0xe900,  0xce00,  0x5500,  0x2800,  0xdf00, 
 0x8c00,  0xa100,  0x8900,  0xd00,   0xbf00,  0xe600,  0x4200,  0x6800, 
 0x4100,  0x9900,  0x2d00,  0xf00,   0xb000,  0x5400,  0xbb00,  0x1600 },

{0x630000,  0x7c0000,  0x770000,  0x7b0000,  0xf20000,  0x6b0000,  0x6f0000,  0xc50000, 
 0x300000,  0x10000,   0x670000,  0x2b0000,  0xfe0000,  0xd70000,  0xab0000,  0x760000, 
 0xca0000,  0x820000,  0xc90000,  0x7d0000,  0xfa0000,  0x590000,  0x470000,  0xf00000, 
 0xad0000,  0xd40000,  0xa20000,  0xaf0000,  0x9c0000,  0xa40000,  0x720000,  0xc00000, 
 0xb70000,  0xfd0000,  0x930000,  0x260000,  0x360000,  0x3f0000,  0xf70000,  0xcc0000, 
 0x340000,  0xa50000,  0xe50000,  0xf10000,  0x710000,  0xd80000,  0x310000,  0x150000, 
 0x40000,   0xc70000,  0x230000,  0xc30000,  0x180000,  0x960000,  0x50000,   0x9a0000, 
 0x70000,   0x120000,  0x800000,  0xe20000,  0xeb0000,  0x270000,  0xb20000,  0x750000, 
 0x90000,   0x830000,  0x2c0000,  0x1a0000,  0x1b0000,  0x6e0000,  0x5a0000,  0xa00000, 
 0x520000,  0x3b0000,  0xd60000,  0xb30000,  0x290000,  0xe30000,  0x2f0000,  0x840000, 
 0x530000,  0xd10000,  0x0,       0xed0000,  0x200000,  0xfc0000,  0xb10000,  0x5b0000, 
 0x6a0000,  0xcb0000,  0xbe0000,  0x390000,  0x4a0000,  0x4c0000,  0x580000,  0xcf0000, 
 0xd00000,  0xef0000,  0xaa0000,  0xfb0000,  0x430000,  0x4d0000,  0x330000,  0x850000, 
 0x450000,  0xf90000,  0x20000,   0x7f0000,  0x500000,  0x3c0000,  0x9f0000,  0xa80000, 
 0x510000,  0xa30000,  0x400000,  0x8f0000,  0x920000,  0x9d0000,  0x380000,  0xf50000, 
 0xbc0000,  0xb60000,  0xda0000,  0x210000,  0x100000,  0xff0000,  0xf30000,  0xd20000, 
 0xcd0000,  0xc0000,   0x130000,  0xec0000,  0x5f0000,  0x970000,  0x440000,  0x170000, 
 0xc40000,  0xa70000,  0x7e0000,  0x3d0000,  0x640000,  0x5d0000,  0x190000,  0x730000, 
 0x600000,  0x810000,  0x4f0000,  0xdc0000,  0x220000,  0x2a0000,  0x900000,  0x880000, 
 0x460000,  0xee0000,  0xb80000,  0x140000,  0xde0000,  0x5e0000,  0xb0000,   0xdb0000, 
 0xe00000,  0x320000,  0x3a0000,  0xa0000,   0x490000,  0x60000,   0x240000,  0x5c0000, 
 0xc20000,  0xd30000,  0xac0000,  0x620000,  0x910000,  0x950000,  0xe40000,  0x790000, 
 0xe70000,  0xc80000,  0x370000,  0x6d0000,  0x8d0000,  0xd50000,  0x4e0000,  0xa90000, 
 0x6c0000,  0x560000,  0xf40000,  0xea0000,  0x650000,  0x7a0000,  0xae0000,  0x80000, 
 0xba0000,  0x780000,  0x250000,  0x2e0000,  0x1c0000,  0xa60000,  0xb40000,  0xc60000, 
 0xe80000,  0xdd0000,  0x740000,  0x1f0000,  0x4b0000,  0xbd0000,  0x8b0000,  0x8a0000, 
 0x700000,  0x3e0000,  0xb50000,  0x660000,  0x480000,  0x30000,   0xf60000,  0xe0000, 
 0x610000,  0x350000,  0x570000,  0xb90000,  0x860000,  0xc10000,  0x1d0000,  0x9e0000, 
 0xe10000,  0xf80000,  0x980000,  0x110000,  0x690000,  0xd90000,  0x8e0000,  0x940000, 
 0x9b0000,  0x1e0000,  0x870000,  0xe90000,  0xce0000,  0x550000,  0x280000,  0xdf0000, 
 0x8c0000,  0xa10000,  0x890000,  0xd0000,   0xbf0000,  0xe60000,  0x420000,  0x680000, 
 0x410000,  0x990000,  0x2d0000,  0xf0000,   0xb00000,  0x540000,  0xbb0000,  0x160000 },

{0x63000000,  0x7c000000,  0x77000000,  0x7b000000,  0xf2000000,  0x6b000000,  0x6f000000,  0xc5000000, 
 0x30000000,  0x1000000,   0x67000000,  0x2b000000,  0xfe000000,  0xd7000000,  0xab000000,  0x76000000, 
 0xca000000,  0x82000000,  0xc9000000,  0x7d000000,  0xfa000000,  0x59000000,  0x47000000,  0xf0000000, 
 0xad000000,  0xd4000000,  0xa2000000,  0xaf000000,  0x9c000000,  0xa4000000,  0x72000000,  0xc0000000, 
 0xb7000000,  0xfd000000,  0x93000000,  0x26000000,  0x36000000,  0x3f000000,  0xf7000000,  0xcc000000, 
 0x34000000,  0xa5000000,  0xe5000000,  0xf1000000,  0x71000000,  0xd8000000,  0x31000000,  0x15000000, 
 0x4000000,   0xc7000000,  0x23000000,  0xc3000000,  0x18000000,  0x96000000,  0x5000000,   0x9a000000, 
 0x7000000,   0x12000000,  0x80000000,  0xe2000000,  0xeb000000,  0x27000000,  0xb2000000,  0x75000000, 
 0x9000000,   0x83000000,  0x2c000000,  0x1a000000,  0x1b000000,  0x6e000000,  0x5a000000,  0xa0000000, 
 0x52000000,  0x3b000000,  0xd6000000,  0xb3000000,  0x29000000,  0xe3000000,  0x2f000000,  0x84000000, 
 0x53000000,  0xd1000000,  0x0,         0xed000000,  0x20000000,  0xfc000000,  0xb1000000,  0x5b000000, 
 0x6a000000,  0xcb000000,  0xbe000000,  0x39000000,  0x4a000000,  0x4c000000,  0x58000000,  0xcf000000, 
 0xd0000000,  0xef000000,  0xaa000000,  0xfb000000,  0x43000000,  0x4d000000,  0x33000000,  0x85000000, 
 0x45000000,  0xf9000000,  0x2000000,   0x7f000000,  0x50000000,  0x3c000000,  0x9f000000,  0xa8000000, 
 0x51000000,  0xa3000000,  0x40000000,  0x8f000000,  0x92000000,  0x9d000000,  0x38000000,  0xf5000000, 
 0xbc000000,  0xb6000000,  0xda000000,  0x21000000,  0x10000000,  0xff000000,  0xf3000000,  0xd2000000, 
 0xcd000000,  0xc000000,   0x13000000,  0xec000000,  0x5f000000,  0x97000000,  0x44000000,  0x17000000, 
 0xc4000000,  0xa7000000,  0x7e000000,  0x3d000000,  0x64000000,  0x5d000000,  0x19000000,  0x73000000, 
 0x60000000,  0x81000000,  0x4f000000,  0xdc000000,  0x22000000,  0x2a000000,  0x90000000,  0x88000000, 
 0x46000000,  0xee000000,  0xb8000000,  0x14000000,  0xde000000,  0x5e000000,  0xb000000,   0xdb000000, 
 0xe0000000,  0x32000000,  0x3a000000,  0xa000000,   0x49000000,  0x6000000,   0x24000000,  0x5c000000, 
 0xc2000000,  0xd3000000,  0xac000000,  0x62000000,  0x91000000,  0x95000000,  0xe4000000,  0x79000000, 
 0xe7000000,  0xc8000000,  0x37000000,  0x6d000000,  0x8d000000,  0xd5000000,  0x4e000000,  0xa9000000, 
 0x6c000000,  0x56000000,  0xf4000000,  0xea000000,  0x65000000,  0x7a000000,  0xae000000,  0x8000000, 
 0xba000000,  0x78000000,  0x25000000,  0x2e000000,  0x1c000000,  0xa6000000,  0xb4000000,  0xc6000000, 
 0xe8000000,  0xdd000000,  0x74000000,  0x1f000000,  0x4b000000,  0xbd000000,  0x8b000000,  0x8a000000, 
 0x70000000,  0x3e000000,  0xb5000000,  0x66000000,  0x48000000,  0x3000000,   0xf6000000,  0xe000000, 
 0x61000000,  0x35000000,  0x57000000,  0xb9000000,  0x86000000,  0xc1000000,  0x1d000000,  0x9e000000, 
 0xe1000000,  0xf8000000,  0x98000000,  0x11000000,  0x69000000,  0xd9000000,  0x8e000000,  0x94000000, 
 0x9b000000,  0x1e000000,  0x87000000,  0xe9000000,  0xce000000,  0x55000000,  0x28000000,  0xdf000000, 
 0x8c000000,  0xa1000000,  0x89000000,  0xd000000,   0xbf000000,  0xe6000000,  0x42000000,  0x68000000, 
 0x41000000,  0x99000000,  0x2d000000,  0xf000000,   0xb0000000,  0x54000000,  0xbb000000,  0x16000000 }
};

static inline 
unsigned int rotr(const unsigned int x, const unsigned bits)
{
	const unsigned n = bits % 32;
	return (x >> n) | (x << (32 - n));
}

static inline 
unsigned int rotl(const unsigned int x, const unsigned bits)
{
	const unsigned n = bits % 32;
	return (x << n) | (x >> (32 - n));
}

static inline unsigned char
via_byte(const unsigned int x, const unsigned n)
{
	return x >> (n << 3);
}

static inline unsigned char
f_mult (unsigned char m, unsigned char n)
{
	unsigned char x = log_tab[m];
	unsigned char y = x + log_tab[n];

	return pow_tab[y + (y < x ? 1 : 0)];
}

#define ff_mult(a,b)    (a && b ? f_mult(a, b) : 0)

#define ls_box(x)	( fl_tab[0][via_byte(x, 0)] ^ fl_tab[1][via_byte(x, 1)] ^ fl_tab[2][via_byte(x, 2)] ^ fl_tab[3][via_byte(x, 3)] )

#define star_x(x) (((x) & 0x7f7f7f7f) << 1) ^ ((((x) & 0x80808080) >> 7) * 0x1b)

#define imix_col(y,x)       \
    u   = star_x(x);        \
    v   = star_x(u);        \
    w   = star_x(v);        \
    t   = w ^ (x);          \
   (y)  = u ^ v ^ w;        \
   (y) ^= rotr(u ^ t,  8) ^ rotr(v ^ t, 16) ^ rotr(t,24)

#define E_KEY ctx->encrypt_extended_key
#define D_KEY ctx->decrypt_extended_key

///////////////////////////////////////////////////////////////////////////////////////////////////
// Padlock SDK API

AES_RESULT
padlock_aes_setkey(struct ace_aes_context *ctx, const unsigned char *key, KEY_LENGTH key_len)
{
    int i, t, u, v, w;
	int real_key_len,rounds;
	int m, n;
	unsigned int exch;
	AES_RESULT res = AES_SUCCEEDED;

	if(ctx == NULL)
	{
		printf("Fatal error : ace_aes_ctx NULL pointer error!\n");
		res = AES_FAILED;
		return res;
	}

	if(key == NULL)
	{
		printf("Fatal error : key NULL pointer error!\n");
		res = AES_FAILED;
		return res;
	}

	switch(key_len)
	{
		case KEY_128BITS:   real_key_len = 16;
							rounds = 10;
							break;
		case KEY_192BITS:	real_key_len = 24;
							rounds = 12;
							break;
		case KEY_256BITS:	real_key_len = 32;
							rounds = 14;
							break;
		default:			printf("Invalid Key Length!\n");
					        return AES_KEY_NOT_SUPPORTED;
	}

	ctx->key_length = key_len;
	
	E_KEY[0] = (unsigned int)(*(const unsigned int *)(key));
	E_KEY[1] = (unsigned int)(*(const unsigned int *)(key + 4));
	E_KEY[2] = (unsigned int)(*(const unsigned int *)(key + 8));
	E_KEY[3] = (unsigned int)(*(const unsigned int *)(key + 12));
	
	switch (key_len) 
	{
	case KEY_128BITS:
		t = E_KEY[3];
		for (i = 0; i < 10; ++i)
		{   
			t = rotr(t,  8); 
			t = ls_box(t) ^ rco_tab[i];
			t ^= E_KEY[4 * i];
			E_KEY[4 * i + 4] = t;
			t ^= E_KEY[4 * i + 1]; 
			E_KEY[4 * i + 5] = t;
			t ^= E_KEY[4 * i + 2]; 
			E_KEY[4 * i + 6] = t;
			t ^= E_KEY[4 * i + 3]; 
			E_KEY[4 * i + 7] = t;
        }
		break;

	case KEY_192BITS:
		E_KEY[4]		= (unsigned int)(*(const unsigned int *)(key + 16));
		t = E_KEY[5]	= (unsigned int)(*(const unsigned int *)(key + 20));
		for (i = 0; i < 8; ++i)
		{   
			t = rotr(t,  8); 
			t = ls_box(t) ^ rco_tab[i];
			t ^= E_KEY[6 * i];
			E_KEY[6 * i + 6] = t;
			t ^= E_KEY[6 * i + 1]; 
			E_KEY[6 * i + 7] = t;
			t ^= E_KEY[6 * i + 2];
			E_KEY[6 * i + 8] = t;
			t ^= E_KEY[6 * i + 3];
			E_KEY[6 * i + 9] = t;
			t ^= E_KEY[6 * i + 4]; 
			E_KEY[6 * i + 10] = t;
			t ^= E_KEY[6 * i + 5]; 
			E_KEY[6 * i + 11] = t;
		}
		break;

	case KEY_256BITS:
		E_KEY[4]		= (unsigned int)(*(const unsigned int *)(key + 16));
		E_KEY[5]		= (unsigned int)(*(const unsigned int *)(key + 20));
		E_KEY[6]		= (unsigned int)(*(const unsigned int *)(key + 24));
		t = E_KEY[7]	= (unsigned int)(*(const unsigned int *)(key + 28));
		for (i = 0; i < 7; ++i)
		{   
			t = rotr(t,  8); 
			t = ls_box(t) ^ rco_tab[i];
			t ^= E_KEY[8 * i];     
			E_KEY[8 * i + 8] = t;
		    t ^= E_KEY[8 * i + 1]; 
			E_KEY[8 * i + 9] = t;
		    t ^= E_KEY[8 * i + 2]; 
			E_KEY[8 * i + 10] = t;   
		    t ^= E_KEY[8 * i + 3]; 
			E_KEY[8 * i + 11] = t;   
		    t  = E_KEY[8 * i + 4] ^ ls_box(t);  
		    E_KEY[8 * i + 12] = t;                
		    t ^= E_KEY[8 * i + 5]; 
			E_KEY[8 * i + 13] = t;   
		    t ^= E_KEY[8 * i + 6]; 
			E_KEY[8 * i + 14] = t;   
		    t ^= E_KEY[8 * i + 7]; 
			E_KEY[8 * i + 15] = t;   
		}
		break;
	}

	D_KEY[0] = E_KEY[0];
	D_KEY[1] = E_KEY[1];
	D_KEY[2] = E_KEY[2];
	D_KEY[3] = E_KEY[3];

	for (i = 4; i < real_key_len + 24; ++i) 
	{
		imix_col (D_KEY[i], E_KEY[i]);
	}

	for (i = real_key_len + 24; i < real_key_len + 28; ++i)
	{
		D_KEY[i] = E_KEY[i];
	}
	/* Reverse the extended keys for decryption. */
	for (m = 0; m < rounds / 2; m++)
	{
		for (n = 0; n < 4; n++)
		{
			exch = D_KEY[4 * m + n];
			D_KEY[4 * m + n] = D_KEY[4 * (rounds - m) + n];
			D_KEY[4 * (rounds - m) + n] = exch;
		}
	}

	return AES_SUCCEEDED;
}

AES_RESULT
ace_aes_atomic_crypt(struct ace_aes_context *ctx, 
                          int enc,
                          unsigned char *src, 
						  unsigned char *dst, 
						  int nbytes)
{
	AES_RESULT res = AES_SUCCEEDED;
	int count = nbytes / 16;
	size_t bsize = 16;
	unsigned char buf[512];
	unsigned char *p;
	unsigned char *pkey;
	unsigned char *pctrl;
	unsigned char *aligned_iv;
		
	p = ALIGN16(buf);

	// Modified by Danial Miao 2004-06-08
	// Note there is no difference between encryption and decrytion excepting the process of IV
	// for cfb mode ciphergraphy See the original flow of cfb mode ciphergraphy in functions:
	// do_cfb_encrypt() and do_cfb_decrypt() they all call *c->encrypt() to process the data
	switch (ctx->key_length) 
	{
	case KEY_128BITS:
			memcpy(p, E_KEY, 16);
			pkey = p;
			p += 256; //Note: VIA ACE hardare requires 256bytes for storage of cipher key
			          //      regardless the actual size of cipher key.
					  //      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
					  //      for more details about VIA ACE 

			if (enc)
				memcpy(p, &NEH_GENKEY_ENCRYPT_128, 4);
			else
				memcpy(p, &NEH_GENKEY_DECRYPT_128, 4);

			pctrl = p;
			p += 16;

		break;

	case KEY_192BITS:
			if (enc ) 
			{
				memcpy(p, (unsigned char *) (E_KEY), 208);
				pkey = p;
				p += 256;	//Note: VIA ACE hardare requires 256bytes for storage of cipher key
					        //      regardless the actual size of cipher key.
 						    //      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
					        //      for more details about VIA ACE

				memcpy(p, &NEH_LDKEY_ENCRYPT_192, 4);
				pctrl = p;
				p += 16;
			} 
			else 
			{
				if((ctx->mode == ACE_AES_CFB128) || (ctx->mode == ACE_AES_OFB128))
				{
					// Though we decrypt n blocks data,
					// we load encryption extended key for 
					// via c5p ace hardware CFB mode 192 bits key aes algo
					memcpy(p, E_KEY, 208);
					pkey = p;
					p += 256;   //Note: VIA ACE hardare requires 256bytes for storage of cipher key
			                    //      regardless the actual size of cipher key.
								//      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
		 					    //      for more details about VIA ACE
					memcpy(p, &NEH_LDKEY_DECRYPT_192, 4);
					pctrl = p;
					p += 16;
				}
				else
				{
					memcpy(p, D_KEY, 208);
					pkey = p;
					p += 256;  //Note: VIA ACE hardare requires 256bytes for storage of cipher key
			                   //      regardless the actual size of cipher key.
 		   					   //      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
					           //      for more details about VIA ACE
					memcpy(p, &NEH_LDKEY_DECRYPT_192, 4);
					pctrl = p;
					p += 16;
				}
			}

			break;

	case KEY_256BITS:
			if (enc)
			{
				memcpy(p, (unsigned char *) (E_KEY), 240);
				pkey = p;
				p += 256;      //Note: VIA ACE hardare requires 256bytes for storage of cipher key
					           //      regardless the actual size of cipher key.
 		   					   //      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
               				   //      for more details about VIA ACE

				memcpy(p, &NEH_LDKEY_ENCRYPT_256, 4);
				pctrl = p;
				p += 16;
			} 
			else 
			{
				if((ctx->mode == ACE_AES_CFB128) || (ctx->mode == ACE_AES_OFB128))
				{
					// Though we decrypt n blocks data,
					// we load encryption extended key for 
					// via c5p ace hardware CFB mode 256 bits key aes algo
					memcpy(p, E_KEY, 240);
					pkey = p;
					p += 256;   //Note: VIA ACE hardare requires 256bytes for storage of cipher key
			                    //      regardless the actual size of cipher key.
 							    //      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
            					//      for more details about VIA ACE
	
					memcpy(p, &NEH_LDKEY_DECRYPT_256, 4);
					pctrl = p;
					p += 16;
				}
				else
				{
					memcpy(p, D_KEY, 240);
					pkey = p;
					p += 256;   //Note: VIA ACE hardare requires 256bytes for storage of cipher key
			                    //      regardless the actual size of cipher key.
 							    //      Please refer to VIA Nehemiah Advanced Cryptography Engine Programming Guide  
            					//      for more details about VIA ACE
		
					memcpy(p, &NEH_LDKEY_DECRYPT_256, 4);
					pctrl = p;
					p += 16;
				}
			}
			break;
	default:
		    printf("\nInvalid Key Length!\n");
	        return AES_KEY_NOT_SUPPORTED;
	}
	
	if(ctx->iv != NULL)
	{
		aligned_iv = p;
		memcpy(aligned_iv, ctx->iv, bsize);
	}

	switch(ctx->mode)
	{
		case ACE_AES_ECB:	   ace_ecb_op5(pkey, pctrl, src, dst, count);
							   return res;
        case ACE_AES_CBC:      ace_cbc_op6(pkey, pctrl, src, dst, count, aligned_iv);
							   return res;
		case ACE_AES_CFB128:   ace_cfb_op6(pkey, pctrl, src, dst, count, aligned_iv);
		                       return res;
		case ACE_AES_OFB128:   ace_ofb_op6(pkey, pctrl, src, dst, count, aligned_iv);
							   return res;
	}
	
	res = AES_FAILED;
	return res;
}
