using System;
using System.Text; 
using System.Runtime.InteropServices;
using System.Threading; 
using System.Diagnostics; 

namespace System.Sound

{
  public enum WAVEFORMAT : uint
  {
    /// <summary> 11.025 kHz, Mono,   8-bit  </summary>
    _1M08       = 0x00000001,
    /// <summary> 11.025 kHz, Stereo,   8-bit  </summary>
    _1S08       = 0x00000002,
    /// <summary> 11.025 kHz, Mono,   16-bit </summary>
    _1M16       = 0x00000004,       
    /// <summary> 11.025 kHz, Stereo,   16-bit </summary>
    _1S16       = 0x00000008,
    /// <summary> 22.05  kHz, Stereo, 8-bit </summary>
    _2M08       = 0x00000010,
    /// <summary> 22.05  kHz, Stereo, 8-bit </summary>
    _2S08       = 0x00000020,       
    /// <summary> 22.05  kHz, Mono,   16-bit </summary>
    _2M16       = 0x00000040,       
    /// <summary> 22,05   kHz, Stereo, 16-bit </summary>
    _2S16       = 0x00000080,       
    /// <summary> 44.1   kHz, Mono, 8-bit </summary>
    _4M08       = 0x00000100,       
    /// <summary> 44.1   kHz, Stereo, 8-bit </summary>
    _4S08       = 0x00000200,       
    /// <summary> 44.1   kHz, Mono,16-bit </summary>
    _4M16       = 0x00000400,       
    /// <summary> 44.1   kHz, Stereo,16-bit </summary>
    _4S16       = 0x00000800 
  }

  /// <summary>
  /// Encapsulates WaveIn-Device as defined in MMSystem.h for use with coredll.dll
  /// </summary>
  public class WaveIn
  {
    #region types
    /// <summary>
    /// Message-IDs for waveform input callbacks
    /// </summary>
    public enum WIM : int
    {
      OPEN         = 0x3BE,           
      CLOSE        = 0x3BF,
      DATA         = 0x3C0,
    }

    private enum FormatFlags : uint
    {
      INVALIDFORMAT      = 0,       /* invalid format */
      WAVE_FORMAT_QUERY  = 1,
      WAVE_ALLOWSYNC     = 2,
      WAVE_MAPPED        = 4,
      WAVE_FORMAT_DIRECT = 8,
    }

    /// <summary>
    /// delegate type for Wavein_Open callback
    ///  void CALLBACK waveInProc1(HWAVEIN hwi, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
    /// </summary>
    private delegate void WaveInCb(IntPtr hwi, int msg, int data, IntPtr param1, int param2);
    public delegate void WaveInEvent(object sender, WaveInEventArgs e); 
    public class WaveInEventArgs : EventArgs
    {
      public byte[] bArr; 
      public short [] sArr; 
      public int msg; 
    }
    /// <summary>
    /// describes delegate for Open
    /// </summary>
    private enum CALLBACK : long
    {
      TYPEMASK   = 0x00070000L,    /* callback type mask */
      NULL       = 0x00000000L,    /* no callback */
      WINDOW     = 0x00010000L,    /* dwCallback is a HWND */
      //    TASK       = 0x00020000L,    /* dwCallback is a HTASK */
      FUNCTION   = 0x00030000L,    /* dwCallback is a FARPROC */
      //    THREAD     (CALLBACK_TASK)/* thread ID replaces 16 bit task */
      //    EVENT      = 0x00050000L,    /* dwCallback is an EVENT Handle */
    }

    #endregion

    #region Imports
    [DllImport("winmm.dll", SetLastError = true)]
    private static extern bool sndPlaySound(string filename, int fuSound); 
    public static void sndPlaySound(string filename, bool Async)
    {
      if (Async)
        sndPlaySound(filename, 1); 
      else
        sndPlaySound(filename, 0); 
    } 

    [DllImport("winmm.dll", SetLastError = true)]
    private static extern int waveInGetErrorText(int mmrError, StringBuilder pszText, uint cchText); 

    [DllImport("winmm.dll", SetLastError = true)]
    private static extern int waveInGetNumDevs();

    [DllImport("winmm.dll", SetLastError = true)]
    private static extern int waveInClose(System.IntPtr hwi);

    /// <summary>
    /// Opens specified waveform input device for recording. Uses window message for callback.
    /// </summary>
    /// <param name="phwi">Address filled with a handle identifying 
    /// the open waveform-audio input device. Use this handle to identify 
    /// the device later when calling other waveform-audio input functions. </param>
    /// <param name="uDeviceID">Identifier of the waveform-audio input device to open. 
    /// It can be either a device identifier or a Handle to an open waveform-audio input device. 
    /// You can use the flag WAVE_MAPPER (0xffffffff) for automatic selection
    /// instead of a device identifier. </param>
    /// <param name="pwfx">Pointer to a WAVEFORMATEX structure that identifies the 
    /// desired format for recording waveform-audio data. You can free this structure 
    /// immediately after waveInOpen returns. </param>
    /// <param name="hWin">Handle of Window for WIM_OPEN, WIN_DATA, WIN_CLOSE callback messages 
    /// of driver</param>
    /// <param name="dwInstance">Specifies user-instance data passed to the callback mechanism. 
    /// This parameter is not used with the window callback mechanism. </param>
    /// <param name="fdwOpen">Flags for opening the device as defined in FormatFlags enumeration</param>
    /// <returns>Errorcode</returns>
    [DllImport("winmm.dll", EntryPoint = "waveInOpen", SetLastError = true)]
    static extern int waveInOpenHWin(out System.IntPtr phwi, int uDeviceID, WaveFormatEx pwfx, 
      System.IntPtr hWin, uint dwInstance, FormatFlags fdwOpen);

    /// <summary>
    /// Opens specified waveform input device for recording. Uses callback function for callback.
    /// </summary>
    /// <param name="phwi">Address filled with a handle identifying 
    /// the open waveform-audio input device. Use this handle to identify 
    /// the device later when calling other waveform-audio input functions. </param>
    /// <param name="uDeviceID">Identifier of the waveform-audio input device to open. 
    /// It can be either a device identifier or a Handle to an open waveform-audio input device. 
    /// You can use the flag WAVE_MAPPER (0xffffffff) for automatic selection
    /// instead of a device identifier. </param>
    /// <param name="pwfx">Pointer to a WAVEFORMATEX structure that identifies the 
    /// desired format for recording waveform-audio data. You can free this structure 
    /// immediately after waveInOpen returns. </param>
    /// <param name="dwCallback">callback function called with WIM_OPEN, WIN_DATA, WIN_CLOSE messages. 
    /// CAUTION: This function must not call any system (specially waveInXXX) functions except for 
    /// EnterCriticalSection, LeaveCriticalSection, OutputDebugString, PostMessage, PostThreadMessage, 
    /// SetEvent, timeGetSystemTime, timeGetTime, timeKillEvent, and timeSetEvent. Calling other wave 
    /// functions can cause deadlock oder funny behavior. For waveInAddBuffer() call use different 
    /// thread and synchronize mechanism and instead</param>
    /// <param name="dwInstance">Specifies user-instance data passed to the callback mechanism. 
    /// This parameter is not used with the window callback mechanism. </param>
    /// <param name="fdwOpen">Flags for opening the device as defined in FormatFlags enumeration</param>
    /// <returns>Errorcode</returns>
    [DllImport("winmm.dll", EntryPoint = "waveInOpen")]
    static extern int waveInOpenCB(out System.IntPtr phwi, int uDeviceID, WaveFormatEx pwfx, 
      WaveInCb dwCallback, uint dwInstance, FormatFlags fdwOpen);

    /// <summary>
    /// overload for querying device
    /// </summary>
    [DllImport("winmm.dll", EntryPoint = "waveInOpen", SetLastError = true)]
    static extern int waveInOpenQuery(int handle, int uDeviceID, WaveFormatEx pwfx, 
      int cb, int dwInstance, FormatFlags fdwOpen);
  
    /// <summary>
    /// Retrieves the capabilities of a specified waveform-audio input device
    /// </summary>
    /// <param name="uDeviceID">Identifier of the waveform-audio output device. It can be either 
    /// a device identifier or a Handle to an open waveform-audio input device. </param>
    /// <param name="pwic">Pointer to a WAVEINCAPS structure to be filled with information about 
    /// the capabilities of the device. </param>
    /// <param name="cbwic">Size, in bytes, of the WAVEINCAPS structure</param>
    /// <returns>Error code</returns>

    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInGetDevCaps(uint uDeviceID, [Out] WaveInCaps pwic, int cbwic ); 
    /// <summary>
    /// sends an input buffer to the specified waveform-audio input device. When the buffer is filled, 
    /// the application is notified. Buffer must be prepared with the waveInPrepareHeader(). 
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <param name="pwh">Pointer to a WAVEHDR structure that identifies the buffer</param>
    /// <param name="cbwh">Size, in bytes, of the WAVEHDR structure</param>
    /// <returns>MMSYSERR_NOERROR=Success; 
    /// MMSYSERR_INVALHANDLE=Specified device handle is invalid;
    /// MMSYSERR_INVALPARAM=The buffer's base address is not aligned with the sample size;
    /// MMSYSERR_NODRIVER No device driver is present. 
    /// MMSYSERR_NOMEM Unable to allocate or lock memory. 
    /// </returns>
    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInAddBuffer(IntPtr hwi, IntPtr pwh, int cbwh); 

    /// <summary>
    /// prepares a buffer for waveform input.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <param name="pwh">Pointer to a WAVEHDR structure that identifies the buffer</param>
    /// <param name="cbwh">Size, in bytes, of the WAVEHDR structure</param>
    /// <returns>MMSYSERR_NOERROR=Success; 
    /// MMSYSERR_INVALHANDLE=Specified device handle is invalid;
    /// MMSYSERR_INVALPARAM=The buffer's base address is not aligned with the sample size;
    /// MMSYSERR_NODRIVER=No device driver is present. 
    /// MMSYSERR_NOMEM=Unable to allocate or lock memory. 
    /// </returns>
    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInPrepareHeader(IntPtr hwi, IntPtr pwh, int cbwh ); 

    /// <summary>
    /// cleans up the preparation performed by waveInPrepareHeader. The function must 
    /// be called after the device driver fills a data buffer and returns it to the application. 
    /// You must call this function before freeing the data buffer.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <param name="pwh">Pointer to a WAVEHDR structure that identifies the buffer</param>
    /// <param name="cbwh">Size, in bytes, of the WAVEHDR structure</param>
    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInUnprepareHeader(IntPtr hwi, IntPtr pwh, int cbwh);

    /// <summary>
    /// starts input on a specified waveform input device. No effect if already started.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInStart(IntPtr hwi);

    /// <summary>
    /// stops input on a specified waveform input device. No effect if already stopped.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInStop(IntPtr hwi);

    /// <summary>
    /// stops input on a specified waveform input device and resets the current position to 0. All 
    /// pending buffers are marked as done and returned to the application.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    [DllImport("winmm.dll", SetLastError = true)]
    static extern int waveInReset(IntPtr hwi);
    #endregion 

    #region static
    /// <summary>
    /// Retrieves error text for error code
    /// </summary>
    /// <param name="mmrError">error number</param>
    /// <returns>Error text</returns>
    public static string GetErrorText(int mmError)
    {
      const int MAXERRORLENGTH  = 256; 
      StringBuilder pszText = new StringBuilder(MAXERRORLENGTH);
      waveInGetErrorText(mmError, pszText, MAXERRORLENGTH); 
      return pszText.ToString();
    }

    /// <summary>
    /// Retrieves the capabilities of a specified waveform-audio input device.
    /// </summary>
    /// <param name="devID">Identifier of the waveform-audio output device. 
    /// It can be either a device identifier or a Handle to an open waveform-audio input device</param>
    /// <param name="wic">Pointer to a WAVEINCAPS structure to be filled with information about 
    /// the capabilities of the device.</param>
    public static void GetDevCaps(uint devID, WaveInCaps wic)
    {
      HandleErr(waveInGetDevCaps(devID, wic, Marshal.SizeOf(wic)));
    }

    /// <summary>
    /// Retrieves number of installed wavein devices
    /// </summary>
    /// <returns>Number of wavein devices</returns>
    public static int GetNumDevs()
    { 
      return waveInGetNumDevs(); 
    }

    /// <summary>
    /// Tests if device supports requested wave format 
    /// </summary>
    /// <param name="wfe">Address of Structure describing desired wave format</param>
    /// <param name="devId">Id of desired wavein device. WAVEMAPPER (= -1) selects 
    /// next available device</param>
    /// <returns>true, if format is supported</returns>
    public static bool Query(WAVEFORMAT format, int devId)
    {
      return (0 == waveInOpenQuery(0, devId, new WaveFormatEx(format), 0, 0, FormatFlags.WAVE_FORMAT_QUERY)); 
    }

    private static void HandleErr(int mmErr)
    {
      if (mmErr != 0)
        throw new Exception("WaveIn (Error " + mmErr.ToString() + "):\n" + GetErrorText(mmErr)); 
    }
    #endregion
    #region dynamic

    private System.IntPtr hWave = System.IntPtr.Zero;
    private bool _isOpen; 
    private WaveHeader[] wHdr; 

    private AutoResetEvent allowClose = new AutoResetEvent(false);
    private AutoResetEvent allowProcess = new AutoResetEvent(false); 

    private bool resetting; 

    private bool EightBit; 
    private WaveInCb wcb; 

    public event WaveInEvent WIOpen; 
    public event WaveInEvent WIClose; 
    public event WaveInEvent WIData; 

    public void dummy()
    {
    }

    public bool IsOpen
    {
      get
      {
        return _isOpen; 
      }
    }

    protected void OnOpen(object sender, WaveInEventArgs e)
    {
      if (WIOpen != null)
        WIOpen(sender, e); 
    }

    protected void OnClose(object sender, WaveInEventArgs e)
    {
      if (WIClose != null)
        WIClose(sender, e); 
    }

    protected void OnData(object sender, WaveInEventArgs e)
    {
      if (WIData != null)
        WIData(sender, e); 
    }

    private void processMsg(object o)
    {
      while(true) 
      {
        WaveInEventArgs e = new WaveInEventArgs();
        IntPtr actWHdr;  
        allowProcess.WaitOne(); 
        lock(wHdr)  // protect globals
        {
          e.msg = MSG; 
          actWHdr = actWdr; 
        }
        switch ((WIM)e.msg)
        {
          case WIM.OPEN:
            OnOpen(this, e); 
            break; 
          case WIM.CLOSE:
            OnClose(this, e); 
            return; 
            break; 
          case WIM.DATA:
            WaveHeader wh = null;   
            if (WIData != null)  // Handler registriert?
            {
              for (int i=0; i < wHdr.Length; i++)  // find WaveHeader object
                if(actWHdr.Equals(wHdr[i].Handle))
                  wh = wHdr[i]; 
              e.bArr = wh.ToByteArray(); 
              e.sArr = wh.ToShortArray(); 
              OnData(this, e); 
            }
            lock(this) // resetting
            {
              if(resetting)
                allowClose.Set(); 
              else
              {
                wh.Flags = 2; 
                AddBuffer(wh); 
              }
            }
            break; 
        } 
      }
    }


    IntPtr actWdr; 
//    WaveHeader actWaveHeader; 
    int MSG; 
    private void WaveInCbck(System.IntPtr hwi, int msg, int data, IntPtr actWHdr, int param2)
    {
      allowProcess.Reset(); // if last message hasn't been processed yet, it'll be skipped.
      lock(wHdr)   // protect globals
      {
        MSG = msg;          // hand to processMsg() via global var
        if ((WIM)msg == WIM.DATA)
          actWdr = actWHdr; 
      }
      allowProcess.Set();   // Allow thread processMsg() message processing and buffer recycling 
    }

    /// <summary>
    /// Constructor: Opens wavein device. Throws Exceptions if: 
    /// parameters are out of specs, open errors occur, buffer reserving fails, 
    /// or headerpreparing fails
    /// </summary>
    /// <param name="format">desired wave format</param>
    /// <param name="devId">usually 0 or -1 (WaveMapper device); Exception is thrown if device is not
    /// available or unknown</param>
    /// <param name="bufsize">desired size of sound data array presented by data event 
    /// (sbyte[] for 8 bit formats and short[] for 16 bit formats). 
    /// Exception is thrown if value is less than 512 or not dividable by 4</param>
    /// <param name="buffers">Number of internal buffers to be used. Exception is thrown if value is less than 2</param>
    public WaveIn(WAVEFORMAT format, int devId, int bufsize, int bufs) 
    {
      allowClose.Reset();
      const uint _8BitMask   = 0x00000333; 
      EightBit=true; 
      if ((format & (WAVEFORMAT) _8BitMask) == 0) 
      {
        bufsize *= 2; 
        EightBit=false; 
      }
      if (bufsize < 512 || bufs < 1 || bufsize % 4 != 0)
        throw new Exception("Bad Parameters"); 
      wHdr = new WaveHeader[bufs]; 
      wcb = new WaveInCb(WaveInCbck);   // to be conserverd until close() !
      HandleErr(waveInOpenCB(out hWave, devId, new WaveFormatEx(format), wcb, 0, (FormatFlags)CALLBACK.FUNCTION));
      ThreadPool.QueueUserWorkItem(new WaitCallback(processMsg));   // for callback work, e.g. waveInAddBuffer()
      for (int i=0; i<wHdr.Length; i++)   // initialize waveheaders and buffers
      {
        wHdr[i] = new WaveHeader(bufsize); 
        PrepareHeader(wHdr[i]); 
        AddBuffer(wHdr[i]); 
      }
      _isOpen = true; 
    }

    /// <summary>
    /// Constructor: Opens wavein device with WaveMapper and 4 buffers each containig 1024 sample data
    /// </summary>
    /// <param name="format">Description of desired wave format</param>
    public WaveIn(WAVEFORMAT format): this(format, -1, 1024, 4) {}
    /// <summary>
    /// Constructor: Opens wavein device with WAVEFORMAT._2M16, WaveMapper
    /// and 4 buffers each containig 1024 sample data
    /// </summary>
    /// <param name="format">Description of desired wave format</param>
    public WaveIn() : this(WAVEFORMAT._2M16, -1, 1024, 4) {}

    /// <summary>
    /// prepares Wave Header for waveform input.
    /// </summary>
    /// <param name="wHdr"></param>
    private void PrepareHeader(WaveHeader Hdr)
    {
      HandleErr(waveInPrepareHeader(hWave, Hdr.Handle, WaveHeader.size)); 
    }

    /// <summary>
    /// cleans up the preparation performed by PrepareHeader(). The function must be called after 
    /// the device driver fills a data buffer and returns it to the application. 
    /// You must call this function before freeing the data buffer.
    /// </summary>
    /// <param name="wHdr">Wave Header with buffer</param>
    private void UnPrepareHeader(WaveHeader Hdr)
    {
        HandleErr(waveInUnprepareHeader(hWave, Hdr.Handle, WaveHeader.size )); 
    }

    public void Start()
    {
      allowClose.Reset();
      lock(this)
      {
        resetting = false;
      }
      HandleErr(waveInStart(hWave)); 
    }

    public void Stop()
    {
      int res = waveInStop(hWave);
      HandleErr(res); 
    }
    public void Reset()
    {
      Stop(); 
      lock(this)
      {
        resetting = true; 
        HandleErr(waveInReset(hWave)); 
      }
    }

    /// <summary>
    /// Adds a wave buffer to the wavein device. 
    /// </summary>
    /// <param name="wHdr">Wave header with unmanaged buffer</param>
    private void AddBuffer(WaveHeader wHdrs)
    {
      HandleErr(waveInAddBuffer(hWave, wHdrs.Handle, WaveHeader.size )); 
    }
    /// <summary>
    /// starts closing of wavein device. 
    /// end of closing will be annouced by WIClose event
    /// </summary>
    /// <param name="hin">handle for wavein device</param>
    public void Close()
    {
      int id = AppDomain.GetCurrentThreadId(); 
      if (_isOpen)
      {
        _isOpen = false; 
        Reset();                // synchronisiert
        allowClose.WaitOne();   // warte bis aktueller Puffer zurck
        for (int i=0; i<wHdr.Length; i++)
        {
          UnPrepareHeader(wHdr[i]); 
          wHdr[i].Dispose();
        }
        HandleErr(waveInClose(hWave)); 
      }
    }

    ~WaveIn()
    {
      if (_isOpen)
        Close(); 
    }
    #endregion
  }
  #region Strukturen

  [StructLayout(LayoutKind.Sequential, Size = 80)]
  public class WaveInCaps
  {
    /// <summary>
    /// Manufacturer ID
    /// </summary>
    public ushort  wMid;                    
    /// <summary>
    ///  Product ID 
    /// </summary>
    public ushort  wPid;                  
    /// <summary>
    /// Version of the driver
    /// </summary>
    public byte ver0, ver1, ver2, ver3;   // vDriverVersion;  
    /// <summary>
    /// Product name (Marshalled from NULL terminated string)
    /// </summary>
    [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
    public string szName; 
    /// <summary>
    /// Formats supported
    /// </summary>
    public uint   dwFormats;               
    /// <summary>
    /// Number of channels supported
    /// </summary>
    public short    wChannels;               
    /// <summary>
    /// structure packing
    /// </summary>
    public short    wReserved1;              
  }
  #endregion
}

