/* BaseThread.CS
 * Nachkonstruktion der Delphi-Klasse TThread, ziemlich schnrkellos.
 * Abstrakt, nur als Basis eigener Ableitungen zu verwenden, die
 * ihrerseits die Methode Execute implementieren. Typische Variante:
 * protected override void Execute()
 * { while(!Terminated)
 *     // ...;  
 * Assembly kann wahlweise mit Dispose-Implementierung kompiliert werden
 */ 

#define WithDispose  // Fr IDisposable-Variante
using System;
using System.Threading;

namespace System.Threading
{
  public delegate void ThreadCB(BaseThread Sender);

  abstract public class BaseThread 
#if WithDispose    
    :IDisposable
#endif
  {
    private Thread ExecThread; // Steuerung des Threads
    private bool FSuspended;   // angehalten
    private bool FTerminated;  // beendet (write only)
    // Rckmeldung, nachdem Execute durchgelaufen ist. Async!
    public event ThreadCB Terminating;

    // Parameterloser Konstruktor startet den Thread sofort
    public BaseThread() : this (false)
    {
    }

    public BaseThread(bool suspended)
    {
      // Erst Execute, dann OnTerminate
      ExecThread =  new Thread(new ThreadStart(ExecFrame));
      Suspended = suspended;
    }

    // Anhalten und Fortsetzen des Threads (= Execute)
    public bool Suspended
    {
      get { return FSuspended; }
      set 
      {
        FSuspended = value;
        if (value & ExecThread.IsAlive) 
          ExecThread.Suspend();
        else if (!(value | ExecThread.IsAlive))
        {
          if ((ExecThread.ThreadState & ThreadState.Unstarted) != 0)
            ExecThread.Start();
          else if ((ExecThread.ThreadState & ThreadState.Suspended) != 0)
            ExecThread.Resume();
        }
      }
    }

    public void Terminate()
    {
      FTerminated = true;
    }

    public bool Terminated
    {
      get { return FTerminated; }
    }

    // Rckmeldung, nachdem Execute durch ist (siehe ExecFrame)
    protected virtual void OnTerminate(BaseThread sender)
    {
      if (Terminating != null)
        Terminating(sender);
    }

    public void Abort()  // durchgereicht
    {
      ExecThread.Abort();
    }

    // muss in abgeleiteten Klassen implementiert werden
    protected abstract void Execute();

    private void ExecFrame()
    {
      try
      {
        Execute();
      }
      finally
      {
        OnTerminate(this);  // Ereignis wird auch bei Abort() signalisiert
#if WithDispose   
        Dispose();          // Aufrumen
#endif
      }
    }

#if WithDispose    // Implementation mit Dispose
    public void Dispose()   // bliche Disposelogik
    {
      Dispose(true);
    }

    // Kann von Ableitung berschrieben werden
    protected virtual void Dispose(bool disposing)  // Ressourcen wegrumen
    { 
    }
    ~BaseThread()           // Destruktor
    {
      Dispose(false);
    }
#endif
  }
}
