/**
 * OnTime: misst die Zeit, die das Gert angeschaltet ist
 * Verwendet die Nachrichten sysNotifySleepNotifyEvent und
 * sysNotifyLateWakeupEvent
 * luft ab Palm OS 3.5 
 */
#include <PalmOS.h>
#include "OnTimer_res.h"

// Creator ID dieser Anwendung
#define CREATORID 'ctOT'
// Features zum Speichern der Zwischenzeiten
#define FTR_SUM_ON_TIME 1
#define FTR_LAST_WAKEUP_TIME 2
#define FTR_ON_SINCE 3

/**
 * Gib gesamte "An"-Zeit in Sekunden zurck.
 * Summe aus Zeit bis zum letzten Ausschalten 
 * und Zeit seit aktuellem Anschalten.
 */
UInt32 totalWakeTime()
{
UInt32 lastontime=0;
UInt32 prevwaketime=0;
FtrGet(CREATORID,FTR_SUM_ON_TIME,&prevwaketime);
if(FtrGet(CREATORID,FTR_LAST_WAKEUP_TIME,&lastontime)!=errNone)
	{
	// Feature nicht gefunden, beginne mit aktueller Zeit
	FtrSet(CREATORID,FTR_LAST_WAKEUP_TIME,TimGetSeconds());
	FtrSet(CREATORID,FTR_ON_SINCE,TimGetSeconds());
	return 0;
	}
return prevwaketime+TimGetSeconds()-lastontime;
}

/**
 * Formatiere und schreibe Zeiten
 * auf den Bildschirm
 */
static void drawTime()
{
FormPtr frm=FrmGetActiveForm();
RectangleType rect;
Coord screenwidth,screenheight;
Char str[50];
DateTimeType dt;
DateType date;
UInt32 seconds;

// Bestimme Bildschirmbreite
WinGetDisplayExtent(&screenwidth,&screenheight);

// Kumulierte Zeit
TimSecondsToDateTime(totalWakeTime(),&dt);
date.year=dt.year-1904;
date.month=dt.month;
date.day=dt.day;
StrIToA(str,DateToDays(date));StrCat(str,"d ");
StrIToA(str+StrLen(str),dt.hour);StrCat(str,"h ");
StrIToA(str+StrLen(str),dt.minute);StrCat(str,"m ");
StrIToA(str+StrLen(str),dt.second);StrCat(str,"s");
FrmGetObjectBounds(frm,FrmGetObjectIndex(frm,lblOnTime),&rect);
rect.topLeft.x+=rect.extent.x+FntCharWidth(' ');
rect.extent.x=screenwidth-rect.topLeft.x;
WinEraseRectangle(&rect,0); // lsche alten Text
WinDrawChars(str,StrLen(str),rect.topLeft.x,rect.topLeft.y);

// Datum seit Beginn der Messung
FtrGet(CREATORID,FTR_ON_SINCE,&seconds);
TimSecondsToDateTime(seconds,&dt);
DateToAscii(dt.month,dt.day,dt.year,dfDMYWithDots,str);
FrmGetObjectBounds(frm,FrmGetObjectIndex(frm,lblOnSince),&rect);
rect.topLeft.x+=rect.extent.x+FntCharWidth(' ');
rect.extent.x=screenwidth-rect.topLeft.x;
WinEraseRectangle(&rect,0); // lsche alten Text
WinDrawChars(str,StrLen(str),rect.topLeft.x,rect.topLeft.y);
}

/**
 * Event-Schleife des Forms
 */
Boolean frmMain_HandleEvent(EventPtr event)
{
	FormPtr form;
	Boolean handled = false;

	switch (event->eType)
	{
		case ctlSelectEvent:
			switch (event->data.ctlSelect.controlID)
			{
				case btnReset:
					// setze alle Features zurck
					FtrSet(CREATORID,FTR_SUM_ON_TIME,0);
					FtrSet(CREATORID,FTR_ON_SINCE,TimGetSeconds());
					FtrSet(CREATORID,FTR_LAST_WAKEUP_TIME,TimGetSeconds());
					drawTime();		
					handled = true;
					break;
			}
			break;
		case frmOpenEvent:
			form = FrmGetActiveForm();
			FrmDrawForm(form);
			drawTime();
			handled = true;
			break;
		case nilEvent:
			// aktualisiere Display
			drawTime();
			break;
		default:
			break;
	}

	return handled;
}

/**
 * Gibt true zurck, wenn das Gert Nachrichten untersttzt
 */
static Boolean hasNotifications()
{
UInt32 value;
Err err = FtrGet(sysFtrCreator,
sysFtrNumNotifyMgrVersion, &value);
return (err==errNone && value!=0);
}

/**
 * Registriere das Programm auf die interessierenden Nachrichten
 */
static void registerNotifications()
{
UInt16 cardNo;
LocalID dbID;
SysCurAppDatabase(&cardNo,&dbID);
SysNotifyRegister (cardNo,dbID,sysNotifySleepNotifyEvent,NULL,sysNotifyNormalPriority,NULL);
SysNotifyRegister (cardNo,dbID,sysNotifyLateWakeupEvent,NULL,sysNotifyNormalPriority,NULL);
}

static Boolean ApplicationHandleEvent(EventPtr event)
{
	UInt16 formID;
	FormPtr form;
	Boolean handled = false;

	switch (event->eType)
	{
		case frmLoadEvent:
			formID = event->data.frmLoad.formID;
			form = FrmInitForm(formID);
			FrmSetActiveForm(form);		

			switch (formID)
			{
				case frmMain:
					// Set event handler for frmMain
					FrmSetEventHandler(form, (FormEventHandlerPtr) frmMain_HandleEvent);
					break;
				default:
					break;
			}
			handled = true;
			break;
		default:
			break;
	}

	return handled;
}


static void EventLoop(void)
{
	Err error;
	EventType event;

	do
		{
		// Hole Events mindestens 1x pro Sekunde (Display-Aktualisierung)
		EvtGetEvent(&event, 1UL*SysTicksPerSecond()); 
		if (!SysHandleEvent(&event))
			if (!MenuHandleEvent(0, &event, &error))
				if (!ApplicationHandleEvent(&event))
					FrmDispatchEvent(&event);	
		}
	while (event.eType != appStopEvent);
}

/**
 * main()-Routine jedes Palm-OS-Programms.
 * Diese Routine wird beim Programmstart und 
 * auch fr alle Nachrichten und Launch Codes aufgerufen.
 */
UInt32 PilotMain(UInt16 cmd, void *cmdPBP, UInt16 launchFlags)
{
	switch (cmd) 
 	{
		case sysAppLaunchCmdNormalLaunch:
			if(!hasNotifications())
				{
				FrmAlert(alertNotifications);
				return 0;
				}
			// registriere Anwendung erneut, falls Registrierung verloren
			// gegangen ist, z.B. bei einem Soft Reset
            if(hasNotifications())
            	registerNotifications();
            FrmGotoForm(frmMain);
			EventLoop();
			FrmCloseAllForms();
			break;
			
		case sysAppLaunchCmdNotify:
			switch(((SysNotifyParamType*)cmdPBP)->notifyType)
				{
				case sysNotifySleepNotifyEvent:
					// Gert geht in den Schlafmodus ber
					// aktualisiere Zeitsumme
					FtrSet(CREATORID,FTR_SUM_ON_TIME,totalWakeTime());
					break;
				case sysNotifyLateWakeupEvent:
					// Gert wacht gerade auf, speichere Aufwachzeit
					FtrSet(CREATORID,FTR_LAST_WAKEUP_TIME,TimGetSeconds());
					break;
				default:
					break;
				} 
		case sysAppLaunchCmdSyncNotify:
			// registriere Anwendung automatisch nach der Installation
			if(hasNotifications())
				registerNotifications();
			break;									
		case sysAppLaunchCmdSystemReset:
			// registriere Anwendung automatisch nach Reset (Vorsicht!) 
			//if(hasNotifications())
			//	registerNotifications();
			break;									
		default:
			break;

	}

	return 0;
}

