//----------------------------------------------------------------------------
// WinATA.cpp
// Befehle an ATA-Gerte schicken
// fr Windows NT4(ab SP5)/2000/XP
// 2002 c't/Matthias Withopf

#include <stdio.h>
#include <stdlib.h>
#define WIN32_LEAN_AND_MEAN
#include <windows.h>

//----------------------------------------------------------------------------
// Klassendefinition von CWinATA

class CWinAta  
{
  public:
    CWinAta(UINT DeviceNum);
    ~CWinAta();
       
    // Fehlercodes der Klasse CWinAta
    typedef enum 
    {
        aec_Ok               = 0, 
        aec_OSNotSupported   = 1, 
        aec_InvalidDevice    = 2, 
        aec_InvalidParameter = 3, 
        aec_OutOfMemory      = 4, 
        aec_CommandFailed    = 5, 
        aec_LAST
      } TWinAtaErrorCode;
    TWinAtaErrorCode WinAtaErrorCode;

    // SendCommand sendet ATA-Befehle
    // Registersatz des ATA-Hostadapter
    typedef struct
    {
        UCHAR Reg[7];
    } TATARegs, *PATARegs;
   
    BOOL SendCommand(PATARegs pATARegs, void *pBuf, unsigned int BufSize, unsigned int *pResultSize);
    
  private:
    // Handle des geffneten Gertetreibers
    HANDLE Device;
    
    // Windows NT 4 verwendet eine andere Schnittstelle
    BOOL   IsNT4;
    
    // interne Funktionen, je nach Betriebssystem
    BOOL SendCommand_WinNT(PATARegs pATARegs, void *pBuf, unsigned int BufSize, unsigned int *pResultSize);
    BOOL SendCommand_Win2000(PATARegs pATARegs, void *pBuf, unsigned int BufSize, unsigned int *pResultSize);
    
    //---------------------------------------------------
    // Registerdefinition des ATA-Controllers laut Windows-Schnittstelle
    // aus DDK-Headerfile ntdddisk.h bernommen
    
    typedef struct
    {
        UCHAR bFeaturesReg;
        UCHAR bSectorCountReg;
        UCHAR bSectorNumberReg;
        UCHAR bCylLowReg;
        UCHAR bCylHighReg;
        UCHAR bDriveHeadReg;
        UCHAR bCommandReg;
        UCHAR bReserved;
    } IDEREGS, *PIDEREGS;
};

//----------------------------------------------------------------------------
// Der Konstruktor prft das Betriebssystem und ffnet den Gertetreiber

CWinAta::CWinAta(UINT DeviceNum)
{
  IsNT4  = FALSE;
  Device = NULL;
  DWORD V = GetVersion();
  UCHAR MajorVersion = (UCHAR)V;
  
  
  if (V & 0x80000000)
    // Windows 9x wird nicht untersttzt
    WinAtaErrorCode = aec_OSNotSupported;     
  else
  {
      if (MajorVersion < 4)
        // Windows NT vor Version 4 wird nicht untersttzt
        WinAtaErrorCode = aec_OSNotSupported; 
      else if (MajorVersion == 4)
      {
          IsNT4 = TRUE;
          WinAtaErrorCode = aec_Ok;           // Windows NT 4
      }
      else
        WinAtaErrorCode = aec_Ok;             // Windows 2000 oder neuer
  }

  if (!WinAtaErrorCode)
  {
      if (DeviceNum < 8)
      {
          // Gertetreiber als Datei ffnen
          char tmp[128];
          sprintf(tmp, "\\\\.\\PhysicalDrive%u", DeviceNum);
          Device = CreateFile(tmp, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);
          if (Device == INVALID_HANDLE_VALUE)
            WinAtaErrorCode = aec_InvalidDevice;
          else
            WinAtaErrorCode = aec_Ok;
      }
      else
        WinAtaErrorCode = aec_InvalidDevice;
  }
}

//----------------------------------------------------------------------------
// Destruktor schliet Gertetreiber
CWinAta::~CWinAta()
{
  if (Device)
    CloseHandle(Device);
}

//----------------------------------------------------------------------------
// SendCommand prft Parameter
BOOL CWinAta::SendCommand(PATARegs pATARegs, void *pBuf, unsigned int BufSize, unsigned int *pResultSize)
{
  
  // Vorbeugend Gre der Antwort auf 0 setzten
  if (pResultSize) *pResultSize = 0;
  
  //  Parameterprfung 
  if (!pATARegs)
    WinAtaErrorCode = aec_InvalidParameter;
  else if ((pBuf && !BufSize) || (!pBuf && BufSize))
    WinAtaErrorCode = aec_InvalidParameter;
  
  else
  { 
    // Parameter o.k.
    
    if (IsNT4)
      return SendCommand_WinNT(pATARegs, pBuf, BufSize, pResultSize);
    else
      return SendCommand_Win2000(pATARegs, pBuf, BufSize, pResultSize);
  }
  
  // Return bei falschen Parametern
  return FALSE;
}

// Definitionen fr Windows NT sowie 2000 und XP 
// aus dem DDK-Headerfile ntddscsi.h bernommen
#define FILE_DEVICE_CONTROLLER      0x00000004
#define IOCTL_SCSI_BASE             FILE_DEVICE_CONTROLLER
#define FILE_READ_ACCESS            0x0001
#define FILE_WRITE_ACCESS           0x0002
#define METHOD_BUFFERED             0
#define CTL_CODE(DeviceType, Function, Method, Access) (((DeviceType) << 16) | ((Access) << 14) | ((Function) << 2) | (Method))


//----------------------------------------------------------------------------
// ATA-Befehle unter Windows NT 4 versenden
// fr Windows 2000 und Nachfolger nicht geeignet

BOOL CWinAta::SendCommand_WinNT(PATARegs pATARegs, void *pBuf, unsigned int BufSize, unsigned int *pResultSize)
{
  
  //---------------------------------------------------
  // Definitionen und Strukturen fr Windows NT
  // aus dem DDK-Headerfile ntddscsi.h bernommen
  
  // dokumentierter Funktionscode fr SCSI
  #define IOCTL_SCSI_PASS_THROUGH     CTL_CODE(IOCTL_SCSI_BASE, 0x0401, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS)
  #define CDB10GENERIC_LENGTH         10
  #define SCSI_IOCTL_DATA_OUT          0
  #define SCSI_IOCTL_DATA_IN           1
  #define SCSI_IOCTL_DATA_UNSPECIFIED  2
   // undokomentierter Pseudo-SCSI-Befehl
  #define SCSIOP_ATA_PASSTHROUGH      0xCC

  // Datenstruktur fr SCSI-Befehle
  typedef struct
  {
      USHORT Length;
      UCHAR  ScsiStatus;
      UCHAR  PathId;
      UCHAR  TargetId;
      UCHAR  Lun;
      UCHAR  CdbLength;
      UCHAR  SenseInfoLength;
      UCHAR  DataIn;
      UCHAR  _pad[3];
      ULONG  DataTransferLength;
      ULONG  TimeOutValue;
#if (_WIN32_WINNT >= 0x0500)
      ULONG_PTR DataBufferOffset;
#else
      ULONG  DataBufferOffset;
#endif
      ULONG  SenseInfoOffset;
      UCHAR  Cdb[16];
  } SCSI_PASS_THROUGH;

  // Datenstruktur fr SCSI-Befehl mit Daten
  typedef struct
  {
      SCSI_PASS_THROUGH spt;
      ULONG             Filler;
      UCHAR             ucSenseBuf[32];
      UCHAR             ucDataBuf[2 * 1024];
  } SCSI_PASS_THROUGH_WITH_BUFFERS, *PSCSI_PASS_THROUGH_WITH_BUFFERS;
  //---------------------------------------------------
  
  BOOL Result = FALSE;
   
  // Speicherplatz fr Befehle und Daten anlegen
  SCSI_PASS_THROUGH_WITH_BUFFERS PT;
  memset(&PT, 0, sizeof PT);
  
  // fr ATA eigentlich unntige SCSI-Parameter angeben
  PT.spt.Length           = sizeof(SCSI_PASS_THROUGH);
  PT.spt.PathId           = 0;
  PT.spt.TargetId         = 1;
  PT.spt.Lun              = 0;
  PT.spt.CdbLength        = CDB10GENERIC_LENGTH;
  PT.spt.SenseInfoLength  = 24;
  PT.spt.TimeOutValue     = 2;
  PT.spt.SenseInfoOffset  = FIELD_OFFSET(SCSI_PASS_THROUGH_WITH_BUFFERS, ucSenseBuf);
  
  // Position der eigentlichen Daten
  PT.spt.DataBufferOffset = FIELD_OFFSET(SCSI_PASS_THROUGH_WITH_BUFFERS, ucDataBuf);
  memset(PT.ucDataBuf, 0, sizeof PT.ucDataBuf);
  
  if (pBuf)
  {
    unsigned int TransferSize = BufSize;
    
    if (TransferSize > sizeof PT.ucDataBuf)
      TransferSize = sizeof PT.ucDataBuf;
    
    PT.spt.DataIn             = SCSI_IOCTL_DATA_IN;
    PT.spt.DataTransferLength = TransferSize;
  }
  else
  {
    PT.spt.DataIn             = SCSI_IOCTL_DATA_UNSPECIFIED;
    PT.spt.DataTransferLength = 0;
  }
  
  // Pseudo-SCSI-Befehl zum bermitteln eine ATA-Befehls
  PT.spt.Cdb[0]          = SCSIOP_ATA_PASSTHROUGH;
  // ATA-Register folgen in PT.spt.Cdb
  PIDEREGS pRegs = (PIDEREGS)&PT.spt.Cdb[2];
  pRegs->bFeaturesReg     = pATARegs->Reg[0];
  pRegs->bSectorCountReg  = pATARegs->Reg[1];
  pRegs->bSectorNumberReg = pATARegs->Reg[2];
  pRegs->bCylLowReg       = pATARegs->Reg[3];
  pRegs->bCylHighReg      = pATARegs->Reg[4];
  pRegs->bDriveHeadReg    = pATARegs->Reg[5];
  pRegs->bCommandReg      = pATARegs->Reg[6];
  
  // Gesamtgre von Befehlen und Datenpuffer berechnen
  unsigned int DataBufOfs = FIELD_OFFSET(SCSI_PASS_THROUGH_WITH_BUFFERS, ucDataBuf);
  DWORD length = DataBufOfs + PT.spt.DataTransferLength;
  
  
  // Gertetreiber meldet Gre der zurckgegebenen Daten
  DWORD BytesReturned = 0;
  
  // Gertetreiber aufrufen 
  BOOL Status = DeviceIoControl(Device, IOCTL_SCSI_PASS_THROUGH, 
                            &PT, sizeof(SCSI_PASS_THROUGH), &PT, length, &BytesReturned, FALSE);
                            
  // bei Erfolg liefert DeviceIoControl einen Wert ungleich 0      
  if (!Status)
    WinAtaErrorCode = aec_CommandFailed;
  else
  {
    // DeviceIOControl erfolgreich	
    Result = TRUE;
    if (pBuf)
    {
      if (BufSize) 
        memset(pBuf, 0, BufSize);
        
      
      unsigned int ReturnedSize = BytesReturned;
      if (ReturnedSize >= DataBufOfs)
      {  ReturnedSize -= DataBufOfs;
           
         if (pResultSize) 
           *pResultSize = ReturnedSize;
      
         if (ReturnedSize > BufSize)
           ReturnedSize = BufSize;
         
         // Daten kopieren  
         memcpy(pBuf, PT.ucDataBuf, ReturnedSize);
      }
    }
  }
  return Result;
}

//----------------------------------------------------------------------------
// ATA-Befehle unter Windows 2000 oder XP versenden

BOOL CWinAta::SendCommand_Win2000(PATARegs pATARegs, void *pBuf, unsigned int BufSize, unsigned int *pResultSize)
{
  
  //---------------------------------------------------
  // Definitionen und Datenstruktur fr Windows 2000 und XP
  // aus dem DDK-Headerfile ntddscsi.h bernommen
  
  #define IOCTL_IDE_PASS_THROUGH  CTL_CODE(IOCTL_SCSI_BASE, 0x040A, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS)
  #define FILE_ANY_ACCESS         0
  #define IOCTL_SCSI_RESCAN_BUS   CTL_CODE(IOCTL_SCSI_BASE, 0x0407, METHOD_BUFFERED, FILE_ANY_ACCESS)

  typedef struct
  {
      IDEREGS IdeReg;
      ULONG   DataBufferSize;
      UCHAR   DataBuffer[1];
  } ATA_PASS_THROUGH;
  //---------------------------------------------------
  
  
  BOOL Result = FALSE;
  
  unsigned int Size = sizeof(ATA_PASS_THROUGH) + BufSize;
  // Datenpuffer mit VirtualAlloc anfordern, damit der Gertetreiber den Speicher ansprechen kann
  ATA_PASS_THROUGH *pAPT = (ATA_PASS_THROUGH *)VirtualAlloc(NULL, Size, MEM_COMMIT, PAGE_READWRITE);
  
  if (pAPT)
  {
      memset(pAPT, 0, Size);
      // Gre des Datenpuffers setzen
      pAPT->DataBufferSize          = BufSize;
      // ATA-Register setzen
      pAPT->IdeReg.bFeaturesReg     = pATARegs->Reg[0];
      pAPT->IdeReg.bSectorCountReg  = pATARegs->Reg[1];
      pAPT->IdeReg.bSectorNumberReg = pATARegs->Reg[2];
      pAPT->IdeReg.bCylLowReg       = pATARegs->Reg[3];
      pAPT->IdeReg.bCylHighReg      = pATARegs->Reg[4];
      pAPT->IdeReg.bDriveHeadReg    = pATARegs->Reg[5];
      pAPT->IdeReg.bCommandReg      = pATARegs->Reg[6];

      if (pAPT->IdeReg.bCommandReg == 0xEC)  // ist es ATA IDENTIFY?
      {
	  // Windows XP merkt sich den IDE-Konfigurationssektor 
          // daher vorher mit IOCTL_SCSI_RESCAN_BUS fr ungltig erklren
          DWORD BytesReturned = 0;
          (void)DeviceIoControl(Device, IOCTL_SCSI_RESCAN_BUS, NULL, 0, NULL, 0, &BytesReturned, FALSE);
          // eine halbe Sekunde warten
          Sleep(500);    
      }

      DWORD BytesReturned = 0;
      
      BOOL Status = DeviceIoControl(Device, IOCTL_IDE_PASS_THROUGH, 
                                    pAPT, Size, pAPT, Size, &BytesReturned, FALSE);
      if (!Status)
        WinAtaErrorCode = aec_CommandFailed;
      else
      {
          if (pBuf)
          {
              if (BufSize) 
                memset(pBuf, 0, BufSize);
              if (BytesReturned > sizeof(ATA_PASS_THROUGH))
              {
                  unsigned int ReturnedSize = BytesReturned - sizeof(ATA_PASS_THROUGH);
                  if (pResultSize) 
                    *pResultSize = ReturnedSize;
                  if (ReturnedSize > BufSize)
                    ReturnedSize = BufSize;
                    
                  memcpy(pBuf, pAPT->DataBuffer, ReturnedSize);
              }
          }
          Result = TRUE;
      }
      VirtualFree(pAPT, Size, MEM_RELEASE);
  }
  else
    WinAtaErrorCode = aec_OutOfMemory;
 
  return Result;
}

//----------------------------------------------------------------------------
// Hilfsfunktionen

char *Title = "WinATA 1.0  2002 c't/Matthias Withopf";

void MsgBox(char *Text)
{
  MessageBox(0, Text, Title, MB_OK);
}

//----------------------------------------------------------------------------
// GetIdentifyStr kopiert Strings aus dem IDE-Konfigurationssektor

static void GetIdentifyStr(char *s, PWORD pIdentifyBuf, unsigned int IdentifyBufSize)
{
  PWORD p  = pIdentifyBuf;
  char *p1 = s;
  
  // wandelt Big Endian in Little Endian
  for (unsigned int i = 0; i < IdentifyBufSize; ++i, ++p)
  {
      *p1++ = (char)(*p >> 8);
      *p1++ = (char)*p;
  }
  *p1 = '\0';
  
  // fhrende Leerzeichen entfernen
  while (s[0] == ' ')
    memmove(s, s + 1, strlen(s));
    
  // abschlieende Leerzeichen entfernen
  unsigned int l = strlen(s);
  while ((l > 0) && (s[l - 1] == ' '))
    s[--l] = '\0';
}

//----------------------------------------------------------------------------
// Hauptprogramm

int WINAPI WinMain(HINSTANCE /*hInstance*/, HINSTANCE /*hPrevInstance*/, LPSTR CmdLine, int /*ShowCmd*/)
{
  char tmp[4 * 1024];
  
  UINT DeviceNum = 0;
  // Nummer der Festplatte kann als optionaler Parameter beim Programmstart angegeben werden
  if (CmdLine && CmdLine[0])
    DeviceNum = atoi(CmdLine);  

  // Objekt der Klasse CWinAta erzeugen
  CWinAta *pAta = new CWinAta(DeviceNum);
  if (pAta)
  {
      if (pAta->WinAtaErrorCode)
      {
          if (pAta->WinAtaErrorCode == CWinAta::aec_OSNotSupported)
            sprintf(tmp, "Dieses Betriebssystem wird nicht untersttzt!\n");
          else
            sprintf(tmp, "Fehler (%u) beim Ansprechen der Festplatte %u!\n", pAta->WinAtaErrorCode, DeviceNum);
          MsgBox(tmp);
      }
      else
      {
          // IDE-Konfigurationssektor auslesen
          WORD IdentifyBuf[256];
          memset(IdentifyBuf, 0, sizeof IdentifyBuf);
          // ATA-Register setzen
          CWinAta::TATARegs ATARegs;
          memset(&ATARegs, 0, sizeof ATARegs);
          // ATA-Befehl IDENTIFY
          ATARegs.Reg[6] = 0xEC;
          
          unsigned int ResultSize;
          // SendCommand der Klasse CWinAta aufrufen, liefert bei Erfolg TRUE
          if (pAta->SendCommand(&ATARegs, IdentifyBuf, sizeof IdentifyBuf, &ResultSize))
          {
              char ModelStr   [40 + 1];
              char FirmwareStr[ 8 + 1];
              char SerialStr  [20 + 1];
              GetIdentifyStr(ModelStr,    IdentifyBuf + 27, 20);
              GetIdentifyStr(FirmwareStr, IdentifyBuf + 23,  4);
              GetIdentifyStr(SerialStr,   IdentifyBuf + 10, 10);

              // untersttzt die Festplatte Accoustic Management? 
              char *Msg;
              BOOL AAMSupported = FALSE;
              BOOL AAMEnabled   = FALSE;
              BYTE AAMValue     = 0;
              
              if ((IdentifyBuf[83] != 0) && (IdentifyBuf[83] != 0xFFFF) && (IdentifyBuf[83] & 0x0200))
              {
                  // Platte untersttzt Accoustic Management
                  AAMSupported = TRUE;
                  if (IdentifyBuf[86] & 0x0200)
                  {
                      AAMEnabled = TRUE;
                      // aktuelle Einstellung der Platte 
                      AAMValue = (BYTE)IdentifyBuf[94];
                      Msg = "untersttzt Akustik-Management";
                  }
                  else
                    Msg = "hat das Akustik-Management deaktiviert";
              }
              else
                Msg = "untersttzt kein Akustik-Management";

              sprintf(tmp, "Die Festplatte\n Modell %s\n Firmware %s\n Seriennummer %s\n%s.", 
                      ModelStr, FirmwareStr, SerialStr, Msg);
              if (!AAMEnabled)
                MsgBox(tmp);
              else
              {
                  BYTE v = AAMValue;
                  if (v < 128)
                    v = 128;
                  else if (v > 254)
                    v = 254;

                  BYTE NewValue;
                  char tmp1[256];
                  strcat(tmp, "\n\nMchten Sie die Einstellung von ");
                  if (AAMValue >= 192)
                  {
                      NewValue = 128;
                      sprintf(tmp1, "Laut (%u) auf Leise (%u)", AAMValue, NewValue);
                  }
                  else
                  {
                      NewValue = 254;
                      sprintf(tmp1, "Leise (%u) auf Laut (%u)", AAMValue, NewValue);
                  }
                  strcat(tmp, tmp1);
                  strcat(tmp, " ndern?");

                  if (MessageBox(0, tmp, Title, MB_YESNO | MB_DEFBUTTON2) == IDYES)
                  {
                      // neuen Wert setzen
                      memset(&ATARegs, 0, sizeof ATARegs);
                      ATARegs.Reg[6] = 0xEF;    // ATA-Befehl SET FEATURES
                      ATARegs.Reg[0] = 0x42;    // Subcommand Enable Automatic Acoustic Management feature set
                      ATARegs.Reg[1] = NewValue;
                      if (pAta->SendCommand(&ATARegs, NULL, 0, &ResultSize))
                      {
                          // IDE-Konfigurationssektor zum Prfen der Einstellung erneut lesen
                          BOOL IsOk = FALSE;
                          memset(&ATARegs, 0, sizeof ATARegs);
                          // ATA-Befehl IDENTIFY
                          ATARegs.Reg[6] = 0xEC;        
                          memset(IdentifyBuf, 0, sizeof IdentifyBuf);
                          unsigned int ResultSize;

                          // SendCommand der Klasse CWinAta aufrufen, liefert bei Erfolg TRUE
                          if (pAta->SendCommand(&ATARegs, IdentifyBuf, sizeof IdentifyBuf, &ResultSize))
                          {
                              BYTE V = (BYTE)IdentifyBuf[94];
                              if (V == NewValue)
                                IsOk = TRUE;
                          }

                          if (IsOk)
                            MsgBox("Die Einstellung wurde gendert.");
                          else
                            MsgBox("Die Einstellung konnte nicht gendert werden!");
                      }
                      else
                      {
                          sprintf(tmp, "Fehler (%u) beim Setzen des Wertes fr Festplatte %u!\n", pAta->WinAtaErrorCode, DeviceNum);
                          MsgBox(tmp);
                      }
                  }
              }
          }
          else
          {
              sprintf(tmp, "Fehler (%u) beim Identifizieren der Festplatte %u!\n", pAta->WinAtaErrorCode, DeviceNum);
              MsgBox(tmp);
          }
      }
      delete pAta;
  }
  return 0;
}

