unit CubeWorldIMU; // 14-MAY-2000 as (Arne Schpers)
{ Flug durch eine Wrfelwelt, im Immediate Mode von Direct3D
  Setzt DirectX 7 und die Header-Dateien von Erik Unger
  sowie das von mir auf Delphi umgesetzte SDK-Rahmenprogramm
  fr den Immediate Mode (D3DApp) voraus.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Math,  // arctan2
  DirectInput, DXInputAS,  // DirectInput, Delphi-Objekte dazu
  Direct3D, DirectDraw,  // Header von Erik Unger
  // D3D-Rahmenprogramm aus dem DX-SDK, auf Delphi umgesetzt
  D3DEnum,D3DApp,D3DFrame,D3DFile,D3DTextr,D3DUtil,D3DMath;

const
  XYZRANGE = 20; // alle drei Arraydimensionen von 0..XYZRANGE-1
  BackBMPName = 'stars.bmp';  // Hintergrund-Bitmap
  CubeXFilename = 'cube.x';   // Wrfel als X-Datei
  TEXREPEAT = 2;   // Wiederholungen des Hintergrund-Bitmaps

type
  TCWForm = class(TD3DApplication)
    procedure FormCreate(Sender: TObject);
    procedure FormActivate(Sender: TObject);
  private  // fr den Sternenhintergrund
    m_BackgroundMesh: Array[0..3] of TD3DTLVERTEX;
  private  // das endlos wiederholt gezeichnete Objekt
    m_pCubeFile: TD3DFile;
    m_pCubeVertices: PD3DVertex;
    m_dwCubeNumVertices: Cardinal;
    m_pCubeIndices: PWord;
    m_dwCubeNumIndices: Cardinal;
  private
    fFogEnd: TD3DValue;
    vEyePt, vUpVec, vDirection: TD3DVector;
    Cubes: Array[0..XYZRANGE-1,0..XYZRANGE-1,
             0..XYZRANGE-1] of Boolean;
    // Winkelgeschwindigkeit, Speed geradeaus
    LRSpeed, UpDownSpeed, LinSpeed: TD3DValue;
    procedure SetCameraPosition;
  protected
    procedure PopulateCubes(x,y,z: Integer);  // Random-Funktion
  protected
    function DrawBackground: HRESULT;
    function DrawCubes: HRESULT;
  protected  // Overrides fr TD3DApplication
    function OneTimeSceneInit: HResult; override;
    function InitDeviceObjects: HResult; override;
    function DeleteDeviceObjects: HResult; override;
    function Render: HResult; override;
    function FrameMove(fTimeKey: FLOAT): HResult; override;
    function FinalCleanup: HResult; override;
  end;

var
  CWForm: TCWForm;

implementation
{$R *.DFM}

procedure TCWForm.FormCreate(Sender: TObject);
begin
  Caption := 'CubeWorld (Immediate Mode) - H for Help';
  m_bAppUseZBuffer  := TRUE;
  m_bAppUseStereo   := TRUE;
  m_bShowStats      := TRUE;
  CreateFramework; // D3D-Rahmen anlegen, ruft OneTimeSceneInit auf
end;

// Tastaturabfrage ber DirectInput
procedure TCWForm.FormActivate(Sender: TObject);
begin
  inherited;
  DIKeyboard.CooperativeLevel := DISCL_FOREGROUND
    or DISCL_NONEXCLUSIVE;
end;

procedure TCWForm.PopulateCubes(x,y,z: Integer);
begin  // rund 30 Prozent besetzen
  Cubes[x,y,z] := Random(10) > 6;
end;

// Vom Grafiktreiber und -modus unabhngigen Initialisierungen
function TCWForm.OneTimeSceneInit: HResult;
var x,y,z: Integer;
begin
  Randomize; // einmal das gesamte Array besetzen
  for x := 0 to XYZRANGE-1 do
    for y := 0 to XYZRANGE-1 do
      for z := 0 to XYZRANGE-1 do
        PopulateCubes(x,y,z);

  // Mesh fr das Hintergrund-Bitmap
  m_BackgroundMesh[0] := D3DTLVERTEX(D3DVECTOR(0,0,0.99),
        0.5, $ffffffff, 0, 0, 1 );
  m_BackgroundMesh[1] := D3DTLVERTEX(D3DVECTOR(0,0,0.99),
        0.5, $ffffffff, 0, 0, 0 );
  m_BackgroundMesh[2] := D3DTLVERTEX(D3DVECTOR(0,0,0.99),
        0.5, $ffffffff, 0, 1, 1 );
  m_BackgroundMesh[3] := D3DTLVERTEX(D3DVECTOR(0,0,0.99),
        0.5, $ffffffff, 0, 1, 0 );

  // Hintergrund-Bitmap daselbst
  Result := D3DTextr_CreateTextureFromFile(BackBMPName, 0, 0);
  if FAILED(Result) then
  begin
    ShowMessage('Error Loading '+BackBMPName); Exit;
  end;

  // die X-Datei des Wrfels
  m_pCubeFile := TD3DFile.Create;
  Result := m_pCubeFile.Load(CubeXFileName);
  if FAILED(Result) then
  begin
    ShowMessage('Error loading '+CubeXFileName); Exit;
  end;

  m_pCubeFile.Scale(0.1);  // Wrfel verkleinern

  // Vertices und Indices einmal ermitteln - und nicht bei
  // jedem Rendering erneut
  Result := Result or m_pCubeFile.GetMeshVertices('',
     m_pCubeVertices, m_dwCubeNumVertices);
  Result := Result or m_pCubeFile.GetMeshIndices('',
     m_pCubeIndices, m_dwCubeNumIndices);
end;

// Gegenstck zu OneTimeSceneInit
function TCWForm.FinalCleanup: HResult;
begin
  m_pCubeFile.Free;
  Result := S_OK;
end;

// Von Grafiktreiber und -modus abhngige Initialisierungen,
// wird bei jedem Wechsel erneut aufgerufen
function TCWForm.InitDeviceObjects: HResult;
var vp: TD3DViewport7; mtrl: TD3DMaterial7;
    matWorld, matProj: TD3DMatrix;
    fFogStart: TD3DValue;

  procedure AddLight(Index: Integer; X, Y, R,G,B: TD3DValue);
  var Light: TD3DLight7;
  begin   // R, G und W-Licht an der Mitte der Z-Achse
    D3DUtil_InitLight(Light, D3DLIGHT_DIRECTIONAL, X, Y, 0);
    with Light do  // dvcAmbient bleibt auf 0
    begin
       dcvDiffuse.r := R; dcvDiffuse.g := G; dcvDiffuse.b := B;
       dcvSpecular := dcvDiffuse;
    end;
    m_pd3dDevice.SetLight(Index, Light );
    m_pd3dDevice.LightEnable(Index, True);
  end;

begin
  // Hintergrund-Mesh fllt den gesamten Viewport
  m_pd3dDevice.GetViewport(vp);
  m_BackgroundMesh[0].sy := vp.dwHeight;  // Pixel!
  m_BackgroundMesh[2].sy := vp.dwHeight;
  m_BackgroundMesh[2].sx := vp.dwWidth;
  m_BackgroundMesh[3].sx := vp.dwWidth;

  with m_pd3dDevice do
  begin
    // Projektion ist statisch
    D3DUtil_SetProjectionMatrix( matProj, g_PI/4, 1, 1, 60);
    SetTransform( D3DTRANSFORMSTATE_PROJECTION, matProj );
    // Welt und View werden ber Render und FrameMove gesetzt
    D3DUtil_SetIdentityMatrix( matWorld );
    SetTransform( D3DTRANSFORMSTATE_WORLD, matWorld );
    // Kameraposition und -richtung
    vEyePt    := D3DVECTOR( 0, 0, 0);
    vDirection := D3DVECTOR(0, 0, 1);
    vUpVec    := D3DVECTOR( 0, 1,   0  );  // Kopf senkrecht
    // Bewegungsgeschwindigkeit in allen Richtungen 0
    LinSpeed := 0; LRSpeed := 0; UpDownSpeed := 0;
    SetCameraPosition;
    // weie Oberflche fr die Wrfel, Alpha = 1
    D3DUtil_InitMaterial( mtrl, 1.0, 1.0, 1.0, 1.0);
    SetMaterial(mtrl);
    // Texturen - hier nur die Minimalvorgabe
    D3DTextr_RestoreAllTextures(m_pd3dDevice);
    SetRenderState(D3DRENDERSTATE_DITHERENABLE, Ord(TRUE));
    SetRenderState(D3DRENDERSTATE_SPECULARENABLE, Ord(FALSE) );
    // Licht
    if  m_pDeviceInfo.ddDeviceDesc.dwVertexProcessingCaps
      and D3DVTXPCAPS_DIRECTIONALLIGHTS <> 0 then
    begin
        SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(True) );
        SetRenderState(D3DRENDERSTATE_AMBIENT, D3DRGB(0,0,0));

        AddLight(0, -10, 0, 1.0, 0, 0);   // links, rot
        AddLight(1, 10,  0, 0, 1.0, 0);  // rechts, grn
        AddLight(2, 0, 10, 1.0, 1.0, 1.0); // oben, wei
    end
     else
        SetRenderState(D3DRENDERSTATE_AMBIENT, $ffffffff);

    fFogStart := XYZRANGE*0.3; fFogEnd := XYZRANGE * 0.4;

    SetRenderState(D3DRENDERSTATE_FOGCOLOR, 0);  // schwarz
    // das wre Pixelfog
//    SetRenderState(D3DRENDERSTATE_FOGTABLEMODE, Ord(D3DFOG_LINEAR));
    SetRenderState(D3DRENDERSTATE_FOGVERTEXMODE, Ord(D3DFOG_LINEAR));
    SetRenderState(D3DRENDERSTATE_FOGSTART, PDWord(@fFogStart)^);
    SetRenderState(D3DRENDERSTATE_FOGEND, PDWord(@fFogEnd)^);
  end;
  Result := S_OK;
end;

// Gibt die bei InitDeviceObjects angelegten Objekte wieder frei
// Bei jedem Wechsel von Grafiktreiber und -modus aufgerufen
function TCWForm.DeleteDeviceObjects: HResult;
begin
  // DirectDraw-Oberflchen der Texturen. Textur-Bitmaps
  // bleiben davon unberhrt
  D3DTextr_InvalidateAllTextures;
  // Lichtquellen etc. werden beim Abbau der Gerteschnittstelle
  // automatisch beseitigt
  Result := S_OK;
end;

function TCWForm.Render: HResult;
begin
  // Viewport lschen - hier nur den Z-Puffer auf 1.0
  m_pd3dDevice.Clear(0, nil, D3DCLEAR_ZBUFFER, 0, 1.0, 0);

  if SUCCEEDED(m_pd3dDevice.BeginScene) then
  begin
    DrawBackground;
    DrawCubes;
    m_pd3dDevice.EndScene;
  end;
  Result := S_OK;
end;

function TCWForm.DrawBackground: HRESULT;
begin // XRef: Render. Zeichnet das Hintergrund-Bitmap (Sterne)
  with m_pd3dDevice do
  begin
    SetTexture(0, D3DTextr_GetSurface(BackBMPName));  // Textur
    // keine Beleuchtung, keine Z-Ebene
    SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(False) );
    SetRenderState(D3DRENDERSTATE_ZENABLE, Ord(False));
    // sonst verschwindet das Hintergrund-Bitmap im Nebel
    SetRenderState(D3DRENDERSTATE_FOGENABLE, Ord(False));
    SetRenderState(D3DRENDERSTATE_TEXTUREPERSPECTIVE, Ord(False));
    // D3DFVF_*TL*VERTEX = fertig berechnete Pixelpositionen
    Result := DrawPrimitive(D3DPT_TRIANGLESTRIP,
       D3DFVF_TLVERTEX, m_BackgroundMesh, 4, 0 );
    SetTexture(0, nil);  // sonst kme die auch bei den Wrfeln!
  end;
end;

// XRef: Render. Zeichnet die Wrfel. Material und Licht in
// InitDeviceObjects gesetzt, keine Textur
function TCWForm.DrawCubes: HRESULT;
var matWorld: TD3DMatrix; x,y,z: Integer;
    AngleX, AngleXY: TD3DValue;
    DistX2, DistY2: TD3DValue;
begin
  D3DUtil_SetIdentityMatrix(matWorld);
  // Wrfel verwenden im Gegensatz um Hintergrund-Bitmap den
  // Z-Puffer, Licht und Fogging
  m_pd3dDevice.SetRenderState(D3DRENDERSTATE_ZENABLE, Ord(True));
  m_pd3dDevice.SetRenderState(D3DRENDERSTATE_FOGENABLE, Ord(True));
  m_pd3dDevice.SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(True) );

  with m_pD3DDevice do
    for x := 0 to XYZRANGE-1 do
    begin
       matWorld._41 := x - XYZRANGE div 2;
       DistX2 := Sqr(matWorld._41-vEyePt.x);  // Camera Distance
       AngleX := vDirection.x * (matWorld._41 - vEyePt.x);
       for y := 0 to XYZRANGE-1 do
       begin
         matWorld._42 := y - XYZRANGE div 2;
         DistY2 := Sqr(matWorld._42-vEyePt.y);
         AngleXY := AngleX + vDirection.y * (matWorld._42 - vEyePt.y);
         for z := 0 to XYZRANGE-1 do
          if Cubes[x,y,z] then
          begin // Rasterposition besetzt: Wrfel ber matWorld
            // positionieren. x und y stehen bereits
            matWorld._43 := z - XYZRANGE div 2;
            // einfacher 90 Grad-Test und Entfernungsprfung
            // if vDirection.x*(CubePt.x-vEyePt.x) +
            //  + vDirection.y*(CubePt.y-vEyePt.y) +
            //  + vDirection.z*(Pt.z-vEyePt.z) > 0 then
            //    if VectorMagnitude(VectorSub(CubePt, vEyePt))
            //        <= fFogEnd then ...
            if (AngleXY+vDirection.z*(matWorld._43-vEyePt.z) >= 0)
              and (Sqrt(DistX2+DistY2+
                   Sqr(matWorld._43-vEyePt.z)) <= fFogEnd) then
            begin
              SetTransform(D3DTRANSFORMSTATE_WORLD, matWorld);

// Sichtbarkeitstest mit dem Frustrum, SphereRadius auf ca. 0.1
// setzen, SphereVec auf (0,0,0), - matWorld bereits gesetzt(!)
// ComputeSphereVisibility(SphereVec, SphereRadius, 1, 0, SRes);
// if SRes and D3DVIS_OUTSIDE_FRUSTUM = 0 then <Wrfel zeichnen>

              m_pd3dDevice.DrawIndexedPrimitive(D3DPT_TRIANGLELIST,
                D3DFVF_VERTEX,
                m_pCubeVertices^, m_dwCubeNumVertices,
                m_pCubeIndices^, m_dwCubeNumIndices, 0);

// Im Prinzip dasselbe, kostet aber doppelt so viel Rechenzeit
// wg. linearer Suche in den Dateistrukturen, und weil das gute
// Stck jedesmal sein Material erneut einsetzt
                // m_pCubeFile.Render(m_pd3dDevice);
            end;  // if AngleXY
          end; // if Cubes[x,y,z]
       end;  // for y
    end;  // for x

  // Aufrumen ist eigentlich unntig
  D3DUtil_SetIdentityMatrix(matWorld);
  m_pd3dDevice.SetTransform(D3DTRANSFORMSTATE_WORLD, matWorld);

  Result := S_OK;
end;

procedure TCWForm.SetCameraPosition;
var vLookatPt: TD3DVector;
begin // XRef: InitDeviceObjects, FrameMove
  vEyePt := VectorAdd(vEyePt, VectorMulS(vDirection, LinSpeed));
  vLookatPt := VectorAdd(vEyePt, VectorMulS(vDirection,20));
  SetViewParams( @vEyePt, @vLookAtPt, @vUpVec, 0.1);
end;

procedure RotateVector(var RotVector: TD3DVector;
     Axis: TD3DVector; Angle: TD3DValue);
var matRotate, matVec: TD3DMatrix;
begin
  D3DUtil_SetRotationMatrix(matRotate, Axis, Angle);
  D3DUtil_SetIdentityMatrix(matVec);
  matVec._14 := RotVector.x;
  matVec._24 := RotVector.y;
  matVec._34 := RotVector.z;
  D3DMath_MatrixMultiply(matVec, matRotate, matVec);
  RotVector.x := matVec._14;
  RotVector.y := matVec._24;
  RotVector.z := matVec._34;
end;

// Tastatur-Schnittstelle und Fortschreibung der Bewegung
function TCWForm.FrameMove(fTimeKey: FLOAT): HResult;
const AngDelta = 0.001; AngMax = 0.1;
      LinDelta = 0.01; LinMax = 1;
var tu, tv: FLOAT;
    Data: TDIKeyboardState;  // Tastaturabfrage

  function KeyDown(Index: Integer): Boolean;
  begin
    Result := Data[Index] and $80 <> 0;
  end;

begin
  if SUCCEEDED(DIKeyboard.Acquire) and SUCCEEDED(
    DIKeyboard.DIObject.GetDeviceState(SizeOf(Data),@Data)) then
  begin
    if KeyDown(DIK_H) then
      ShowMessage('CubeWorld - simple interactive D3DIM Demo '+
         'as (Arne Schpers) JUN-2000'#13#10+
         'T = forward, G = stop, V = backward, S = reset; '+
         'arrow keys to change direction');

    if KeyDown(DIK_LEFT) or KeyDown(DIK_NUMPAD4) then
    begin  // Linkspfeiltaste
      LRSpeed := LRSpeed - AngDelta;
      if LRSpeed < -AngMax then LRSpeed := -AngMax;
    end;   // Rechtspfeiltaste
    if KeyDown(DIK_RIGHT) or Keydown(DIK_NUMPAD6) then
    begin
      LRSpeed := LRSpeed + AngDelta;
      if LRSpeed > AngMax then LRSpeed := AngMax;
    end;
    if KeyDown(DIK_UP) or KeyDown(DIK_NUMPAD8) then
    begin  // Nose up
      UpDownSpeed := UpDownSpeed + AngDelta;
      if UpDownSpeed > AngMax then UpdownSpeed := AngMax;
    end;
    if KeyDown(DIK_DOWN) or KeyDown(DIK_NUMPAD2) then
    begin // Nose down
      UpdownSpeed := UpDownSpeed - AngDelta;
      if UpDownSpeed < -AngMax then UpDownSpeed := -AngMax;
    end;
    if KeyDown(DIK_S) then  // Reset
    begin
      LinSpeed := 0; UpdownSpeed := 0; LRSpeed := 0;
      vUpVec := D3DVector(0,1,0);
      vDirection := D3DVector(0,0,1);
      vEyePt := D3DVector(0,0,0);
    end;
    if KeyDown(DIK_T) then   // Speed up
    begin
      LinSpeed := LinSpeed + LinDelta;
      if LinSpeed > LinMax then LinSpeed := LinMax;
    end;
    if KeyDown(DIK_G) then  // Stop
    begin
      LinSpeed := 0;
    end;
    if KeyDown(DIK_V) then  // Brake
    begin
      LinSpeed := LinSpeed - LinDelta;
      if LinSpeed < -LinMax then LinSpeed := -LinMax;
    end;
  end;

  // Rotation um Up (senkrecht zu Direction)
  RotateVector(vDirection, vUpVec, LRSpeed);
  // Rotation um die Achse quer zur Blickrichtung
  RotateVector(vUpVec, VectorCrossProduct(vUpVec,vDirection),
     UpDownSpeed);
  RotateVector(vDirection, VectorCrossProduct(vUpVec,vDirection),
     UpDownSpeed);

  SetCameraPosition;
  LinSpeed := LinSpeed * 0.9;
  LRSpeed := LRSpeed * 0.9; UpDownSpeed := UpDownSpeed * 0.9;

  // Koordinaten der Hintergrund-Textur: horizontaler Winkel,
  // umgesetzt auf den Bereich von -1..+1. Bei Bewegungen nach
  // rechts rollt die Textur nach links, deshalb negatives Vorz.
  tu := -arctan2(vDirection.x, vDirection.z) / Pi;
  tu := tu * 4;   // schlicht ausprobiert
  m_BackgroundMesh[0].tu := tu; m_BackgroundMesh[1].tu := tu;
  m_BackgroundMesh[2].tu := tu - TEXREPEAT;  // n-fache Wdh.
  m_BackgroundMesh[3].tu := tu - TEXREPEAT;

  // dito fr den vertikalen Winkel
  tv := -4 * arctan2(vDirection.y, vDirection.z) / Pi;
  m_BackgroundMesh[0].tv := tv;
  m_BackgroundMesh[1].tv := tv - TEXREPEAT;
  m_BackgroundMesh[2].tv := tv;
  m_BackgroundMesh[3].tv := tv - TEXREPEAT;
  Result := S_OK;
end;

end.
