/* lpr.c $Id: lpr.c,v 1.2 1999/10/24 05:45:30 angus Exp $
 *
 * Copyright (C) Angus J. C. Duggan, 1996-1998
 *
 * LPR client for NT; uses registry and PRINTER environment variable for
 * printer, accepts piped input. PRINTERHOST registry key and environment
 * variable used for LPD host.
 * LPQ and LPRM also provided.
 */

#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <fcntl.h>
#include <io.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "common.h"

char *program ;

static void usage(void)
{
  error("Copyright (C) Angus J. C. Duggan, 1996-1999. See file LICENSE for details.\nUsage:\n  %s [-Pprinter] [-Hhost] [-ps|-dvi|-text|-page|-ctrl|-troff|-ditroff]\n   [-Iindent] [-Wwidth] [-Ttitle] [-Jjobname] [-Cclass] [-M] [-Zoptions] [file...]\n", program) ;
}

/* Make a file seekable, using temporary files if necessary */
FILE *seekable(FILE *fp)
{
  FILE *ft ;
  long r, w ;
  char *p, buffer[BUFSIZ] ;
  struct _stat fs ;

  if ( _fstat(fileno(fp), &fs) == 0 && (fs.st_mode & _S_IFREG) != 0 )
    return fp ;

  if ( (ft = tmpfile()) == NULL )
    return NULL ;

  while ( (r = fread(p = buffer, sizeof(char), BUFSIZ, fp)) > 0 ) {
    do {
      if ( (w = fwrite(p, sizeof(char), r, ft)) == 0 )
	return NULL ;
      p += w ;
      r -= w ;
    } while ( r > 0 ) ;
  }

  if ( !feof(fp) )
    return NULL ;

  /* discard the input file, and rewind the temporary */
  (void)fclose(fp) ;
  if ( fseek(ft, 0L, SEEK_SET) != 0 )
    return NULL ;

  return ft ;
}

int PrintFile(char *printer, char *hostname, SOCKET printsock, va_list args)
{
  FILE *infile = va_arg(args, FILE *) ;
  char *filename = va_arg(args, char *) ;
  char *user = va_arg(args, char *) ;
  int format = va_arg(args, int) ;
  char *jobname = va_arg(args, char *) ;
  char *title = va_arg(args, char *) ;
  int width = va_arg(args, int) ;
  int indent = va_arg(args, int) ;
  int mail = va_arg(args, int) ;
  char *class = va_arg(args, char *) ;
  char *zopts = va_arg(args, char *) ;
  long bytesread, totalbytes = 0 ;
  long cflen, dflen = 0 ;
  int jobno = NextJobNumber() ;
  int result ;
  char buffer[BUFSIZ] ;
  char localhost[BUFSIZ] ;

  if ( jobname == NULL )
    jobname = filename ;

  if ( title == NULL )
    title = jobname ;

  if ( (infile = seekable(infile)) == NULL ) {
    warning("%s: can't make file %s seekable\n", program, filename) ;
    return 1 ;
  }

  if ( SOCKET_ERROR == gethostname(localhost, BUFSIZ) ) {
    warning("%s: can't get own host name, error %d\n", program,
	    WSAGetLastError()) ;
    return 1 ;
  }

  /* Get length of data file */
  {
    struct _stat fs ;

    if ( _fstat(fileno(infile), &fs) == 0 && (fs.st_mode & _S_IFREG) != 0 )
      dflen = fs.st_size ;
  }

  /* Truncate host name at first dot */
  {
    char *dot = strchr(localhost, '.') ;

    if ( dot )
      *dot = '\0' ;
  }

  /* Send print command */
  if ( !Command("%c%s\n", LPD_PRINT, printer) || !Acknowledge(0) )
    return 1 ;

  /* Create control file in static buffer */
  {
    char *cf = buffer ;

    sprintf(cf, "H%s\nP%s\nJ%s\n", localhost, user, jobname) ;
    cf += strlen(cf) ;

    if ( class ) {
      sprintf(cf, "C%s\n", class) ;
      cf += strlen(cf) ;
    }

    if ( zopts ) { /* LPRng and PLP Z options */
      sprintf(cf, "Z%s\n", zopts) ;
      cf += strlen(cf) ;
    }

    /* Banner page printed if 'L' here */

    switch ( format ) {
    case FMT_PAGE:
      sprintf(cf, "T%s\n", title) ;
      cf += strlen(cf) ;
    case FMT_TEXT:
      sprintf(cf, "I%d\n", indent) ;
      cf += strlen(cf) ;
    case FMT_CTRL:
      sprintf(cf, "W%d\n", width) ;
      cf += strlen(cf) ;
    }

    if ( mail ) {
      sprintf(cf, "M%s\n", user) ;
      cf += strlen(cf) ;
    }

    sprintf(cf, "%cdfA%03d%s\nUdfA%03d%s\nN%s\n", format, jobno, localhost,
	    jobno, localhost, filename) ;

    cflen = strlen(buffer) ;
  }

  /* Send control file */
  if ( !Command("%c%d cfA%03d%s\n", LPR_CONTROL, cflen, jobno, localhost) ||
       !Acknowledge(0) )
    return 1 ;

  {
    int totalsent = 0 ;
    do {
      int bytessent ;

      if ( SOCKET_ERROR == (bytessent = send(printsock, buffer + totalsent, cflen - totalsent, 0)) ) {
	warning("%s: send of control file to %s failed at byte %d, error %d\n",
		program, hostname, totalsent, WSAGetLastError()) ;
	return 1 ;
      }
      totalsent += bytessent ;
    } while ( totalsent < cflen ) ;
  }

  if ( !Acknowledge(1) ) /* Acknowledge control file transmission */
    return 1 ;

  /* Send data file */
  if ( !Command("%c%d dfA%03d%s\n", LPR_DATA, dflen, jobno, localhost) ||
       !Acknowledge(0) )
    return 1 ;

  do {
    bytesread = fread(buffer, sizeof(char), BUFSIZ, infile) ;
    totalbytes += bytesread ;

    if ( bytesread > 0 ) {
      int totalsent = 0 ;

      do {
	int bytessent ;

	if ( (bytessent = send(printsock, buffer + totalsent, bytesread - totalsent, 0)) == SOCKET_ERROR ) {
	  warning("%s: send to %s failed after %d bytes, error %d\n", program,
		  hostname, totalbytes - bytesread + bytessent, WSAGetLastError()) ;
	  return 1 ;
	}
	totalsent += bytessent ;
      } while ( totalsent < bytesread ) ;
    }
  } while ( bytesread == BUFSIZ ) ;

  if ( dflen == 0 ) {
    struct { 
      int l_onoff; 
      int l_linger; 
    } linger ;
    linger.l_onoff = 1 ;	/* wait to send data */
    linger.l_onoff = 10 ;	/* seconds timeout */
    if ( setsockopt(printsock, SOL_SOCKET, SO_LINGER, (char *)&linger,
		    sizeof(linger)) != 0 ) {
      warning("%s: set socket options failed, error %d\n", program,
	      WSAGetLastError()) ;
      return 1 ;
    }
  } else {
    if ( dflen != totalbytes )
      warning("%s: file length %d bytes, but read %d bytes\n", program,
	      dflen, totalbytes) ;
    if ( !Acknowledge(1) ) /* Acknowledge data file transmission */
      return 1 ;
  }

  if ( (result = ferror(infile)) != 0 ) {
    warning("%s: read from file failed after %d bytes, error %d\n", program,
	    filename, totalbytes, result) ;
    return 1 ;
  }

  return 0 ;	/* Success! */
}

int main(int argc, char *argv[])
{
  FILE *infile = stdin ;
  char *printer = getenv(PRINTER) ;
  char *host = DefaultHost(printer) ;
  char *user = getenv("USERNAME") ;
  int result = 0 ;
  int format = FMT_TEXT ; /* Text, will be interpreted by daemon */
  char *jobname = NULL ;
  char *title = NULL ;
  char *class = NULL ;
  char *zopts = NULL ;
  int width = 0, indent = 0, mail = 0 ;

  if ( user == NULL )
    user = getenv("USER") ;

  if ( user == NULL )
    user = DEFAULTUSER ;

  for (program = *argv++ ; --argc ; argv++) {
    if (argv[0][0] == '-') {
      switch (argv[0][1]) {
      case 'P':	/* printer */
	printer = &(argv[0][2]) ;
        host = DefaultHost(printer) ;
	break;
      case 'H': case 'S':
        host = &(argv[0][2]) ;
        break ;
      case 'p':
	if ( argv[0][2] == '\0' || strcmp(*argv, "-ps") == 0 )
	  format = FMT_PS ;
	else if ( strcmp(*argv, "-page") == 0 )
	  format = FMT_PAGE ;
	else
	  usage() ;
	break ;
      case 'd':
	if ( argv[0][2] == '\0' || strcmp(*argv, "-dvi") == 0 )
	  format = FMT_DVI ;
	else if ( strcmp(*argv, "-ditroff") == 0 )
	  format = FMT_DITROFF ;
	else
	  usage() ;
	break ;
      case 't':
	if ( argv[0][2] == '\0' || strcmp(*argv, "-text") == 0 )
	  format = FMT_TEXT ;
	else if ( strcmp(*argv, "-troff") == 0 )
	  format = FMT_TROFF ;
	else
	  usage() ;
	break ;
      case 'c':
	if ( argv[0][2] != '\0' && strcmp(*argv, "-ctrl") != 0 )
	  usage() ;
	format = FMT_CTRL ;
	break ;
      case 'I':
	indent = atoi(&(argv[0][2])) ;
	break ;
      case 'W':
	width = atoi(&(argv[0][2])) ;
	break ;
      case 'C':
	class = &(argv[0][2]) ;
	break ;
      case 'J':
	jobname = &(argv[0][2]) ;
	break ;
      case 'T':
	title = &(argv[0][2]) ;
	break ;
      case 'M':
	mail = !mail ;
	break ;
      case 'Z':
        if ( zopts == NULL ) {
          zopts = strdup(&(argv[0][2])) ;
        } else {
          zopts = realloc(zopts, strlen(zopts) + strlen(&(argv[0][2]) + 1)) ;
          strcat(zopts, &(argv[0][2])) ;
        }
      default:
	usage();
      }
    } else if ((infile = fopen(*argv, "rb")) != NULL) {
      result = LPD(printer, host, &PrintFile, infile, *argv, user, format,
		   jobname, title, width, indent, mail, class, zopts) ;
      fclose(infile) ;

      if ( result )
	warning("%s: problems printing file %s, continuing\n", program,
		*argv) ;
    } else
      warning("%s: can't open input file %s, skipping\n", program, *argv);
  }

  if ( infile == stdin ) {
    int fd = fileno(stdin) ;
    if ( setmode(fd, O_BINARY) < 0 )
      error("%s: can't reset stdin to binary mode\n", program) ;
    result = LPD(printer, host, &PrintFile, infile, "(stdin)", user, format,
		 jobname, title, width, indent, mail, class, zopts) ;
  }

  if ( zopts )
    free(zopts) ;
  return result ;
}

/*
 * $Log: lpr.c,v $
 * Revision 1.2  1999/10/24 05:45:30  angus
 *
 * Add support for -Z options
 * Allow printer@host syntax for all programs
 * Use SO_DONTLINGER to enable socket re-use
 *
 * Revision 1.1.1.1  1999/07/02 06:21:11  angus
 * Initial CVS import
 *
 */
