/*************************************************************************
 *
 * Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
 *
 * This file is part of Qt Jambi.
 *
 * ** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.

 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *************************************************************************/

package com.trolltech.tools.designer;

import com.trolltech.qt.*;
import com.trolltech.qt.core.*;

import java.lang.reflect.*;
import java.util.*;

public class MemberSheet extends JambiMemberSheet {

    private static abstract class Entry {
        public abstract String name();
        public abstract String signature();
        public abstract String group();
        public abstract boolean isWidget();
        public boolean visible() { return visible; }
        public void setVisible(boolean b) { visible = b; }

        private boolean visible = true;
    }

    private static class SignalEntry extends Entry {
        Field signal;

        public String name() {
            return signal.getName();
        }

        public String signature() {
            StringBuilder s = new StringBuilder();
            s.append(signal.getName());
            Type type = signal.getGenericType();

            if (type instanceof ParameterizedType) {
                ParameterizedType pt = (ParameterizedType) type;
                Type types[] = pt.getActualTypeArguments();

                if (types.length > 0)
                    s.append("<");

                for (int i = 0; i < types.length; ++i) {
                    Class signal_type = (Class) types[i];
                    if (i != 0)
                        s.append(",");
                    s.append(signal_type.getName());
                }

                if (types.length > 0)
                    s.append(">");
            }


            return s.toString();
        }

        public String group() {
            return signal.getDeclaringClass().getName();
        }

        public boolean isWidget() { return signal.getDeclaringClass() == com.trolltech.qt.gui.QWidget.class; }
    }

    private static class SlotEntry extends Entry {
        Method method;

        public String group() { return method.getDeclaringClass().getName(); }
        public String name() { return method.getName(); }
        public String signature() {
            StringBuilder s = new StringBuilder();
            s.append(method.getName());
            s.append("(");

            Class cl[] = method.getParameterTypes();
            for (int i=0; i<cl.length; ++i) {
                if (i != 0)
                    s.append(",");
                s.append(cl[i].getName());
            }

            s.append(")");

            return s.toString();
        }

        public boolean isWidget() { return method.getDeclaringClass() == com.trolltech.qt.gui.QWidget.class; }
    }

    public MemberSheet(QObject object, QObject parent) {
        super(parent);
        this.object = object;
        build();
    }


    public int count() {
        return entries.size();
    }

    public String declaredInClass(int i) {
        if (i >= entries.size())
            return null;

        return entries.get(i).group();
    }

    public int indexOf(String name) {
        return 0;
    }

    public boolean inheritedFromWidget(int i) {
        if (i >= entries.size())
            return false;
        return entries.get(i).isWidget();
    }

    public boolean isSignal(int i) {
        if (i >= entries.size())
            return false;

        return !(entries.get(i) instanceof SlotEntry);
    }

    public boolean isSlot(int i) {
        if (i >= entries.size())
            return false;

        return entries.get(i) instanceof SlotEntry;
    }

    public boolean isVisible(int i) {
        if (i >= entries.size())
            return false;
        return entries.get(i).visible();
    }

    public String memberGroup(int i) {
        if (i >= entries.size())
            return null;
        return entries.get(i).group();
    }

    public String memberName(int i) {
        if (i >= entries.size())
            return null;
        return entries.get(i).name();
    }

    public List<QByteArray> parameterNames(int i) {
        return null;
    }

    public List<QByteArray> parameterTypes(int i) {
        return null;
    }

    public void setMemberGroup(int i, String arg__2) {
    }

    public void setVisible(int i, boolean arg) {
        if (i >= entries.size()) {
            entries.get(i).setVisible(arg);
        }
    }

    public String signature(int index) {
        return entries.get(index).signature();
    }

    public static boolean signalMatchesSlot(String signal, String slot) {
        if (signal.equals("<signal>") || slot.equals("<slot>")
            || signal.length() == 0 || slot.length() == 0)
            return true;

        // void slots always match...
        if (slot.contains("()"))
            return true;

        int signalIndex = signal.indexOf('<');

        // Match only if () slot which is covered above already...
        if (signalIndex < 0)
            return false;

        int slotIndex = slot.indexOf('(');
        if (slotIndex < 0) {
            throw new IllegalArgumentException("slot doesn't contain () as expected, '"
                                               + slot + "'");
        }

        String signalArguments[] = signal.substring(signalIndex + 1, signal.length() - 1).split(",");
        String slotArguments[] = slot.substring(slotIndex + 1, slot.length() - 1).split(",");

        if (slotArguments.length > signalArguments.length)
            return false;

        for (int i=0; i<slotArguments.length; ++i) {
            if (!matchTypes(signalArguments[i], slotArguments[i]))
                return false;
        }

        return true;
    }

    private static boolean matchTypes(String a, String b) {
        return (a.equals(b) || (typeMap.get(a) != null && typeMap.get(a).equals(b)));
    }

    private void build(Class cl, List<Entry> entries) {
        buildSlots(cl, entries);
        buildSignals(cl, entries);
    }

    private static boolean shouldReject(Method m) {
        if (m.isAnnotationPresent(QtBlockedSlot.class)) return true;

        int mods = m.getModifiers();
        if (Modifier.isStatic(mods)) return true;

        if (blockedSlotClasses.contains(m.getDeclaringClass())) return true;

        String n = m.getName();
        if (n.startsWith("__qt_")) return true;
        if (n.endsWith("Event")) return true;
        if (n.equals("event") || n.equals("eventFilter")) return true;

        return false;
    }

    private void buildSlots(Class cl, List<Entry> entries) {
        Method methods[] = cl.getMethods();
        for (Method m : methods) {
            if (shouldReject(m))
                continue;
            SlotEntry e = new SlotEntry();
            e.method = m;
            entries.add(e);
        }
    }

    private static boolean shouldReject(Field f) {
        return !AbstractSignal.class.isAssignableFrom(f.getType());
    }

    private void buildSignals(Class cl, List<Entry> entries) {
        Field fields[] = cl.getFields();
        for (Field f : fields) {
            if (shouldReject(f))
                continue;

            SignalEntry e = new SignalEntry();
            e.signal = f;
            entries.add(e);
        }
    }

    private void build() {
        Class cl = object.getClass();
        List<Entry> entries = new ArrayList<Entry>();
        build(cl, entries);

        this.entries = entries;
    }

    private List<Entry> entries;
    private QObject object;

    private static HashMap<String, String> typeMap;
    private static HashSet<Class> blockedSlotClasses;
    static {
        typeMap = new HashMap<String, String>();
        typeMap.put("java.lang.Boolean", "boolean");
        typeMap.put("java.lang.Byte", "byte");
        typeMap.put("java.lang.Char", "char");
        typeMap.put("java.lang.Short", "short");
        typeMap.put("java.lang.Integer", "int");
        typeMap.put("java.lang.Long", "long");
        typeMap.put("java.lang.Float", "float");
        typeMap.put("java.lang.Double", "double");

        blockedSlotClasses = new HashSet<Class>();
        blockedSlotClasses.add(QtJambiObject.class);
        blockedSlotClasses.add(QSignalEmitter.class);
        blockedSlotClasses.add(Object.class);
    }

}
