/*  This file is part of the KDE project.

    Copyright (C) 2007 Trolltech ASA. All rights reserved.

    This library is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 2.1 or 3 of the License.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "quicktimestreamreader.h"

QT_BEGIN_NAMESPACE

namespace Phonon
{
namespace QT7
{

QuickTimeStreamReader::QuickTimeStreamReader(const Phonon::MediaSource &source)
{
    m_client = 0;
    connect(&m_server, SIGNAL(newConnection()), this, SLOT(newNetworkConnection()));
    m_server.listen();
    connectToSource(source);
}

QuickTimeStreamReader::~QuickTimeStreamReader()
{
}

void QuickTimeStreamReader::newNetworkConnection()
{
    delete m_client;
    m_client = m_server.nextPendingConnection();
	connect(m_client, SIGNAL(readyRead()), this, SLOT(readNetworkClient()));
}

void QuickTimeStreamReader::readNetworkClient()
{
    QDataStream stream(m_client);
    // QString msg = m_client->readLine();
    // while (!msg.isEmpty()){
    //     qDebug() << msg;
    //     msg = m_client->readLine();
    // }
    
    QString header("HTTP/1.1 200 OK\r\n");
    // header += "Server: Phonon\r\n";
    header += "Connection: Keep-Alive\r\n";
    header += "Content-Length: " + QString::number(m_size) + "\r\n";
    header += "Content-Type: audio/x-mp3\r\n";
    header += "\r\n\r\n";
    stream.writeBytes(header.toAscii().data(), header.toAscii().length());
    
    int size = 1000000;
    char *data = new char[size];
    
    for (int offset=0; offset<m_size; offset += size){
        int read = readData(offset, size, data);
        stream.writeBytes(data, read);
        m_client->flush();
    }
    delete[] data;
}

QUrl QuickTimeStreamReader::mediaUrl()
{
    QUrl url;
    url.setScheme("http");
    url.setHost("localhost");
    url.setPort(m_server.serverPort());
    return url;
}

int QuickTimeStreamReader::readData(long offset, long size, void *data)
{
//    QReadLocker readLocker(&m_lock);
    if (streamSize() != 1 && offset + size > streamSize()){
        size = streamSize() - offset;
    }

    if (currentPos() - currentBufferSize() != offset)
        setCurrentPos(offset);

    int oldSize = currentBufferSize();
    while (currentBufferSize() < int(size)) {
        needData();
        if (oldSize == currentBufferSize())
            break;
        oldSize = currentBufferSize();
    }
    
    int bytesRead = qMin(currentBufferSize(), int(size));
//    QWriteLocker writeLocker(&m_lock);
    qMemCopy(data, m_buffer.data(), bytesRead);
    m_buffer = m_buffer.mid(bytesRead);

    return bytesRead;
}
    
void QuickTimeStreamReader::writeData(const QByteArray &data)
{
    QWriteLocker locker(&m_lock);
    m_pos += data.size();
    m_buffer += data;
}

void QuickTimeStreamReader::endOfData()
{
}

void QuickTimeStreamReader::setStreamSize(qint64 newSize)
{
    m_size = newSize;
}

qint64 QuickTimeStreamReader::streamSize() const
{
    return m_size;
}

void QuickTimeStreamReader::setStreamSeekable(bool s)
{
    m_seekable = s;
}

bool QuickTimeStreamReader::streamSeekable() const
{
    return m_seekable;
}

void QuickTimeStreamReader::setCurrentPos(qint64 pos)
{
    QWriteLocker locker(&m_lock);
    m_pos = pos;
    seekStream(pos);
    m_buffer.clear();
}

qint64 QuickTimeStreamReader::currentPos() const
{
    return m_pos;
}

int QuickTimeStreamReader::currentBufferSize() const
{
    QReadLocker locker(&m_lock);
    return m_buffer.size();
}

}} //namespace Phonon::QT7

QT_END_NAMESPACE

#include "moc_quicktimestreamreader.cpp"

