/****************************************************************************
**
** Copyright (C) 2006-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the Patternist project on Trolltech Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <QtCore/QDir>
#include <QtCore/QtDebug>
#include <QtCore/QFile>
#include <QtCore/QFileInfo>
#include <QtCore/QStringList>
#include <QtCore/QTextCodec>
#include <QtCore/QTextStream>
#include <QtCore/QUrl>
#include <QtCore/QVariant>
#include <QtCore/QVector>

#include <QtXmlPatterns/QXmlFormatter>
#include <QtXmlPatterns/QXmlItem>
#include <QtXmlPatterns/QXmlQuery>
#include <QtXmlPatterns/QXmlSerializer>

#include "qapplicationargument_p.h"
#include "qapplicationargumentparser_p.h"
#include "qcoloringmessagehandler_p.h"

#include "main.h"

QT_USE_NAMESPACE

/*!
 \internal
 Represents the name and value found in "-param name=value".
 */
typedef QPair<QString, QString> Parameter;

/*!
 \class PatternistApplicationParser
 \brief Subclass to handle -param name=value
 \internal
 \since 4.4
 \reentrant
 */
class PatternistApplicationParser : public QApplicationArgumentParser
{
public:
    inline PatternistApplicationParser(int argc, char **argv) : QApplicationArgumentParser(argc, argv)
    {
    }
protected:
    virtual QVariant convertToValue(const QApplicationArgument &arg,
                                    const QString &input) const
    {
        if(arg.name() == QLatin1String("param"))
        {
            const int assign = input.indexOf(QLatin1Char('='));

            if(assign == -1)
            {
                // TODO need error reporting function QTextStream(stderr) << PatternistCLI::tr("Each binding must contain an equal sign.");
                return QVariant();
            }

            const QString name(input.left(assign));
            const QString value(input.mid(assign + 1));

            /* TODO check that name is valid. */

            return qVariantFromValue(Parameter(name, value));
        }
        else
            return QApplicationArgumentParser::convertToValue(arg, input);
    }

    virtual QString typeToName(const QApplicationArgument &argument) const
    {
        if(argument.name() == QLatin1String("param"))
            return QLatin1String("name=value");
        else
            return QApplicationArgumentParser::typeToName(argument);
    }
};

Q_DECLARE_METATYPE(Parameter);

int main(int argc, char **argv)
{
    enum ExitCode
    {
        QueryFailure = 1,
        QueryInexistent,
        OpenFailure,
        StdOutFailure
    };

    const QCoreApplication app(argc, argv);
    QCoreApplication::setApplicationName(QLatin1String("patternist"));

    PatternistApplicationParser parser(argc, argv);
    parser.setApplicationDescription(QLatin1String("A tool for running XQuery queries."));
    parser.setApplicationVersion(QLatin1String("0.1"));

    /* Is there a better way to do this? */
    const int parameterType = qVariantFromValue(Parameter()).userType();

    QApplicationArgument param(QLatin1String("param"),
                               PatternistCLI::tr("Binds an external variable. Declare it first in the query with 'declare variable $name external;'."),
                               parameterType);
    parser.addArgument(param);

    QApplicationArgument format(QLatin1String("format"),
                                PatternistCLI::tr("Whether the XML output should be indented and formatted for readability."));
    parser.addArgument(format);

    QApplicationArgument queryURI(QString(), /* Nameless. */
                                  PatternistCLI::tr("A URI pointing to the query to run. If the name ends with .xq it's assumed "
                                                    "to be an XQuery query. (In other cases too, but that interpretation may change in a future release of Qt.)"),
                                  QVariant::Url);
    queryURI.setMinimumOccurrence(1);
    parser.addArgument(queryURI);

    QApplicationArgument output(QLatin1String("output"),
                                PatternistCLI::tr("A local file to which the output should be written. The file is overwritten, or if not exist, created. If absent, stdout is used."),
                                QVariant::String);
    parser.addArgument(output);
                                
    if(!parser.parse())
        return parser.exitCode();

    QXmlQuery query;

    QFile myStdout;

    if(!myStdout.open(stdout, QIODevice::WriteOnly))
    {
        QTextStream(stderr) << PatternistCLI::tr("Failed to open standard out.") << endl;
        return StdOutFailure;
    }

    /* Bind external variables. */
    {
        const QVariantList parameters(parser.values(param));
        const int len = parameters.count();

        for(int i = 0; i < len; ++i)
        {
            const Parameter p(qVariantValue<Parameter>(parameters.at(i)));
            query.bindVariable(p.first, QXmlItem(p.second));
        }
    }

    /* The final preparations and execute the query. */
    ColoringMessageHandler messageHandler;
    query.setMessageHandler(&messageHandler);

    const QUrl userURI(parser.value(queryURI).toUrl());
    const QUrl effectiveURI(QUrl::fromLocalFile(QDir::current().absolutePath() + QLatin1Char('/')).resolved(userURI));

    Q_ASSERT_X(userURI.isValid(), Q_FUNC_INFO,
               "QApplicationArgumentParser should promise us this.");

    query.setQuery(effectiveURI);

    if(query.isValid())
    {
        QAbstractXmlReceiverPointer receiver;

        if(parser.has(format))
            receiver = QAbstractXmlReceiverPointer(new QXmlFormatter(query, &myStdout));
        else
            receiver = QAbstractXmlReceiverPointer(new QXmlSerializer(query, &myStdout));

        if(query.evaluateToReceiver(receiver))
            return parser.exitCode();
        else
            return QueryFailure;
    }
    else
        return QueryFailure;
}

