/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the Qt Linguist of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

/*  TRANSLATOR MsgEdit

  This is the right panel of the main window.
*/

#include "msgedit.h"
#include "trwindow.h"
#include "simtexth.h"
#include "messagemodel.h"
#include "phrasemodel.h"
#include "languagesmanager.h"

#include <QMenu>
#include <QAction>
#include <QApplication>
#include <QClipboard>
#include <QLabel>
#include <QLayout>
#include <QTextEdit>
#include <QPalette>
#include <QString>
#include <QPainter>
#include <QHeaderView>
#include <QDockWidget>
#include <QFont>
#include <QTreeView>
#include <QScrollArea>
#include <QtGui/QTextDocumentFragment>
#include <QtGui/QTextCursor>
#include <QtGui/QTextBlock>
#include <QtGui/QTextFragment>
#include <QtGui/QTextImageFormat>
#include <QtGui/QImage>
#include <QtCore/QUrl>
#include <QAbstractTextDocumentLayout>

QT_BEGIN_NAMESPACE

static const int MaxCandidates = 5;

const char MessageEditor::backTab[] = "\a\b\f\n\r\t";
const char * const MessageEditor::friendlyBackTab[] = {
        QT_TRANSLATE_NOOP("MessageEditor", "bell"),
        QT_TRANSLATE_NOOP("MessageEditor", "backspace"),
        QT_TRANSLATE_NOOP("MessageEditor", "new page"),
        QT_TRANSLATE_NOOP("MessageEditor", "new line"),
        QT_TRANSLATE_NOOP("MessageEditor", "carriage return"),
        QT_TRANSLATE_NOOP("MessageEditor", "tab")
    };

const char *bellImageName = "trolltech/bellImage";
const char *backspaceImageName = "trolltech/bsImage";
const char *newpageImageName = "trolltech/newpageImage";
const char *newlineImageName = "trolltech/newlineImage";
const char *crImageName = "trolltech/crImage";
const char *tabImageName = "trolltech/tabImage";
const char *backTabImages[] = {
    bellImageName,
    backspaceImageName,
    newpageImageName,
    newlineImageName,
    crImageName,
    tabImageName};

ExpandingTextEdit::ExpandingTextEdit(QWidget *parent)
    : QTextEdit(parent)
{
    QAbstractTextDocumentLayout *docLayout = document()->documentLayout();
    connect(docLayout, SIGNAL(documentSizeChanged(QSizeF)), SLOT(updateHeight(QSizeF)));
}

void ExpandingTextEdit::updateHeight(const QSizeF &documentSize)
{
    int contentsHeight = qRound(documentSize.height());

    // This is just for safety, in practice the document height seems to be at least
    // a line high, even when the document is empty.
    int fontHeight = fontMetrics().height();
    if (contentsHeight < fontHeight)
        contentsHeight = fontHeight;

    if (contentsHeight != height()) {
        setMinimumHeight(contentsHeight);
        setMaximumHeight(contentsHeight);
    }
}

class BackTabTextEdit : public ExpandingTextEdit
{
public:
    BackTabTextEdit(QWidget *parent = 0);

    virtual QVariant loadResource(int type, const QUrl &name);

    virtual void keyPressEvent(QKeyEvent *e);
    virtual void focusInEvent(QFocusEvent *e);

    QMap<QUrl, QImage> m_backTabOmages;
    QImage m_tabImg;
    QImage m_newlineImg;
};

BackTabTextEdit::BackTabTextEdit(QWidget *parent)
    : ExpandingTextEdit(parent)
{
    setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::MinimumExpanding));
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setLineWrapMode(QTextEdit::WidgetWidth);
}

QVariant BackTabTextEdit::loadResource(int type, const QUrl &name)
{
    QImage img;
    if (type == QTextDocument::ImageResource) {
        img = m_backTabOmages.value(name);
        if (img.isNull()) {
            for (uint i = 0; i < qstrlen(MessageEditor::backTab); ++i) {
                if (backTabImages[i] && name == QUrl(QLatin1String(backTabImages[i]))) {

                    QFont fnt = font();
                    fnt.setItalic(true);
                    QFontMetrics fm(fnt);
                    int h = fm.height();

                    QString str = QString::fromAscii("(%1)").arg(QLatin1String(MessageEditor::friendlyBackTab[i]));
                    int w = fm.boundingRect(str).width() + 1;   //###
                    QImage textimg(w, h, QImage::Format_RGB32);
                    textimg.fill(qRgb(255,255,255));

                    QPainter p(&textimg);
                    p.setPen(QColor(Qt::blue));
                    p.setFont(fnt);
                    p.drawText(0, fm.ascent(), str);            //###
                    document()->addResource(QTextDocument::ImageResource, QUrl(QLatin1String(backTabImages[i])), textimg);

                    m_backTabOmages.insert(name, textimg);
                    return textimg;
                }
            }
        }
    }
    return img;
}

void BackTabTextEdit::focusInEvent(QFocusEvent *e)
{
    TransEditor *te = qobject_cast<TransEditor*>(parent());
    te->gotFocusInEvent(e);
    ExpandingTextEdit::focusInEvent(e);
}

void BackTabTextEdit::keyPressEvent(QKeyEvent *e)
{
    bool eatevent = false;
    QTextCursor tc = textCursor();
    if (e->modifiers() == Qt::NoModifier) {
        switch (e->key()) {
        case Qt::Key_Tab: {
            tc = textCursor();
            tc.insertImage(QLatin1String(tabImageName));
            eatevent = true;
            break; }
        case Qt::Key_Return: {
            tc = textCursor();
            document()->blockSignals(true);
            tc.beginEditBlock();
            tc.insertImage(QLatin1String(newlineImageName));
            document()->blockSignals(false);
            tc.insertBlock();
            tc.endEditBlock();
            eatevent = true;
            break; }
        case Qt::Key_Backspace:
            if (tc.anchor() == tc.position()) {
                QTextCursor tc = textCursor();
                if (!tc.atStart() && tc.atBlockStart()) {
                    tc.movePosition(QTextCursor::PreviousCharacter, QTextCursor::KeepAnchor, 1);
                    QTextCharFormat fmt = tc.charFormat();
                    if (fmt.isImageFormat()) {
                        tc.movePosition(QTextCursor::PreviousCharacter, QTextCursor::KeepAnchor, 1);
                    }
                    tc.removeSelectedText();
                    eatevent = true;
                }
            }
            break;
        case Qt::Key_Delete:
            if (tc.anchor() == tc.position()) {
                QTextCursor tc = textCursor();
                tc.movePosition(QTextCursor::NextCharacter, QTextCursor::KeepAnchor, 1);
                QTextCharFormat fmt = tc.charFormat();
                if (fmt.isImageFormat()) {
                    if (!tc.atEnd() && tc.atBlockEnd()) {
                        tc.movePosition(QTextCursor::NextCharacter, QTextCursor::KeepAnchor, 1);
                    }
                    tc.removeSelectedText();
                    eatevent = true;
                }
            }
            break;
        }
    }
    // Also accept Key_Enter on the numpad
    if (e->modifiers() == Qt::KeypadModifier && e->key() == Qt::Key_Enter) {
        tc = textCursor();
        document()->blockSignals(true);
        tc.beginEditBlock();
        tc.insertImage(QLatin1String(newlineImageName));
        document()->blockSignals(false);
        tc.insertBlock();
        tc.endEditBlock();
        eatevent = true;
    }

    if (eatevent) e->accept();
    else ExpandingTextEdit::keyPressEvent(e);
}

TransEditor::TransEditor(const QString &title, QWidget *parent)
    : QWidget(parent)
{
    m_editor = new BackTabTextEdit(this);
    m_editor->setObjectName(QLatin1String("translation editor"));
    QPalette p = m_editor->palette();
    p.setColor(QPalette::Disabled, QPalette::Base, p.color(QPalette::Active, QPalette::Base));
    m_editor->setPalette(p);

    m_label = new QLabel(title, this);
    m_label->setFocusProxy(m_editor);

    setFocusProxy(m_editor);

    QVBoxLayout *layout = new QVBoxLayout;
    layout->setSpacing(2);
    layout->setMargin(0);
    layout->addWidget(m_label);
    layout->addWidget(m_editor);
    setLayout(layout);
}

void TransEditor::gotFocusInEvent(QFocusEvent *e)
{
    Q_UNUSED(e);
    emit gotFocusIn();
}

void TransEditor::setLabel(const QString &text)
{
    m_label->setText(text);
    m_label->adjustSize();
}

void TransEditor::setEditingEnabled(bool enabled)
{
    // Use read-only state so that the text can still be copied
    m_editor->setReadOnly(!enabled);
    m_label->setEnabled(enabled);
}

QString TransEditor::translation() const
{
    QString plain;
    QTextBlock tb = m_editor->document()->begin();
    for (int b = 0; b < m_editor->document()->blockCount(); ++b) {
        QTextBlock::iterator it = tb.begin();
        if (it.atEnd()) {
            plain += tb.text();
        } else {
            while ( !it.atEnd() ) {
                QTextCharFormat fmt = it.fragment().charFormat();
                if (fmt.isImageFormat()) {
                    QTextImageFormat tif = fmt.toImageFormat();
                    if (tif.name() == QLatin1String(tabImageName)) plain += QLatin1Char('\t');
                    else if (tif.name() == QLatin1String(newlineImageName)) plain += QLatin1Char('\n');
                } else {
                    plain += it.fragment().text();
                }
                ++it;
            }
        }
        tb = tb.next();
    }
    return plain;
}

void MessageEditor::visualizeBackTabs(const QString &text, QTextEdit *te)
{
    te->clear();
    QTextCursor tc(te->textCursor());
    QTextCharFormat blueFormat = defFormat;
    blueFormat.setForeground(QBrush(Qt::blue));
    blueFormat.setFontItalic(true);
    blueFormat.setProperty(QTextFormat::UserProperty, -1);

    QString plainText;
    for (int i = 0; i < (int) text.length(); ++i)
    {
        int ch = text[i].unicode();
        if (ch < 0x20)
        {
            if (!plainText.isEmpty())
            {
                tc.insertText(plainText, defFormat);
                plainText.clear();
            }
            const char *p = strchr(backTab, ch);
            // store the character in the user format property
            // in the first '(' in the phrase
            blueFormat.setProperty(QTextFormat::UserProperty, ch);
            tc.insertText(QString(QLatin1String("(")), blueFormat);
            blueFormat.setProperty(QTextFormat::UserProperty, -1);
            if (p == 0)
            {
                tc.insertText(QString::number(ch, 16) + QLatin1String(")"), blueFormat);
            }
            else
            {
                tc.insertText(MessageEditor::tr(friendlyBackTab[p - backTab]) + QLatin1String(")"),
                    blueFormat);
                if (backTab[p - backTab] == '\n')
                    tc.insertBlock();
            }
        }
        // if a space is by itself, at the end, or beside other spaces
        else if (ch == ' ')
        {
            if (i == 0 || i == text.length() - 1 || text[i - 1].isSpace() ||
                text[i + 1].isSpace())
            {
                tc.insertText(plainText, defFormat);
                plainText.clear();
                blueFormat.setProperty(QTextFormat::UserProperty, ch);
                tc.insertText(QString(QLatin1String("(")), blueFormat);
                blueFormat.setProperty(QTextFormat::UserProperty, -1);
                tc.insertText(MessageEditor::tr("sp)"), blueFormat);
            }
            else
            {
                plainText += QLatin1Char(' ');
            }
        }
        else
        {
            plainText += QString(ch);
        }
    }
    tc.insertText(plainText, defFormat);
}

SourceTextEdit::SourceTextEdit(QWidget *parent)
    : ExpandingTextEdit(parent),
      srcmenu(0)
{
    setReadOnly(true);
    setFrameStyle(QFrame::NoFrame);
    setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Fixed));
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    actCopy = new QAction(tr("&Copy"), this);
    actCopy->setShortcut(QKeySequence(tr("Ctrl+C")));
    actSelect = new QAction(tr("Select &All"), this);
    actSelect->setShortcut(QKeySequence(tr("Ctrl+A")));

    connect(actCopy, SIGNAL(triggered()), this, SLOT(copySelection()));
    connect(actSelect, SIGNAL(triggered()), this, SLOT(selectAll()));
}

void SourceTextEdit::copySelection()
{
    QTextDocumentFragment tdf = textCursor().selection();
    QTextDocument td;
    QTextCursor tc(&td);
    tc.insertFragment(tdf);
    int ch;

    tc.movePosition(QTextCursor::Start);
    while (!tc.atEnd())
    {
        tc.movePosition(QTextCursor::NextCharacter);
        ch = tc.charFormat().intProperty(QTextFormat::UserProperty);
        if (ch != 0) // if wrong format
        {
            // delete char
            tc.deletePreviousChar();
            if (ch != -1) // insert backtab
                tc.insertText(QString(ch));
        }
    }

    QApplication::clipboard()->setText(td.toPlainText());
}

void SourceTextEdit::contextMenuEvent(QContextMenuEvent *e)
{
    if (!srcmenu) {
        srcmenu = new QMenu(this);
        srcmenu->addAction(actCopy);
        srcmenu->addAction(actSelect);
    }

    actCopy->setEnabled(textCursor().hasSelection());
    actSelect->setEnabled(!document()->isEmpty());

    srcmenu->popup(e->globalPos());
}

AuxTranslationWidget::AuxTranslationWidget(const QString &lang, QWidget *parent)
        : QWidget(parent)
{
    QLayout *layout = new QVBoxLayout;
    layout->setMargin(0);

    m_label = new QLabel(this);
    m_label->setText((tr("Existing %1 translation")).arg(lang));
    layout->addWidget(m_label);

    m_textEdit = new SourceTextEdit(this);
    m_textEdit->setWhatsThis(tr("This area shows text from an auxillary translation."));
    QPalette p = m_textEdit->palette();
    p.setColor(QPalette::Disabled, QPalette::Base, p.color(QPalette::Active, QPalette::Base));
    layout->addWidget(m_textEdit);
    setLayout(layout);

    connect(m_textEdit->document(), SIGNAL(contentsChanged()), this, SIGNAL(textChanged()));
    connect(m_textEdit->document(), SIGNAL(contentsChanged()), this, SLOT(textEditChanged()));
    connect(m_textEdit, SIGNAL(selectionChanged()), this, SIGNAL(selectionChanged()));
}

void AuxTranslationWidget::textEditChanged()
{
    setVisible(!m_textEdit->toPlainText().isEmpty());
}

/*
   EditorPage class impl.

   A frame that contains the source text, translated text and any
   source code comments and hints.
*/
EditorPage::EditorPage(QLocale::Language targetLanguage, MessageEditor *parent, const char *name)
    : QFrame(parent),
      m_layout(new QVBoxLayout)
{
    setObjectName(QLatin1String(name));

    // Due to CSS being rather broken on the Mac style at the moment, only
    // use the border-image on non-Mac systems.
    setStyleSheet(QString(
#ifndef Q_WS_MAC
            "EditorPage { border-image: url(:/images/transbox.png) 12 16 16 12 repeat;"
            "             border-width: 12px 16px 16px 12px; }"
#endif
            "EditorPage { background-color: white; }"
            "QLabel { font-weight: bold; }"
            ));
#ifdef Q_WS_MAC
    setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
#endif
    setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed));

    m_layout->setMargin(2);

    // Use white explicitly as the background color for the editor page.
    QPalette p = palette();
    p.setColor(QPalette::Active,   QPalette::Base,   Qt::white);
    p.setColor(QPalette::Inactive, QPalette::Base,   Qt::white);
    p.setColor(QPalette::Disabled, QPalette::Base,   Qt::white);
    p.setColor(QPalette::Active,   QPalette::Window, Qt::white);
    p.setColor(QPalette::Inactive, QPalette::Window, Qt::white);
    p.setColor(QPalette::Disabled, QPalette::Window, Qt::white);
    parent->setPalette(p);

    m_srcTextLabel = new QLabel(tr("Source text"), this);
    m_layout->addWidget(m_srcTextLabel);

    srcText = new SourceTextEdit(this);
    p = srcText->palette();
    p.setColor(QPalette::Disabled, QPalette::Base, p.color(QPalette::Active, QPalette::Base));
    srcText->setPalette(p);
    srcText->setWhatsThis(tr("This area shows the source text.") );
    m_layout->addWidget(srcText);

    connect(srcText, SIGNAL(selectionChanged()), SLOT(sourceSelectionChanged()));

    cmtText = new SourceTextEdit(this);
    cmtText->setObjectName(QLatin1String("comment/context view"));
    p = cmtText->palette();
    p.setColor(QPalette::Active, QPalette::Base, QColor(236, 245, 255));
    p.setColor(QPalette::Inactive, QPalette::Base, QColor(236, 245, 255));
    cmtText->setPalette(p);
    cmtText->setWhatsThis(tr("This area shows a comment that"
                        " may guide you, and the context in which the text"
                        " occurs.") );
    m_layout->addWidget(cmtText);

    setTargetLanguage(targetLanguage);
    m_pluralEditMode = false;
    addPluralForm(m_invariantForm);

    // Focus
    //setFocusPolicy(Qt::StrongFocus);
    //parent->setFocusProxy(transText);
    //transLbl->setFocusProxy(transText);
    //m_srcTextLabel->setFocusProxy(transText);
    //srcText->setFocusProxy(transText);
    //cmtText->setFocusProxy(transText);
    //setFocusProxy(transText);

    updateCommentField();
    setLayout(m_layout);
}

void EditorPage::showNothing()
{
    srcText->clear();
    srcText->setToolTip("");
    cmtText->clear();
    foreach (AuxTranslationWidget *widget, auxTranslations)
        widget->clearText();
    handleChanges();
}

void EditorPage::setAuxLanguages(const QStringList &languages)
{
    qDeleteAll(auxTranslations);
    auxTranslations.clear();

    foreach (QString language, languages) {
        AuxTranslationWidget *auxWidget = new AuxTranslationWidget(language, this);
        m_layout->insertWidget(m_layout->indexOf(cmtText), auxWidget);

        connect(auxWidget, SIGNAL(selectionChanged()), SLOT(sourceSelectionChanged()));

        auxTranslations.append(auxWidget);
    }
}

void EditorPage::handleChanges()
{
    adjustTranslationFieldHeights();
    updateCommentField();
}

void EditorPage::addPluralForm(const QString &label)
{
    TransEditor *transEditor = new TransEditor(label, this);
    if (m_transTexts.count())
        transEditor->setVisible(false);
    m_layout->addWidget(transEditor);

    connect(transEditor->editor(), SIGNAL(selectionChanged()), SLOT(translationSelectionChanged()));
    connect(transEditor, SIGNAL(gotFocusIn(void)), SIGNAL(currentTranslationEditorChanged(void)));

    m_transTexts << transEditor;
}

int EditorPage::currentTranslationEditor()
{
    for (int i = 0; i < m_transTexts.count(); ++i) {
        QTextEdit *te = m_transTexts[i]->editor();
        if (te->hasFocus()) return i;
    }
    return -1;  //no focus
}

/*! internal
    Returns all translations for an item.
    The number of translations is dependent on if we have a plural form or not.
    If we don't have a plural form, then this should only contain one item.
    Otherwise it will contain the number of numerus forms for the particular language.
*/
QStringList EditorPage::translations() const
{
    QStringList translations;
    for (int i = 0; i < m_transTexts.count() && m_transTexts.at(i)->isVisible(); ++i) {
        translations << m_transTexts[i]->translation();
    }
    return translations;
}

/*
   Don't show the comment field if there are no comments.
*/
void EditorPage::updateCommentField()
{
    cmtText->setVisible(!cmtText->toPlainText().isEmpty());
}

void EditorPage::resizeEvent(QResizeEvent*)
{
    adjustTranslationFieldHeights();
}

void EditorPage::adjustTranslationFieldHeights()
{
    if (srcText->textCursor().hasSelection())
        translationSelectionChanged();
}

// makes sure only one of the textedits has a selection
void EditorPage::sourceSelectionChanged()
{
    for (int i = 0; i < m_transTexts.count(); ++i) {
        QTextEdit *te = m_transTexts[i]->editor();
        bool oldBlockState = te->blockSignals(true);
        QTextCursor c = te->textCursor();
        c.clearSelection();
        te->setTextCursor(c);
        te->blockSignals(oldBlockState);
    }
    emit selectionChanged();
}

void EditorPage::translationSelectionChanged()
{
    bool oldBlockState = srcText->blockSignals(true);
    QTextCursor c = srcText->textCursor();
    c.clearSelection();
    srcText->setTextCursor(c);
    srcText->blockSignals(oldBlockState);

    // clear the selection for all except the sender
    QTextEdit *te = qobject_cast<QTextEdit*>(sender());
    for (int i = 0; i < m_transTexts.count(); ++i) {
        QTextEdit *t = m_transTexts[i]->editor();
        if (t != te) {
            oldBlockState = t->blockSignals(true);
            QTextCursor c = t->textCursor();
            c.clearSelection();
            t->setTextCursor(c);
            t->blockSignals(oldBlockState);
        }
    }

    emit selectionChanged();
}

int EditorPage::activeTranslationNumerus() const
{
    for (int i = 0; i < m_transTexts.count(); ++i) {
        if (m_transTexts[i]->editor()->hasFocus()) {
            return i;
        }
    }
    //### hmmm.....
    if (m_transTexts.count()) return 0;
    return -1;
}

void EditorPage::setTargetLanguage(QLocale::Language lang)
{
    if (uint(lang) > uint(QLocale::LastLanguage))
        m_invariantForm = tr("Translation");
    else
        m_invariantForm = tr("%1 translation").arg(QLocale::languageToString(lang));
}

void EditorPage::setNumerusForms(const QStringList &numerusForms)
{
    m_numerusForms = numerusForms;

    if (!m_pluralEditMode) {
        m_transTexts[0]->setLabel(m_invariantForm);
    } else {
        m_transTexts[0]->setLabel(tr("Translation (%1)").arg(m_numerusForms[0]));
    }
    int i;
    for (i = 1; i < m_numerusForms.count(); ++i) {
        QString label = tr("Translation (%1)").arg(m_numerusForms[i]);
        if (i >= m_transTexts.count()) {
            addPluralForm(label);
        } else {
            m_transTexts[i]->setLabel(label);
        }
        m_transTexts[i]->setVisible(m_pluralEditMode);
    }
    for (int j = m_transTexts.count() - i; j > 0; --j) {
        TransEditor *te = m_transTexts.takeLast();
        delete te;
        ++i;
    }
}

QTextEdit *EditorPage::activeTransText() const
{
    int numerus = activeTranslationNumerus();
    if (numerus != -1) {
        return m_transTexts[numerus]->editor();
    }
    return 0;
}

/*
   MessageEditor class impl.

   Handles layout of dock windows and the editor page.
*/
MessageEditor::MessageEditor(LanguagesManager *languages, QMainWindow *parent)
    : QScrollArea(parent->centralWidget()),
      m_languages(languages),
      cutAvail(false),
      copyAvail(false),
      doGuesses(true)
{
    bottomDockWidget = new QDockWidget(parent);
    bottomDockWidget->setObjectName(QLatin1String("PhrasesDockwidget"));
    bottomDockWidget->setAllowedAreas(Qt::AllDockWidgetAreas);
    bottomDockWidget->setFeatures(QDockWidget::AllDockWidgetFeatures);
    bottomDockWidget->setWindowTitle(tr("Phrases and guesses"));

    phraseTv = new QTreeView(bottomDockWidget);
    phraseTv->setObjectName(QLatin1String("phrase list view"));
    phrMdl = new PhraseModel(phraseTv);
    phraseTv->setModel(phrMdl);
    phraseTv->setAlternatingRowColors(true);
    phraseTv->setSelectionBehavior(QAbstractItemView::SelectRows);
    phraseTv->setSelectionMode(QAbstractItemView::SingleSelection);
    phraseTv->setRootIsDecorated(false);
    phraseTv->setItemsExpandable(false);

    phraseTv->header()->setResizeMode(QHeaderView::Stretch);
    phraseTv->header()->setClickable(true);

    bottomDockWidget->setWidget(phraseTv);

    parent->addDockWidget(Qt::BottomDockWidgetArea, bottomDockWidget);

    for (int i = 0; i < 9; ++i)
        (void) new GuessShortcut(i, this, SLOT(guessActivated(int)));

    setObjectName(QLatin1String("scroll area"));

    editorPage = new EditorPage(m_languages->mainModel()->language(), this, "editor page");

    setWidget(editorPage);
    setWidgetResizable(true);

    defFormat = editorPage->srcText->currentCharFormat();

    // Signals
    connect(this, SIGNAL(translationChanged(const QStringList &)),
            this, SLOT(updateButtons()));

    connect(this, SIGNAL(translationChanged(const QStringList &)),
            this, SLOT(checkUndoRedo()));
    connect(editorPage, SIGNAL(currentTranslationEditorChanged()),
            this, SLOT(checkUndoRedo()));

    connect(editorPage, SIGNAL(selectionChanged()),
            this, SLOT(updateCutAndCopy()));
    connect(qApp->clipboard(), SIGNAL(dataChanged()),
            this, SLOT(clipboardChanged()));
    connect(phraseTv, SIGNAL(doubleClicked(QModelIndex)),
            this, SLOT(insertPhraseInTranslation(QModelIndex)));

    connect(m_languages, SIGNAL(listChanged()), this, SLOT(messageModelListChanged()));
    connect(m_languages->mainModel(), SIGNAL(languageChanged(QLocale::Language)), editorPage, SLOT(setTargetLanguage(QLocale::Language)));

    clipboardChanged();

    phraseTv->installEventFilter(this);

    // What's this
    this->setWhatsThis(tr("This whole panel allows you to view and edit "
                          "the translation of some source text."));
    showNothing();
}

void MessageEditor::checkUndoRedo()
{
    QTextEdit *te = editorPage->activeTransText();
    emit undoAvailable(te->document()->isUndoAvailable());
    emit redoAvailable(te->document()->isRedoAvailable());
}

void MessageEditor::messageModelListChanged()
{
    QStringList languages;
    foreach (MessageModel *model, m_languages->auxModels()) {
        languages << QLocale::languageToString(model->language());
    }
    editorPage->setAuxLanguages(languages);
    editorPage->setTargetLanguage(m_languages->mainModel()->language());
}


bool MessageEditor::eventFilter(QObject *o, QEvent *e)
{
    // handle copying from the source
    if ((e->type() == QEvent::KeyPress) ||
        (e->type() == QEvent::ShortcutOverride))
    {
        QKeyEvent *ke = static_cast<QKeyEvent *>(e);

        // handle return key in phrase list
        if (o == phraseTv
            && e->type() == QEvent::KeyPress
            && ke->modifiers() == Qt::NoModifier
            && ke->key() == Qt::Key_Return
            && phraseTv->currentIndex().isValid())
        {
            insertPhraseInTranslationAndLeave(phraseTv->currentIndex());
            return false;
        }

        if (ke->modifiers() & Qt::ControlModifier)
        {
            if ((ke->key() == Qt::Key_A) &&
                editorPage->srcText->underMouse())
            {
                editorPage->srcText->selectAll();
                return true;
            }
            if ((ke->key() == Qt::Key_C) &&
                editorPage->srcText->textCursor().hasSelection())
            {
                editorPage->srcText->copySelection();
                return true;
            }
        }
    }

    return QScrollArea::eventFilter(o, e);
}

QTreeView *MessageEditor::phraseView() const
{
    return phraseTv;
}

void MessageEditor::showNothing()
{
    setEditingEnabled(false);
    sourceText.clear();
    setTranslation(QString(), 0, false);
    editorPage->showNothing();
}

static CandidateList similarTextHeuristicCandidates( MessageModel::iterator it,
                        const char *text,
                        int maxCandidates )
{
    QList<int> scores;
    CandidateList candidates;

    StringSimilarityMatcher stringmatcher(QString::fromLatin1(text));

    for (MessageItem *m = 0; (m = it.current()); ++it) {
        TranslatorMessage mtm = m->message();
        if ( mtm.type() == TranslatorMessage::Unfinished ||
             mtm.translation().isEmpty() )
            continue;

        QString s = m->sourceText();

        int score = stringmatcher.getSimilarityScore(s);

        if ( (int) candidates.count() == maxCandidates &&
             score > scores[maxCandidates - 1] )
            candidates.removeAt( candidates.size()-1 );
        if ( (int) candidates.count() < maxCandidates && score >= textSimilarityThreshold ) {
            Candidate cand( s, mtm.translation() );

            int i;
            for ( i = 0; i < (int) candidates.size(); i++ ) {
                if ( score >= scores.at(i) ) {
                    if ( score == scores.at(i) ) {
                        if ( candidates.at(i) == cand )
                            goto continue_outer_loop;
                    } else {
                        break;
                    }
                }
            }
            scores.insert( i, score );
            candidates.insert( i, cand );
        }
        continue_outer_loop:
        ;
    }
    return candidates;
}


void MessageEditor::showMessage(const QString &context,
                                const QString &text,
                                const QString &comment,
                                const QString &fullContext,
                                const QString &fileName,
                                const int lineNumber,
                                const QStringList &translations,
                                TranslatorMessage::Type type,
                                const QList<Phrase> &phrases)
{
    phraseTv->clearSelection();

    bool obsolete = (type == TranslatorMessage::Obsolete);
    setEditingEnabled(!obsolete);
    sourceText = text;

    visualizeBackTabs(text, editorPage->srcText);

    if (!fileName.isNull()) {
        QString toolTip = tr("'%1'\nLine: %2").arg(fileName, QString::number(lineNumber));
        editorPage->srcText->setToolTip(toolTip);
    } else {
        editorPage->srcText->setToolTip("");
    }

    QList<MessageModel *> models = m_languages->auxModels();
    int i;
    for (i = 0; i < models.size(); i++)
        visualizeBackTabs(models.at(i)->translator()->translate(context.simplified().toLatin1(),
                                                   text.simplified().toLatin1(),
                                                   comment.simplified().toLatin1()),
                                                   editorPage->auxTranslations.at(i)->getTextEdit());

    if (!fullContext.isEmpty() && !comment.isEmpty())
        visualizeBackTabs(fullContext.simplified() + QLatin1String("\n") +
            comment.simplified(), editorPage->cmtText);
    else if (!fullContext.isEmpty() && comment.isEmpty())
        visualizeBackTabs(fullContext.simplified(), editorPage->cmtText);
    else if (fullContext.isEmpty() && !comment.isEmpty())
        visualizeBackTabs(comment.simplified(), editorPage->cmtText);
    else
        editorPage->cmtText->clear();

    editorPage->m_pluralEditMode = translations.count() > 1;
    if (!editorPage->m_pluralEditMode) {
        editorPage->m_transTexts[0]->setLabel(editorPage->m_invariantForm);
    } else {
        editorPage->m_transTexts[0]->setLabel(tr("Translation (%1)").arg(editorPage->m_numerusForms.first()));
    }
    for (i = 0; i < qMax(1, editorPage->m_numerusForms.count()); ++i) {
        bool shouldShow = i < translations.count();
        if (shouldShow) {
            setTranslation(translations[i], i, false);
        } else {
            setTranslation(QString(), i, false);
        }
        if (i >= 1) {
            editorPage->m_transTexts[i]->setVisible(shouldShow);
        }
    }
    phrMdl->removePhrases();

    foreach (Phrase p, phrases) {
        phrMdl->addPhrase(p);
    }

    if (doGuesses && !sourceText.isEmpty()) {
        CandidateList cl = similarTextHeuristicCandidates(m_languages->mainModel()->begin(),
            sourceText.toLatin1(), MaxCandidates);
        int n = 0;
        QList<Candidate>::Iterator it = cl.begin();
        while (it != cl.end()) {
            QString def;
            if (n < 9)
                def = tr("Guess (%1)").arg(QString(QKeySequence(Qt::CTRL | (Qt::Key_0 + (n + 1)))));
            else
                def = tr("Guess");
            phrMdl->addPhrase(Phrase((*it).source, (*it).target, def, n));
            ++n;
            ++it;
        }
    }
    phrMdl->resort();
    editorPage->handleChanges();
}

void MessageEditor::setNumerusForms(const QStringList &numerusForms)
{
    // uninstall the emitTranslationChanged slots and remove event filters
    for (int i = 0; i  < editorPage->m_transTexts.count(); ++i) {
        QTextEdit *transText = editorPage->m_transTexts[i]->editor();
        disconnect(transText->document(), SIGNAL(contentsChanged()), this, SLOT(emitTranslationChanged()));
        transText->removeEventFilter(this);
    }
    editorPage->setNumerusForms(numerusForms);

    // reinstall event filters and set up the emitTranslationChanged slot
    for (int i = 0; i  < editorPage->m_transTexts.count(); ++i) {
        QTextEdit *transText = editorPage->m_transTexts[i]->editor();
        transText->installEventFilter(this);
        connect(transText->document(), SIGNAL(contentsChanged()),
                this, SLOT(emitTranslationChanged()));
        // What's this
        transText->setWhatsThis(tr("This is where you can enter or modify"
                                   " the translation of some source text.") );
    }
}

static void visualizeImages(const QString &text, QTextEdit *te)
{
    te->clear();
    QTextCursor tc(te->textCursor());

    QString plainText;
    for (int i = 0; i < (int) text.length(); ++i)
    {
        int ch = text[i].unicode();
        if (ch < 0x20)
        {
            if (!plainText.isEmpty())
            {
                tc.insertText(plainText);
                plainText.clear();
            }
            const char *p = strchr(MessageEditor::backTab, ch);
            if (p)
            {
                if (backTabImages[p - MessageEditor::backTab]) {
                    tc.insertImage(QLatin1String(backTabImages[p - MessageEditor::backTab]));
                }
                if (MessageEditor::backTab[p - MessageEditor::backTab] == '\n') {
                    tc.insertBlock();
                }
            }
        }
        else
        {
            plainText += QString(ch);
        }
    }
    tc.insertText(plainText);
}

void MessageEditor::setTranslation(const QString &translation, int numerus, bool emitt)
{
    if (numerus >= editorPage->m_transTexts.count()) numerus = 0;
    QTextEdit *transText = editorPage->m_transTexts[numerus]->editor();
    // Block signals so that a signal is not emitted when
    // for example a new source text item is selected and *not*
    // the actual translation.
    if (!emitt)
        transText->document()->blockSignals(true);

    if (translation.isNull())
        transText->clear();
    else
        visualizeImages(translation, transText);

    if (!emitt)
    {
        transText->document()->blockSignals(false);

        //don't undo the change
        emit undoAvailable(false);
        emit redoAvailable(false);
        updateButtons();
    }
    emit cutAvailable(false);
    emit copyAvailable(false);
}

void MessageEditor::setEditingEnabled(bool enabled)
{
    for (int i = 0; i < editorPage->m_transTexts.count(); ++i) {
        TransEditor* te = editorPage->m_transTexts[i];
        te->setEditingEnabled(enabled);
    }

    phraseTv->setEnabled(enabled);
    updateCanPaste();
}

void MessageEditor::undo()
{
    editorPage->activeTransText()->document()->undo();
}

void MessageEditor::redo()
{
    editorPage->activeTransText()->document()->redo();
}

void MessageEditor::cut()
{
    if (editorPage->activeTransText()->textCursor().hasSelection())
        editorPage->activeTransText()->cut();
}

void MessageEditor::copy()
{
    if (editorPage->srcText->textCursor().hasSelection()) {
        editorPage->srcText->copySelection();
    } else if (editorPage->activeTransText()->textCursor().hasSelection()) {
        editorPage->activeTransText()->copy();
    }
}

void MessageEditor::paste()
{
    editorPage->activeTransText()->paste();
}

void MessageEditor::selectAll()
{
    // make sure we don't select the selection of a translator textedit, if we really want the
    // source text editor to be selected.
    if (!editorPage->srcText->underMouse())
        editorPage->activeTransText()->selectAll();
}

void MessageEditor::emitTranslationChanged()
{
    emit translationChanged(editorPage->translations());
}

void MessageEditor::guessActivated(int key)
{
    QModelIndex mi;
    Phrase p;

    for (int i=0; i<phrMdl->phraseList().count(); ++i) {
        mi = phrMdl->QAbstractTableModel::index(i, 0);
        p = phrMdl->phrase(mi);
        if (p.shortcut() == key) {
            insertPhraseInTranslation(mi);
            break;
        }
    }
}

void MessageEditor::insertPhraseInTranslation(const QModelIndex &index)
{
    if (!editorPage->activeTransText()->isReadOnly())
    {
        editorPage->activeTransText()->textCursor().insertText(phrMdl->phrase(index).target());
        emitTranslationChanged();
    }
}

void MessageEditor::insertPhraseInTranslationAndLeave(const QModelIndex &index)
{
    if (!editorPage->activeTransText()->isReadOnly())
    {
        editorPage->activeTransText()->textCursor().insertText(phrMdl->phrase(index).target());

        emitTranslationChanged();
        editorPage->activeTransText()->setFocus();
    }
}

void MessageEditor::updateButtons()
{
    bool overwrite = (!editorPage->activeTransText()->isReadOnly() &&
             (editorPage->activeTransText()->toPlainText().trimmed().isEmpty() ||
              mayOverwriteTranslation));
    mayOverwriteTranslation = false;
    emit updateActions(overwrite);
}

void MessageEditor::beginFromSource()
{
    mayOverwriteTranslation = true;
    for (int i = 0; i < editorPage->m_transTexts.count(); ++i) {
        setTranslation(sourceText, i, true);
    }
    setEditorFocus();
}

void MessageEditor::setEditorFocus()
{
    if (!editorPage->hasFocus())
        editorPage->activeTransText()->setFocus();
}

void MessageEditor::updateCutAndCopy()
{
    bool newCopyState = false;
    bool newCutState = false;
    if (editorPage->srcText->textCursor().hasSelection()) {
        newCopyState = true;
    } else if (editorPage->activeTransText()->textCursor().hasSelection()) {
        newCopyState = true;
        newCutState = true;
    }

    if (newCopyState != copyAvail) {
        copyAvail = newCopyState;
        emit copyAvailable(copyAvail);
    }

    if (newCutState != cutAvail) {
        cutAvail = newCutState;
        emit cutAvailable(cutAvail);
    }
}

void MessageEditor::clipboardChanged()
{
    // this is expensive, so move it out of the common path in updateCanPaste
    clipboardEmpty = qApp->clipboard()->text().isNull();
    updateCanPaste();
}

void MessageEditor::updateCanPaste()
{
    emit pasteAvailable(!editorPage->activeTransText()->isReadOnly() && !clipboardEmpty);
}

void MessageEditor::toggleGuessing()
{
    doGuesses = !doGuesses;
    if (!doGuesses) {
        phrMdl->removePhrases();
    }
}


QT_END_NAMESPACE
