/****************************************************************************
**
** Copyright (C) 2006-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <QtDebug>

#include "qxmlformatter.h"
#include "qxpathhelper_p.h"
#include "qxmlserializer_p.h"

QT_BEGIN_NAMESPACE

using namespace QPatternist;

class QXmlFormatterPrivate : public QXmlSerializerPrivate
{
public:
    inline QXmlFormatterPrivate(const QXmlQuery &q,
                                QIODevice *const outputDevice);

    int             indentationDepth;
    int             currentDepth;
    QString         characterBuffer;
    QString         indentString;

    /**
     * Whether we /have/ sent nodes like processing instructions and comments
     * to QXmlSerializer.
     */
    QStack<bool>    canIndent;
};

QXmlFormatterPrivate::QXmlFormatterPrivate(const QXmlQuery &query,
                                           QIODevice *const outputDevice) : QXmlSerializerPrivate(query, outputDevice)
                                                                          , indentationDepth(4)
                                                                          , currentDepth(0)
{
    indentString.reserve(30);
    indentString.resize(1);
    indentString[0] = QLatin1Char('\n');
    canIndent.push(false);
}

/*!
   \class QXmlFormatter
   \brief The QXmlFormatter class indents and pretty-formats XML events.
   \reentrant
   \since 4.4
   \ingroup xml-tools

   QXmlFormatter behaves just like its subclass QXmlFormatter but in addition adds, removes and modifies
   nodes that consists of whitespace only, in order to make produced document be more pleasent for human
   consumption. In addition it may modify whitespace where insignificant, such as between attributes and the document prolog.
   
   For instance, instead of that "<a><b/><c/><p>Some Text</p></a>" is written out, the following is written:

   \code
<a>
    <b/>
    <c/>
    <p>Some Text</p>
</a>
    \endcode

    Indentation modifies the data, and therefore this can be considered harmful. However, typically
    this form of indentation is ok for common formats such as XHTML, Docbook and SVG.

    The document that QXmlFormatter writes out to the outputDevice() is typically easier for humans to read, at the
    cost of storage size and computational overhead.

    The way QXmlFormatter formats is loosely defined and may change in future versions of Qt. If a deterministic
    result is of interest, don't pretty format, use QXmlSerializer directly.
 */

/*!
 \typedef QXmlFormatterPointer

 A typedef for a smart pointer to a QXmlFormatter instance.
*/

/*!
 Constructs a formatter that uses the name pool and message handler in \a query, and writes the result
 to \a outputDevice.
 */
QXmlFormatter::QXmlFormatter(const QXmlQuery &query,
                             QIODevice *outputDevice) : QXmlSerializer(new QXmlFormatterPrivate(query, outputDevice))
{
}

void QXmlFormatter::startFormattingContent()
{
    Q_D(QXmlFormatter);
    if(QPatternist::XPathHelper::isWhitespaceOnly(d->characterBuffer) && d->canIndent.top())
    {
        /* Do the hula hula dance. */
        QXmlSerializer::characters(d->indentString);
    }
    else if(!d->characterBuffer.isEmpty())
    {
        /* Ao ao. Significant data, we don't touch it. */
        QXmlSerializer::characters(d->characterBuffer);
    }

    d->characterBuffer.clear();
}

/*!
  \reimp
 */
void QXmlFormatter::startElement(const QXmlName &name)
{
    Q_D(QXmlFormatter);
    startFormattingContent();
    ++d->currentDepth;
    d->indentString.append(QString(d->indentationDepth, QLatin1Char(' ')));
    d->canIndent.push(true);

    QXmlSerializer::startElement(name);
}

/*!
  \reimp
 */
void QXmlFormatter::endElement()
{
    Q_D(QXmlFormatter);
    --d->currentDepth;
    d->indentString.chop(d->indentationDepth);

    if(!d->hasClosedElement.top().second)
        d->canIndent.top() = false;

    startFormattingContent();

    d->canIndent.pop();
    d->canIndent.top() = true;
    QXmlSerializer::endElement();
}

/*!
  \reimp
 */
void QXmlFormatter::attribute(const QXmlName &name,
                              const QString &value)
{
    QXmlSerializer::attribute(name, value);
}

/*!
 \reimp
 */
void QXmlFormatter::comment(const QString &value)
{
    Q_D(QXmlFormatter);
    startFormattingContent();
    QXmlSerializer::comment(value);
    d->canIndent.top() = true;
}

/*!
 \reimp
 */
void QXmlFormatter::characters(const QString &value)
{
    Q_D(QXmlFormatter);
    d->characterBuffer += value;
}

/*!
 \reimp
 */
void QXmlFormatter::processingInstruction(const QXmlName &name,
                                          const QString &value)
{
    Q_D(QXmlFormatter);
    startFormattingContent();
    QXmlSerializer::processingInstruction(name, value);
    d->canIndent.top() = true;
}

/*!
 \reimp
 */
void QXmlFormatter::atomicValue(const QVariant &value)
{
    startFormattingContent();
    QXmlSerializer::atomicValue(value);
}

/*!
 \reimp
 */
void QXmlFormatter::startDocument()
{
    QXmlSerializer::startDocument();
}

/*!
 \reimp
 */
void QXmlFormatter::endDocument()
{
    QXmlSerializer::endDocument();
}

/*!
 Returns the amount of spaces QXmlFormatter will output for each indentation level.

 The default is four.

 \sa setIndentationDepth()
 */
int QXmlFormatter::indentationDepth() const
{
    Q_D(const QXmlFormatter);
    return d->indentationDepth;
}

/*!
 Sets the amount of spaces QXmlFormatter writes out for each indentation level to \a depth.

 \sa indentationDepth()
 */
void QXmlFormatter::setIndentationDepth(int depth)
{
    Q_D(QXmlFormatter);
    d->indentationDepth = depth;
}

QT_END_NAMESPACE
