/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtScript module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "qscriptvalueiteratorimpl_p.h"

#ifndef QT_NO_SCRIPT

#include "qscriptengine_p.h"
#include "qscriptcontext_p.h"
#include "qscriptvalueimpl_p.h"
#include "qscriptmember_p.h"
#include "qscriptobject_p.h"

QT_BEGIN_NAMESPACE

QScriptValueIteratorImpl::QScriptValueIteratorImpl(const QScriptValueImpl &obj)
{
    Q_ASSERT(obj.isObject());
    m_object = obj;
    m_member.invalidate();
    m_index = 0;
    m_classDataIterator = 0;
    m_useClassDataIterator = false;
    m_ignoresDontEnum = true;
}

QScriptValueIteratorImpl::~QScriptValueIteratorImpl()
{
    if (m_classDataIterator) {
        delete m_classDataIterator;
        m_classDataIterator = 0;
    }
}

bool QScriptValueIteratorImpl::ignoresDontEnum() const
{
    return m_ignoresDontEnum;
}

void QScriptValueIteratorImpl::setIgnoresDontEnum(bool ignore)
{
    m_ignoresDontEnum = ignore;
}

bool QScriptValueIteratorImpl::acceptsMember(const QScript::Member &m) const
{
    return m.isValid() && (m_ignoresDontEnum || !m.dontEnum());
}

QScriptClassDataIterator *QScriptValueIteratorImpl::getClassDataIterator()
{
    if (!m_classDataIterator) {
        QScriptClassData *data = m_object.classInfo()->data().data();
        if (!data)
            return 0;
        m_classDataIterator = data->newIterator(m_object);
    }
    return m_classDataIterator;
}

bool QScriptValueIteratorImpl::hasNext()
{
    if (m_useClassDataIterator)
        return m_classDataIterator->hasNext();

    const int count = m_object.memberCount();
    for (int i = m_index; i < count; ++i) {
        QScript::Member m;
        m_object.member(i, &m);
        if (acceptsMember(m))
            return true;
    }

    QScriptClassDataIterator *it = getClassDataIterator();
    if (!it)
        return false;

    m_useClassDataIterator = true;
    return it->hasNext();
}

void QScriptValueIteratorImpl::next()
{
    if (m_useClassDataIterator) {
        m_classDataIterator->next(&m_member);
        return;
    }

    const int count = m_object.memberCount();
    for (int i = m_index; i < count; ++i) {
        QScript::Member m;
        m_object.member(i, &m);
        if (acceptsMember(m)) {
            m_member = m;
            m_index = i + 1;
            return;
        }
    }

    QScriptClassDataIterator *it = getClassDataIterator();
    if (!it)
        return;

    m_useClassDataIterator = true;
    it->next(&m_member);
}

bool QScriptValueIteratorImpl::hasPrevious()
{
    if (m_useClassDataIterator) {
        if (m_classDataIterator->hasPrevious())
            return true;
    }

    for (int i = m_index - 1; i >= 0; --i) {
        QScript::Member m;
        m_object.member(i, &m);
        if (acceptsMember(m))
            return true;
    }
    return false;
}

void QScriptValueIteratorImpl::previous()
{
    if (m_useClassDataIterator) {
        if (m_classDataIterator->hasPrevious()) {
            m_classDataIterator->previous(&m_member);
            return;
        } else {
            m_useClassDataIterator = false;
            m_index = m_object.memberCount();
        }
    }

    for (int i = m_index - 1; i >= 0; --i) {
        QScript::Member m;
        m_object.member(i, &m);
        if (acceptsMember(m)) {
            m_member = m;
            m_index = i;
            return;
        }
    }
}

QScript::Member *QScriptValueIteratorImpl::member()
{
    return &m_member;
}

QScriptNameIdImpl *QScriptValueIteratorImpl::nameId() const
{
    return m_member.nameId();
}

QString QScriptValueIteratorImpl::name() const
{
    if (!m_member.isValid())
        return QString();
    if (m_member.nameId())
        return m_member.nameId()->s;
    else
        return QScriptEnginePrivate::toString(m_member.id());
}

QScriptValueImpl QScriptValueIteratorImpl::value() const
{
    if (!m_member.isValid())
        return QScriptValueImpl();
    QScriptValueImpl result;
    m_object.get(m_member, &result);
    return result;
}

void QScriptValueIteratorImpl::setValue(const QScriptValueImpl &value)
{
    if (!m_member.isValid())
        return;
    m_object.put(m_member, value);
}

uint QScriptValueIteratorImpl::flags() const
{
    return m_member.flags();
}

QScriptValueImpl QScriptValueIteratorImpl::object() const
{
    return m_object;
}

void QScriptValueIteratorImpl::setObject(const QScriptValueImpl &obj)
{
    Q_ASSERT(obj.isObject());
    m_object = obj;
    if (m_classDataIterator) {
        delete m_classDataIterator;
        m_classDataIterator = 0;
    }
    toFront();
}

void QScriptValueIteratorImpl::remove()
{
    m_object.removeMember(m_member);
}

void QScriptValueIteratorImpl::toFront()
{
    m_member.invalidate();
    m_index = 0;
    m_useClassDataIterator = false;
    if (m_classDataIterator)
        m_classDataIterator->toFront();
}

void QScriptValueIteratorImpl::toBack()
{
    m_member.invalidate();
    m_index = m_object.memberCount();
    QScriptClassDataIterator *it = getClassDataIterator();
    if (it) {
        it->toBack();
        m_useClassDataIterator = true;
    } else {
        m_useClassDataIterator = false;
    }
}

QT_END_NAMESPACE

#endif // QT_NO_SCRIPT
