/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QT_NO_GRAPHICSVIEW

#include "qgraphicslayout.h"
#include "qgraphicsproxywidget.h"
#include "private/qgraphicsproxywidget_p.h"
#include "private/qwidget_p.h"
#include "private/qapplication_p.h"

#include <QtCore/qdebug.h>
#include <QtGui/qevent.h>
#include <QtGui/qgraphicsscene.h>
#include <QtGui/qgraphicssceneevent.h>
#include <QtGui/qpainter.h>
#include <QtGui/qstyleoption.h>
#include <QtGui/qgraphicsview.h>

QT_BEGIN_NAMESPACE

//#define GRAPHICSPROXYWIDGET_DEBUG

/*!
    \class QGraphicsProxyWidget
    \brief The QGraphicsProxyWidget class provides a proxy layer for embedding
    a QWidget in a QGraphicsScene.
    \since 4.4
    \ingroup multimedia

    QGraphicsProxyWidget embeds any QWidget-based widget into
    QGraphicsScene. It forwards events between the two and translates between
    QWidget's integer-based geometry and QGraphicsWidget's qreal-based
    geometry. QGraphicsProxyWidget supports all core features of QWidget,
    including tab focus, keyboard input, Drag & Drop, and popups, and you can
    also embed widgets with subwidgets.

    Popup children of embedded widgets are automatically embedded as children
    of this proxy widget (e.g., QComboBox's popup is embedded together with
    the combo box).

    There are two ways to embed a widget using QGraphicsProxyWidget. The most
    common way is to pass a widget pointer to QGraphicsScene::addWidget()
    together with any relevant \l Qt::WindowFlags. This function returns a pointer
    to a QGraphicsProxyWidget. You can then choose to reparent or position the
    proxy or the widget directly. Alternatively, you can create a new
    QGraphicsProxyWidget item first, and then call setWidget() to embed a
    QWidget later. The widget() function returns a pointer to the embedded
    widget.

    Any widget, including complex widgets with children, and dialogs, can be
    embedded.

    Example:

    \code
    int main(int argc, char **argv)
    {
        QApplication app(argc, argv);

        QTabWidget *tabWidget = new QTabWidget;

        QGraphicsScene scene;
        QGraphicsProxyWidget *tabWidget = scene.addWidget(tabWidget);

        QGraphicsView view(&scene);
        view.show();

        return app.exec();
    }
    \endcode

    QGraphicsProxyWidget keeps its state in sync with the embedded widget
    (i.e., if the proxy is hidden, so is the embedded widget). As a widget is
    embedded by calling addWidget(), QGraphicsProxyWidget copies the state
    from the widget into the proxy, and after that, the two stay synchronized
    where possible. If the proxy moves or resizes, the embedded widget will
    also move or resize (i.e., the proxy and embedded widget's geometries are
    in sync), and vice versa. If you show the embedded widget, the proxy will
    also become visible. As a side effect, QGraphicsProxyWidget becomes hidden
    if you embed a hidden line edit.

    Example:

    \code
        QGraphicsScene scene;

        QLineEdit *edit = new QLineEdit;
        QGraphicsProxyWidget *proxy = scene.addWidget(edit);

        edit->isVisible();  // returns false, as QWidget is hidden by default
        proxy->isVisible(); // also returns false

        edit->show();

        edit->isVisible(); // returns true
        proxy->isVisible(); // returns true
    }
    \endcode

    QGraphicsProxyWidget maintains symmetry for the following states:

    \table
    \header \o QWidget state      \o QGraphicsProxyWidget state  \o Notes
    \row    \o QWidget::enabled   \o QGraphicsProxyWidget::enabled \o
    \row    \o QWidget::visible   \o QGraphicsProxyWidget::visible \o
    \row    \o QWidget::geometry  \o QGraphicsProxyWidget::geometry   \o Geometry is only guaranteed to be symmetric while the embedded widget is visible.
    \row    \o QWidget::layoutDirection \o QGraphicsProxyWidget::layoutDirection \o
    \row    \o QWidget::style \o QGraphicsProxyWidget::style \o
    \row    \o QWidget::palette \o QGraphicsProxyWidget::palette \o
    \row    \o QWidget::font \o QGraphicsProxyWidget::font \o
    \row    \o QWidget::cursor \o QGraphicsProxyWidget::cursor \o The proxy cursor changes based on which embedded subwidget is currently under the mouse.
    \row    \o QWidget::sizeHint() \o QGraphicsProxyWidget::sizeHint() \o All size hint functionality from the embedded widget is forwarded by the proxy.
    \row    \o QWidget::getContentsMargins() \o QGraphicsProxyWidget::getContentsMargins() \o Updated once by setWidget().
    \row    \o QWidget::windowTitle \o QGraphicsProxyWidget::windowTitle \o Updated once by setWidget().
    \endtable

    Note: QGraphicsScene keeps the embedded widget in a special state that
    prevents it from disturbing other widgets (both embedded and not embedded)
    while embedded. In this state, it may differ slightly in behavior from
    when it is not embedded.

    \sa QGraphicsScene::addWidget(), QGraphicsWidget
*/

extern bool qt_sendSpontaneousEvent(QObject *, QEvent *);

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::init()
{
    Q_Q(QGraphicsProxyWidget);
    q->setFocusPolicy(Qt::WheelFocus);
    q->setAcceptDrops(true);
}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::sendWidgetMouseEvent(QGraphicsSceneHoverEvent *event)
{
    QGraphicsSceneMouseEvent mouseEvent(QEvent::GraphicsSceneMouseMove);
    mouseEvent.setPos(event->pos());
    mouseEvent.setScreenPos(event->screenPos());
    mouseEvent.setButton(Qt::NoButton);
    mouseEvent.setButtons(0);
    mouseEvent.setModifiers(event->modifiers());
    sendWidgetMouseEvent(&mouseEvent);
    event->setAccepted(mouseEvent.isAccepted());
}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::sendWidgetMouseEvent(QGraphicsSceneMouseEvent *event)
{
    if (!event || !widget || !widget->isVisible())
        return;

    // Find widget position and receiver.
    QPointF pos = event->pos();
    QPointer<QWidget> alienWidget = widget->childAt(pos.toPoint());
    QPointer<QWidget> receiver =  alienWidget ? alienWidget : widget;

    // Translate QGraphicsSceneMouse events to QMouseEvents.
    QEvent::Type type = QEvent::None;
    switch (event->type()) {
    case QEvent::GraphicsSceneMousePress:
        type = QEvent::MouseButtonPress;
        if (!buttonDown)
            buttonDown = receiver;
        break;
    case QEvent::GraphicsSceneMouseRelease:
        type = QEvent::MouseButtonRelease;
        if (buttonDown)
            receiver = buttonDown;
        break;
    case QEvent::GraphicsSceneMouseDoubleClick:
        type = QEvent::MouseButtonDblClick;
        break;
    case QEvent::GraphicsSceneMouseMove:
        type = QEvent::MouseMove;
        if (buttonDown)
            receiver = buttonDown;
        break;
    default:
        Q_ASSERT_X(false, "QGraphicsProxyWidget", "internal error");
        break;
    }

    if (!lastWidgetUnderMouse) {
        QApplicationPrivate::dispatchEnterLeave(buttonDown ? buttonDown : widget, 0);
        lastWidgetUnderMouse = widget;
    }

    // Map event position from us to the receiver
    pos = mapToReceiver(pos, receiver);

    // Set click focus on the correct child.
    if (type == QEvent::MouseButtonPress) {
        QWidget *newFw = receiver;
        QWidget *fw = widget->focusWidget();
        do {
            if (newFw->isEnabled() && (newFw->focusPolicy() & Qt::ClickFocus)) {
                if (fw)
                    removeSubFocusHelper(fw, Qt::MouseFocusReason);
                setSubFocusHelper(newFw, Qt::MouseFocusReason);
                break;
            }
            newFw = newFw->parentWidget();
        } while (newFw);
    }

    // Send mouse event.
    QMouseEvent *mouseEvent = QMouseEvent::createExtendedMouseEvent(type, pos,
                                                                    receiver->mapToGlobal(pos.toPoint()), event->button(),
                                                                    event->buttons(), event->modifiers());

    QWidget *buttonDownPtr = (QWidget *)buttonDown;
    QApplicationPrivate::sendMouseEvent(receiver, mouseEvent, alienWidget, widget,
                                        &buttonDownPtr, lastWidgetUnderMouse);
    buttonDown = buttonDownPtr;

    // Update 'buttonDown'. We need to handle this ourself as the viewport is registered as the current
    // mouse grabber, thus we cannot do the "right" thing in QApplicationPrivate::sendMouseEvent.
    if (buttonDown && type == QEvent::MouseButtonRelease && !event->buttons()) {

        Q_Q(QGraphicsProxyWidget);
        if (q->rect().contains(event->pos()) && q->acceptsHoverEvents())
            lastWidgetUnderMouse = alienWidget ? alienWidget : widget;
        else // released on the frame our outside the item, or doesn't accept hover events.
            lastWidgetUnderMouse = 0;

        QApplicationPrivate::dispatchEnterLeave(lastWidgetUnderMouse, buttonDown);
        buttonDown = 0;

#ifndef QT_NO_CURSOR
        // ### Restore the cursor, don't override it.
        if (!lastWidgetUnderMouse)
            q->unsetCursor();
#endif
    }

    event->setAccepted(mouseEvent->isAccepted());
    delete mouseEvent;
}

void QGraphicsProxyWidgetPrivate::sendWidgetKeyEvent(QKeyEvent *event)
{
    if (!event || !widget || !widget->isVisible())
        return;

    QPointer<QWidget> receiver = widget->focusWidget();
    if (!receiver)
        receiver = widget;
    Q_ASSERT(receiver);

#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "   send to" << receiver;
#endif

    QApplication::sendEvent(receiver, event);

    // ### Remove, this should be done by proper focusIn/focusOut events.
    if ((event->key() == Qt::Key_Tab || event->key() == Qt::Key_Backtab)
         && receiver && !receiver->hasFocus()) {
        receiver->update();
        receiver = widget->focusWidget();
        if (receiver && receiver->hasFocus())
            receiver->update();
    }
}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::setSubFocusHelper(QWidget *widget, Qt::FocusReason reason)
{
    widget->setFocus(reason);
    QFocusEvent event(QEvent::FocusIn, reason);
    QApplication::sendEvent(widget, &event);

}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::removeSubFocusHelper(QWidget *widget, Qt::FocusReason reason)
{
    QFocusEvent event(QEvent::FocusOut, reason);
    QApplication::sendEvent(widget, &event);
}

/*!
    \internal
*/
QWidget *QGraphicsProxyWidgetPrivate::findFocusChild(QWidget *child, bool next) const
{
    // Run around the focus chain until we find a widget that can take tab focus.
    if (!child) {
	child = next ? (QWidget *)widget : widget->d_func()->focus_prev;
    } else {
        child = next ? child->d_func()->focus_next : child->d_func()->focus_prev;
        if ((next && child == widget) || (!next && child == widget->d_func()->focus_prev)) {
             return 0;
	}
    }

    QWidget *oldChild = child;
    do {
        if (child->isEnabled()
	    && child->isVisibleTo(widget)
            && (child->focusPolicy() & Qt::TabFocus)) {
	    return child;
        }
        child = next ? child->d_func()->focus_next : child->d_func()->focus_prev;
    } while (child != oldChild && !(next && child == widget) && !(!next && child == widget->d_func()->focus_prev));
    return 0;
}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::_q_removeWidgetSlot()
{
    Q_Q(QGraphicsProxyWidget);
    widget = 0;
    delete q;
}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::updateWidgetGeometryFromProxy()
{
}

/*!
    \internal
*/
void QGraphicsProxyWidgetPrivate::updateProxyGeometryFromWidget()
{
    Q_Q(QGraphicsProxyWidget);
    if (!widget)
        return;
    QRectF widgetGeometry = widget->geometry();
    if (widget->isWindow()) {
        if (QWidget *parentWidget = widget->parentWidget()) {
            // Translate the widget's position to parent coordinates.
            widgetGeometry.moveTo(parentWidget->mapFromGlobal(widget->pos()));
        } else {
            // ### screen coordinates, what to do? Assume scene coordinates?
            widgetGeometry.moveTo(q->mapFromScene(widget->pos()));
        }
    }
    if (widget->size().isValid()) {
        q->setGeometry(QRectF(widgetGeometry));
    } else {
        q->setGeometry(QRectF(widgetGeometry.topLeft(), widget->sizeHint()));
    }
}

/*!
    \internal

    Embeds \a subWin as a subwindow of this proxy widget. \a subWin must be a top-level
    widget and a descendant of the widget managed by this proxy. A separate subproxy
    will be created as a child of this proxy widget to manage \a subWin.
*/
void QGraphicsProxyWidgetPrivate::embedSubWindow(QWidget *subWin)
{
    QTLWExtra *extra;
    if (!((extra = subWin->d_func()->maybeTopData()) && extra->proxyWidget)) {
        QGraphicsProxyWidget *subProxy = new QGraphicsProxyWidget(q_func());
        subProxy->setWidget(subWin);
    } else
        extra->proxyWidget->d_func()->updateProxyGeometryFromWidget();
}

/*!
     \internal
*/
QPointF QGraphicsProxyWidgetPrivate::mapToReceiver(const QPointF &pos, const QWidget *receiver) const
{
    QPointF p = pos;
    // Map event position from us to the receiver, preserving its
    // precision (don't use QWidget::mapFrom here).
    while (receiver && receiver != widget) {
        p -= QPointF(receiver->pos());
        receiver = receiver->parentWidget();
    }
    return p;
}

/*!
    Constructs a new QGraphicsProxy widget. \a parent and \a wFlags are passed
    to QGraphicsItem's constructor.
*/
QGraphicsProxyWidget::QGraphicsProxyWidget(QGraphicsItem *parent, Qt::WindowFlags wFlags)
    : QGraphicsWidget(*new QGraphicsProxyWidgetPrivate, parent, 0, wFlags)
{
    Q_D(QGraphicsProxyWidget);
    d->init();
}

/*!
    Destroys the proxy widget, and any embedded widget.
*/
QGraphicsProxyWidget::~QGraphicsProxyWidget()
{
    Q_D(QGraphicsProxyWidget);
    if (d->widget) {
        QWidget *w = d->widget;
        setWidget(0);
        delete w;
    }
}

/*!
    Embeds \a widget into this proxy widget. \a widget must be a top-level
    widget whose parent is 0.

    You can also pass 0 for the \a widget argument. This will remove the widget
    from this proxy, and pass ownership of the currently embedded widget to
    the caller.

    \sa widget()
*/
void QGraphicsProxyWidget::setWidget(QWidget *widget)
{
    Q_D(QGraphicsProxyWidget);
    if (widget == d->widget)
        return;
    if (d->widget) {
        disconnect(d->widget, SIGNAL(destroyed()), this, SLOT(_q_removeWidgetSlot()));
        d->widget->removeEventFilter(this);
        d->widget->setAttribute(Qt::WA_GraphicsWidget, false);
        d->widget->d_func()->extra->topextra->proxyWidget = 0;
        d->widget = 0;
#ifdef Q_WS_MAC
        d->pixmap = QPixmap();
#endif
#ifndef QT_NO_CURSOR
        unsetCursor();
#endif
        setAcceptHoverEvents(false);
    }
    if (!widget)
        return;
    if (!widget->isWindow()) {
        qWarning("QGraphicsProxyWidget::setWidget: cannot embed widget %p"
                 " that is not a window", widget);
        return;
    }

    // Register this proxy within the widget's private.
    // ### This is a bit backdoorish
    if (QTLWExtra *extra = widget->d_func()->topData()) {
        QGraphicsProxyWidget **proxyWidget = &extra->proxyWidget;
        if (*proxyWidget) {
            if (*proxyWidget != this) {
                qWarning("QGraphicsProxyWidget::setWidget: cannot embed widget %p"
                         "; already embedded", widget);
            }
            return;
        }
        *proxyWidget = this;
    }

    widget->setAttribute(Qt::WA_GraphicsWidget);
    widget->ensurePolished();
    // Do not wait for this widget to close before the app closes ###
    // shouldn't this widget inherit the attribute?
    widget->setAttribute(Qt::WA_QuitOnClose, false);
    setAcceptHoverEvents(true);

    d->widget = widget;
#ifdef Q_WS_MAC
    d->pixmap = QPixmap(d->widget->size());
#endif

    // Changes only go from the widget to the proxy.
    d->enabledChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
    d->visibleChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
    d->posChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
    d->sizeChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;

    // Copy the state from the widget onto the proxy.
#ifndef QT_NO_CURSOR
    if (widget->testAttribute(Qt::WA_SetCursor))
        setCursor(widget->cursor());
#endif
    setEnabled(widget->isEnabled());
    setVisible(widget->isVisible());
    setLayoutDirection(widget->layoutDirection());
    if (widget->testAttribute(Qt::WA_SetStyle))
        setStyle(widget->style());
    if (widget->testAttribute(Qt::WA_SetFont))
        setFont(widget->font());
    if (widget->testAttribute(Qt::WA_SetPalette))
        setPalette(widget->palette());
    setMinimumSize(widget->minimumSize());
    setMaximumSize(widget->maximumSize());
    if (!widget->testAttribute(Qt::WA_Resized))
        widget->adjustSize();
    setGeometry(widget->geometry());

    int left, top, right, bottom;
    widget->getContentsMargins(&left, &top, &right, &bottom);
    setContentsMargins(left, top, right, bottom);
    setWindowTitle(widget->windowTitle());

    // Hook up the event filter to keep the state up to date.
    widget->installEventFilter(this);
    connect(widget, SIGNAL(destroyed()), this, SLOT(_q_removeWidgetSlot()));

    // Changes no longer go only from the widget to the proxy.
    d->enabledChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
    d->visibleChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
    d->posChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
    d->sizeChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
}

/*!
    Returns a pointer to the embedded widget.

    \sa setWidget()
*/
QWidget *QGraphicsProxyWidget::widget() const
{
    Q_D(const QGraphicsProxyWidget);
    return d->widget;
}

/*!
    Returns the rectangle for \a widget, which must be a descendent of
    widget(), or widget() itself, in this proxy item's local coordinates.

    If \a widget is 0, this function returns an empty QRectF.

    \sa widget()
*/
QRectF QGraphicsProxyWidget::subWidgetRect(const QWidget *widget) const
{
    Q_D(const QGraphicsProxyWidget);
    if (!widget)
        return QRectF();
    if (d->widget == widget || d->widget->isAncestorOf(widget))
        return QRectF(widget->mapTo(d->widget, QPoint(0, 0)), widget->size());
    return QRectF();
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::setGeometry(const QRectF &rect)
{
    Q_D(QGraphicsProxyWidget);
    bool proxyResizesWidget = !d->posChangeMode && !d->sizeChangeMode;
    if (proxyResizesWidget) {
        d->posChangeMode = QGraphicsProxyWidgetPrivate::ProxyToWidgetMode;
        d->sizeChangeMode = QGraphicsProxyWidgetPrivate::ProxyToWidgetMode;
    }
    QGraphicsWidget::setGeometry(rect);
    if (proxyResizesWidget) {
        d->posChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
        d->sizeChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
    }
}

/*!
    \reimp
*/
QVariant QGraphicsProxyWidget::itemChange(GraphicsItemChange change,
                                          const QVariant &value)
{
    Q_D(QGraphicsProxyWidget);

    switch (change) {
    case ItemPositionChange:
        // The item's position is either changed directly on the proxy, in
        // which case the position change should propagate to the widget,
        // otherwise it happens as a side effect when filtering QEvent::Move.
        if (!d->posChangeMode)
            d->posChangeMode = QGraphicsProxyWidgetPrivate::ProxyToWidgetMode;
        break;
    case ItemPositionHasChanged:
        // Move the internal widget if we're in widget-to-proxy
        // mode. Otherwise the widget has already moved.
        if (d->widget && d->posChangeMode != QGraphicsProxyWidgetPrivate::WidgetToProxyMode)
            d->widget->move(value.toPoint());
        if (d->posChangeMode == QGraphicsProxyWidgetPrivate::ProxyToWidgetMode)
            d->posChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
        break;
    case ItemVisibleChange:
        if (!d->visibleChangeMode)
            d->visibleChangeMode = QGraphicsProxyWidgetPrivate::ProxyToWidgetMode;
        break;
    case ItemVisibleHasChanged:
        if (d->widget && d->visibleChangeMode != QGraphicsProxyWidgetPrivate::WidgetToProxyMode)
            d->widget->setVisible(isVisible());
        if (d->visibleChangeMode == QGraphicsProxyWidgetPrivate::ProxyToWidgetMode)
            d->visibleChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
        break;
    case ItemEnabledChange:
        if (!d->enabledChangeMode)
            d->enabledChangeMode = QGraphicsProxyWidgetPrivate::ProxyToWidgetMode;
        break;
    case ItemEnabledHasChanged:
        if (d->widget && d->enabledChangeMode != QGraphicsProxyWidgetPrivate::WidgetToProxyMode)
            d->widget->setEnabled(isEnabled());
        if (d->enabledChangeMode == QGraphicsProxyWidgetPrivate::ProxyToWidgetMode)
            d->enabledChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
        break;
    default:
        break;
    }
    return QGraphicsWidget::itemChange(change, value);
}

/*!
    \reimp
*/
bool QGraphicsProxyWidget::eventFilter(QObject *object, QEvent *event)
{
    Q_D(QGraphicsProxyWidget);

    if (object == d->widget) {
        switch (event->type()) {
         case QEvent::Resize:
             // If the widget resizes itself, we resize the proxy too.
             // Prevent feed-back by checking the geometry change mode.
             if (!d->sizeChangeMode) {
                 d->sizeChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
                 resize((static_cast<QResizeEvent *>(event))->size());
                 d->sizeChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
             }
            break;
         case QEvent::Move:
             // If the widget moves itself, we move the proxy too.  Prevent
             // feed-back by checking the geometry change mode.
             if (!d->posChangeMode) {
                 d->posChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
                 setPos((static_cast<QMoveEvent *>(event))->pos());
                 d->posChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
             }
            break;
        case QEvent::Hide:
        case QEvent::Show:
            // If the widget toggles its visible state, the proxy will follow.
            if (!d->visibleChangeMode) {
                d->visibleChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
                setVisible(event->type() == QEvent::Show);
                d->visibleChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
            }
            break;
        case QEvent::EnabledChange:
            // If the widget toggles its enabled state, the proxy will follow.
            if (!d->enabledChangeMode) {
                d->enabledChangeMode = QGraphicsProxyWidgetPrivate::WidgetToProxyMode;
                setEnabled(d->widget->isEnabled());
                d->enabledChangeMode = QGraphicsProxyWidgetPrivate::NoMode;
            }
            break;
        case QEvent::UpdateRequest: {
#ifndef Q_WS_WIN
            extern QRegion qt_dirtyRegion(QWidget *);
#endif
            const QVector<QRect> rects = qt_dirtyRegion(d->widget).rects();
            for (int i = 0; i < rects.size(); ++i)
                update(rects.at(i));
            break;
        }
        default:
            break;
        }
    }
    return QGraphicsWidget::eventFilter(object, event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::showEvent(QShowEvent *event)
{
    Q_UNUSED(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::hideEvent(QHideEvent *event)
{
    Q_UNUSED(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::hoverEnterEvent(QGraphicsSceneHoverEvent *event)
{
    Q_UNUSED(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::hoverLeaveEvent(QGraphicsSceneHoverEvent *event)
{
    Q_UNUSED(event);
    Q_D(QGraphicsProxyWidget);
    // If hoverMove was compressed away, make sure we update properly here.
    if (d->lastWidgetUnderMouse && !d->buttonDown) {
        QApplicationPrivate::dispatchEnterLeave(0, d->lastWidgetUnderMouse);
        d->lastWidgetUnderMouse = 0;
    }
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::hoverMoveEvent(QGraphicsSceneHoverEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::hoverMoveEvent";
#endif

    // Ingore events on the window frame.
    if (!d->widget || !rect().contains(event->pos())) {
        if (d->lastWidgetUnderMouse && !d->buttonDown) {
            QApplicationPrivate::dispatchEnterLeave(0, d->lastWidgetUnderMouse);
            d->lastWidgetUnderMouse = 0;
        }
        return;
    }

    d->sendWidgetMouseEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::mouseMoveEvent";
#endif
    d->sendWidgetMouseEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::mousePressEvent";
#endif
    d->sendWidgetMouseEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::mouseDoubleClickEvent";
#endif
    d->sendWidgetMouseEvent(event);
}

/*!
    \reimp
*/
#ifndef QT_NO_WHEELEVENT
void QGraphicsProxyWidget::wheelEvent(QGraphicsSceneWheelEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::wheelEvent";
#endif
    if (!d->widget)
        return;

    QPointF pos = event->pos();
    QPointer<QWidget> receiver = d->widget->childAt(pos.toPoint());

    // Map event position from us to the receiver
    pos = d->mapToReceiver(pos, receiver);

    // Send mouse event.
    QWheelEvent wheelEvent(pos.toPoint(), event->screenPos(), event->delta(),
                           event->buttons(), event->modifiers(), event->orientation());
    QPointer<QWidget> focusWidget = d->widget->focusWidget();
    extern bool qt_sendSpontaneousEvent(QObject *, QEvent *);
    qt_sendSpontaneousEvent(receiver, &wheelEvent);
    event->setAccepted(wheelEvent.isAccepted());

    // ### Remove, this should be done by proper focusIn/focusOut events.
    if (focusWidget && !focusWidget->hasFocus()) {
        focusWidget->update();
        focusWidget = d->widget->focusWidget();
        if (focusWidget && focusWidget->hasFocus())
            focusWidget->update();
    }
}
#endif

/*!
    \reimp
*/
void QGraphicsProxyWidget::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::mouseReleaseEvent";
#endif
    d->sendWidgetMouseEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::keyPressEvent(QKeyEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::keyPressEvent";
#endif
    d->sendWidgetKeyEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::keyReleaseEvent(QKeyEvent *event)
{
    Q_D(QGraphicsProxyWidget);
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::keyReleaseEvent";
#endif
    d->sendWidgetKeyEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::focusInEvent(QFocusEvent *event)
{
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::focusInEvent";
#endif
    Q_D(QGraphicsProxyWidget);
    QGraphicsWidget::focusInEvent(event);

    switch (event->reason()) {
    case Qt::TabFocusReason: {
	if (QWidget *focusChild = d->findFocusChild(0, true))
	    d->setSubFocusHelper(focusChild, event->reason());
        break;
    }
    case Qt::BacktabFocusReason:
	if (QWidget *focusChild = d->findFocusChild(0, false))
	    d->setSubFocusHelper(focusChild, event->reason());
        break;
    default:
	if (d->widget && d->widget->focusWidget()) {
	    d->setSubFocusHelper(d->widget->focusWidget(), event->reason());
	    return;
	}
	break;
    }
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::focusOutEvent(QFocusEvent *event)
{
#ifdef GRAPHICSPROXYWIDGET_DEBUG
    qDebug() << "QGraphicsProxyWidget::focusOutEvent";
#endif
    Q_D(QGraphicsProxyWidget);
    QGraphicsWidget::focusOutEvent(event);
    if (d->widget) {
	if (QWidget *focusWidget = d->widget->focusWidget())
	    d->removeSubFocusHelper(focusWidget, event->reason());
    }
}

/*!
    \reimp
*/
bool QGraphicsProxyWidget::focusNextPrevChild(bool next)
{
    Q_D(QGraphicsProxyWidget);
    if (!d->widget || !d->scene)
        return QGraphicsWidget::focusNextPrevChild(next);

    Qt::FocusReason reason = next ? Qt::TabFocusReason : Qt::BacktabFocusReason;
    QWidget *lastFocusChild = d->widget->focusWidget();
    if (QWidget *newFocusChild = d->findFocusChild(lastFocusChild, next)) {
        if (lastFocusChild)
	    d->removeSubFocusHelper(lastFocusChild, reason);
	d->setSubFocusHelper(newFocusChild, reason);
	return true;
    }

    return QGraphicsWidget::focusNextPrevChild(next);
}

/*!
    \reimp
*/
QSizeF QGraphicsProxyWidget::sizeHint(Qt::SizeHint which, const QSizeF &constraint) const
{
    Q_D(const QGraphicsProxyWidget);
    if (!d->widget)
        return QGraphicsWidget::sizeHint(which, constraint);

    // ### what about constraint and widthForHeight()?

    QSizeF sh;
    switch (which) {
    case Qt::PreferredSize:
        sh = d->widget->sizeHint();
        break;
    case Qt::MinimumSize:
        sh = d->widget->minimumSizeHint();
        break;
    case Qt::MaximumSize:
        sh = QSizeF(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX);
        break;
    case Qt::MinimumDescent:
        sh = constraint;
        break;
    default:
        break;
    }
    return sh;
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    Q_D(QGraphicsProxyWidget);
    if (d->widget) {
        if (d->sizeChangeMode != QGraphicsProxyWidgetPrivate::WidgetToProxyMode)
            d->widget->resize(event->newSize().toSize());
    }
    QGraphicsWidget::resizeEvent(event);
}

/*!
    \reimp
*/
void QGraphicsProxyWidget::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_D(QGraphicsProxyWidget);
    Q_UNUSED(widget);
    if (!d->widget || !d->widget->isVisible())
        return;

    // Filter out repaints on the window frame.
    const QRect exposedWidgetRect = (option->exposedRect & rect()).toRect();
    if (exposedWidgetRect.isEmpty())
        return;

#ifndef Q_WS_MAC
    // Disable QPainter's default pen being cosmetic. This allows widgets and
    // styles to follow Qt's existing defaults without getting ugly cosmetic
    // lines when scaled.
    bool restore = painter->renderHints() & QPainter::NonCosmeticDefaultPen;
    painter->setRenderHints(QPainter::NonCosmeticDefaultPen, true);

    d->widget->render(painter, QPoint(), exposedWidgetRect);

    // Restore the render hints if necessary.
    if (restore)
        painter->setRenderHints(QPainter::NonCosmeticDefaultPen, false);
#else
    const bool widgetSizeChanged = d->pixmap.size() != d->widget->size();
    const QRect exposedRect = widgetSizeChanged ? option->rect : exposedWidgetRect;

    if (widgetSizeChanged)
        d->pixmap = QPixmap(d->widget->size());

    d->widget->render(&d->pixmap, QPoint(), option->rect);
    painter->drawPixmap(exposedRect, d->pixmap, exposedRect);
#endif
}

/*!
    \reimp
*/
int QGraphicsProxyWidget::type() const
{
    return Type;
}

QT_END_NAMESPACE

#include "moc_qgraphicsproxywidget.cpp"
        
#endif //QT_NO_GRAPHICSVIEW
