/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the example classes of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <QtGui>
#include <QtUiTools>
#include <QtXmlPatterns>

#include "querymainwindow.h"

QueryMainWindow::QueryMainWindow()
{
    queryWidget = loadUiFile();

    ui_inputTextEdit = qFindChild<QTextEdit*>(this, "inputTextEdit");
    ui_outputTextEdit = qFindChild<QTextEdit*>(this, "outputTextEdit");
    ui_queryTextEdit = qFindChild<QTextEdit*>(this, "queryTextEdit");

    ui_defaultQueries = qFindChild<QComboBox*>(this, "defaultQueries");

    QMetaObject::connectSlotsByName(this);

    connect(ui_defaultQueries, SIGNAL(currentIndexChanged(int)),
        this, SLOT(displayQuery(int)));

    inputFile.setFileName(":/files/cookbook.xml");
    loadInputFile();
    setCentralWidget(queryWidget);
    queryWidget->show();

    setWindowTitle(tr("Recipes XQuery Example"));
}

QWidget* QueryMainWindow::loadUiFile()
{
    QUiLoader loader;
    QFile file(":/forms/querywidget.ui");
    file.open(QFile::ReadOnly);

    QWidget *formWidget = loader.load(&file, this);
    file.close();

    return formWidget;
}

QSize QueryMainWindow::sizeHint() const
{
    return QSize(700, 430);
}

QString QueryMainWindow::indexToQuery(int index)
{
    QString queryString = "";

    switch(index) {
        case 1: // All recipes
            queryString = "declare variable $inputDocument external;\n"
                "doc($inputDocument)/cookbook/recipe/<p>{title/text()}</p>";
            break;
        case 2: // Mushroom soup
            queryString = "declare variable $inputDocument external;\n"
                "doc($inputDocument)/cookbook/recipe[@xml:id = \"MushroomSoup\"]/ingredient/<p>{@name, @quantity}</p>";
            break;
        case 3: // Liquid ingredients in mushroom soup
            queryString = "declare variable $inputDocument external;\n"
                "doc($inputDocument)/cookbook/recipe[@xml:id = \"MushroomSoup\"]/ingredient[@unit = \"milliliters\"]/<p>{@name, @quantity, @unit}</p>";
            break;
        case 4: // All recipes with preparation times
            queryString = "declare variable $inputDocument external;\n"
                "doc($inputDocument)/cookbook/recipe/\n"
                "<recipe title=\"{title/text()}\" time=\"{time/@quantity} {time/@unit}\" />";
            break;
        case 5: // All recipes with preparation times of 30 minutes or less
            queryString = "declare variable $inputDocument external;\n"
                "doc($inputDocument)/cookbook/recipe/time[@unit = \"minutes\" and xs:integer(@quantity) <= 10]/\n"
                "<p>\n"
                "{\n"
                "  fn:concat(../title/text(), ' (', "
                "xs:string(@quantity), ' ', xs:string(@unit), "
                "xs:string(')'))\n"
                "}\n"
                "</p>";
            break;
        default:
            break;
    }

    return queryString;
}

void QueryMainWindow::displayQuery(int index)
{
    QString displayString = indexToQuery(index);
    ui_queryTextEdit->clear();
    ui_queryTextEdit->setPlainText(displayString);

    evaluate(displayString);
}

void QueryMainWindow::loadInputFile()
{
    if (!inputFile.open(QIODevice::ReadOnly)) {
        QMessageBox::information(this,
            tr("Unable to open file"), inputFile.errorString());
        return;
    }

    QTextStream in(&inputFile);
    QString inputDocument = in.readAll();
    ui_inputTextEdit->setPlainText(inputDocument);
}

void QueryMainWindow::evaluate(const QString &str)
{
    /* This function takes the input string and displays the
     * appropriate output using QXmlQuery.
     */
    QXmlQuery query;
    query.setQuery(str);

    /* Since loadInputFile() reads the device to the very end, we will not get
     * anything if we try to read from it. Therefore we need to call QIODevice::reset(),
     * such that it is "rewinded". */
    inputFile.reset();

    Q_ASSERT(inputFile.isOpen());

    query.bindVariable("inputDocument", &inputFile);

    QByteArray outArray;
    QBuffer buffer(&outArray);
    buffer.open(QIODevice::ReadWrite);
    
    QXmlFormatterPointer formatter(new QXmlFormatter(query, &buffer));
    query.evaluateToReceiver(formatter);
 
    buffer.close();
    ui_outputTextEdit->setPlainText(QString::fromUtf8(outArray.constData()));
    
}
