/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may only be used for
** evaluation and testing purposes.  It may not be used for commercial
** development.  You may use this file in accordance with the terms and
** conditions contained in the either Technology Preview License
** Agreement or the Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "qdynamiccontextstore_p.h"
#include "qpatternistlocale_p.h"

#include "qtemplate_p.h"

QT_BEGIN_NAMESPACE

using namespace QPatternist;

const SourceLocationReflection* Template::actualReflection() const
{
    return this;
}

DynamicContext::Ptr Template::bindParameters(const TemplateInvoker *const invoker,
                                             const DynamicContext::Ptr &context,
                                             const bool isCallTemplate) const
{
    Q_ASSERT(invoker);
    Q_ASSERT(context);

    /* We have:
     * - xsl:params in the target template (if any) which may provide
     *   default values.
     * - xsl:with-params in the caller (if any) which provides values.
     *
     * We need to, for each parameter:
     * - If the called template provides no default value and the caller
     *   has no value, it's an error
     * - If the called template has a default value and the caller provides
     *   none, it should be used
     * - In any case the caller provides a value, it needs to be used.
     *
     * Problems to look out for:
     *
     * - Each xsl:param is in scope for the subsequent xsl:params. Hence,
     *   the evaluation of one xsl:param can depend on another xsl:param,
     *   and so on
     * - The focus for xsl:params is different from the focus for
     *   the xsl:with-params
     * - The xsl:with-params are not in scope for the xsl:params.
     */

    WithParam::Hash withParams(invoker->withParams());

    /**
     * Parameters or not, we must in any case create a new stack frame
     * for the template invocation since otherwise we will trash our existing
     * variables. Hence it's as with calling user functions.
     *
     * This is especially reproducible with recursive functions.
     */
    DynamicContext::Ptr newStack(context->createStack());

    /* We have no parameters, and we have no further error checking to
     * do in the case of not being xsl:apply-templates, so we need to do nothing. */
    if(templateParameters.isEmpty() && (!isCallTemplate || withParams.isEmpty()))
        return newStack;

    DynamicContext::TemplateParameterHash sewnTogether(templateParameters);

    const DynamicContext::TemplateParameterHash::iterator end(sewnTogether.end());

    for(DynamicContext::TemplateParameterHash::iterator it(sewnTogether.begin());
        it != end;
        ++it)
    {
        Expression::Ptr &param = it.value();
        WithParam::Ptr &withParam = withParams[it.key()];

        if(withParam)
            param = Expression::Ptr(new DynamicContextStore(withParam->defaultValue(), context));
        else if(!param)
        {
            /* Ops, no xsl:with-param and no default value to cover up for it.
             */
            context->error(QtXmlPatterns::tr("The parameter %1 is required, but no corresponding %2 is supplied.")
                                             .arg(formatKeyword(context->namePool(), it.key()),
                                                  formatKeyword(QLatin1String("xsl:with-param"))),
                           ReportContext::XTSE0690,
                           this);
        }
    }

    if(isCallTemplate)
    {
        /* Find xsl:with-param that has no corresponding xsl:param. */
        /* Optimization: candidate for threading? */

        const WithParam::Hash::const_iterator end(withParams.constEnd());

        for(WithParam::Hash::const_iterator it(withParams.constBegin()); it != end; ++it)
        {
            if(!templateParameters.contains(it.key()))
            {
                        context->error(QtXmlPatterns::tr("The parameter %1 is passed, but no corresponding %2 exists.")
                                                         .arg(formatKeyword(context->namePool(), it.key()),
                                                              formatKeyword(QLatin1String("xsl:param"))),
                                       ReportContext::XTSE0680,
                                       this);
            }
        }

    }

    newStack->templateParameterStore() = sewnTogether;
    return newStack;
}

void Template::compileParameters(const StaticContext::Ptr &context)
{
    Q_ASSERT(context);

    const DynamicContext::TemplateParameterHash::iterator end(templateParameters.end());

    for(DynamicContext::TemplateParameterHash::iterator it(templateParameters.begin());
        it != end;
        ++it)
    {
        Expression::Ptr &val = it.value();

        /* If our value is required, we don't have a default value. */
        if(val)
        {
            val = val->typeCheck(context, val->staticType());
            val = val->compress(context);
        }
    }
}

Expression::Properties Template::properties() const
{
    Expression::Properties collect(body->properties());

    DynamicContext::TemplateParameterHash::const_iterator end(templateParameters.constEnd());

    for(DynamicContext::TemplateParameterHash::const_iterator it(templateParameters.constBegin());
        it != end;
        ++it)
    {
        if(it.value())
            collect |= it.value()->properties();
    }

    // TODO simplify.
    return collect & (Expression::RequiresFocus | Expression::IsEvaluated | Expression::DisableElimination);
}

Expression::Properties Template::dependencies() const
{
    Expression::Properties collect(body->dependencies());

    DynamicContext::TemplateParameterHash::const_iterator end(templateParameters.constEnd());

    for(DynamicContext::TemplateParameterHash::const_iterator it(templateParameters.constBegin());
        it != end;
        ++it)
    {
        if(it.value())
            collect |= it.value()->dependencies();
    }

    return collect & (Expression::RequiresFocus | Expression::IsEvaluated | Expression::DisableElimination);
}

QT_END_NAMESPACE
