/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtScriptTools module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may only be used for
** evaluation and testing purposes.  It may not be used for commercial
** development.  You may use this file in accordance with the terms and
** conditions contained in the either Technology Preview License
** Agreement or the Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "qscriptdebuggerfrontend_p.h"
#include "qscriptdebuggerfrontend_p_p.h"
#include "qscriptdebuggercommand_p.h"
#include "qscriptdebuggerevent_p.h"
#include "qscriptdebuggerresponse_p.h"
#include "qscriptdebuggereventhandlerinterface_p.h"
#include "qscriptdebuggerresponsehandlerinterface_p.h"

#include <QtCore/qcoreevent.h>
#include <QtCore/qcoreapplication.h>

QT_BEGIN_NAMESPACE

/*!
  \class QScriptDebuggerFrontend
  \since 4.5
  \internal

  \brief The QScriptDebuggerFrontend class is the base class of debugger front-ends.
*/

// helper class that's used to handle our custom Qt events
class QScriptDebuggerFrontendEventReceiver : public QObject
{
public:
    QScriptDebuggerFrontendEventReceiver(QScriptDebuggerFrontendPrivate *frontend,
                                         QObject *parent = 0);
    ~QScriptDebuggerFrontendEventReceiver();

    bool event(QEvent *);

private:
    QScriptDebuggerFrontendPrivate *m_frontend;
};

QScriptDebuggerFrontendEventReceiver::QScriptDebuggerFrontendEventReceiver(
    QScriptDebuggerFrontendPrivate *frontend, QObject *parent)
    : QObject(parent), m_frontend(frontend)
{
}
    
QScriptDebuggerFrontendEventReceiver::~QScriptDebuggerFrontendEventReceiver()
{
}

bool QScriptDebuggerFrontendEventReceiver::event(QEvent *e)
{
    return m_frontend->event(e);
}


QScriptDebuggerFrontendPrivate::QScriptDebuggerFrontendPrivate()
{
    eventHandler = 0;
    eventReceiver = 0;
    nextCommandId = 0;
}

QScriptDebuggerFrontendPrivate::~QScriptDebuggerFrontendPrivate()
{
    delete eventReceiver;
}

void QScriptDebuggerFrontendPrivate::postEvent(QEvent *e)
{
    if (!eventReceiver)
        eventReceiver = new QScriptDebuggerFrontendEventReceiver(this);
    QCoreApplication::postEvent(eventReceiver, e);
}

bool QScriptDebuggerFrontendPrivate::event(QEvent *e)
{
    if (e->type() == QEvent::User+1) {
        processCommands();
        return true;
    }
    return false;
}

void QScriptDebuggerFrontendPrivate::processCommands()
{
    Q_Q(QScriptDebuggerFrontend);
    while (!pendingCommands.isEmpty()) {
        QScriptDebuggerCommand command(pendingCommands.takeFirst());
        int id = pendingCommandIds.takeFirst();
        q->processCommand(id, command);
    }
}

QScriptDebuggerFrontend::QScriptDebuggerFrontend()
    : d_ptr(new QScriptDebuggerFrontendPrivate())
{
    d_ptr->q_ptr = this;
}

QScriptDebuggerFrontend::~QScriptDebuggerFrontend()
{
    delete d_ptr;
}

QScriptDebuggerFrontend::QScriptDebuggerFrontend(QScriptDebuggerFrontendPrivate &dd)
    : d_ptr(&dd)
{
    d_ptr->q_ptr = this;
}

QScriptDebuggerEventHandlerInterface *QScriptDebuggerFrontend::eventHandler() const
{
    Q_D(const QScriptDebuggerFrontend);
    return d->eventHandler;
}

void QScriptDebuggerFrontend::setEventHandler(QScriptDebuggerEventHandlerInterface *eventHandler)
{
    Q_D(QScriptDebuggerFrontend);
    d->eventHandler = eventHandler;
}

/*!
  Schedules the given \a command for execution by this front-end,
  and returns a unique identifier associated with this command.

  Subclasses can call this function to schedule custom commands.

  \sa notifyCommandFinished()
*/
int QScriptDebuggerFrontend::scheduleCommand(
    const QScriptDebuggerCommand &command,
    QScriptDebuggerResponseHandlerInterface *responseHandler)
{
    Q_D(QScriptDebuggerFrontend);
    int id = ++d->nextCommandId;
    d->pendingCommands.append(command);
    d->pendingCommandIds.append(id);
    if (responseHandler)
        d->responseHandlers.insert(id, responseHandler);
    if (d->pendingCommands.size() == 1) {
        QEvent *e = new QEvent(QEvent::Type(QEvent::User+1));
        d->postEvent(e);
    }
    return id;
}

/*!
  Subclasses should call this function when the command identified by
  the given \a id has finished and produced the given \a response.

  \sa processCommand(), notifyEvent()
*/
void QScriptDebuggerFrontend::notifyCommandFinished(int id, const QScriptDebuggerResponse &response)
{
    Q_D(QScriptDebuggerFrontend);
    if (d->responseHandlers.contains(id)) {
        QScriptDebuggerResponseHandlerInterface *handler = d->responseHandlers.take(id);
        Q_ASSERT(handler != 0);
        handler->handleResponse(response, id);
    }
}

/*!
  Subclasses should call this function when the given \a event is
  received from the back-end.

  \sa notifyCommandFinished(), QScriptDebuggerBackend::event()
*/
bool QScriptDebuggerFrontend::notifyEvent(const QScriptDebuggerEvent &event)
{
    Q_D(QScriptDebuggerFrontend);
    if (d->eventHandler)
        return d->eventHandler->debuggerEvent(event);
    return false;
}

int QScriptDebuggerFrontend::scheduledCommandCount() const
{
    Q_D(const QScriptDebuggerFrontend);
    return d->nextCommandId;
}

/*!
  \fn void processCommand(int id, const QScriptDebuggerCommand &command)

  Subclasses must reimplement this function to process the given \a command
  identified by \a id. Call notifyCommandFinished() when processing is
  complete.
*/

QT_END_NAMESPACE
