/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtOpenGL module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the either Technology Preview License Agreement or the
** Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include <private/qdrawhelper_p.h>
#include <private/qgl_p.h>

#include "qglgradientcache_p.h"

void QGLGradientCache::cleanCache() {
    QGLGradientColorTableHash::const_iterator it = cache.constBegin();
    for (; it != cache.constEnd(); ++it) {
        const CacheInfo &cache_info = it.value();
        glDeleteTextures(1, &cache_info.texId);
    }
    cache.clear();
}

GLuint QGLGradientCache::getBuffer(const QGradientStops &stops, qreal opacity, const QGLContext *ctx)
{
    if (buffer_ctx && !qgl_share_reg()->checkSharing(buffer_ctx, ctx))
        cleanCache();

    buffer_ctx = ctx;

    quint64 hash_val = 0;

    for (int i = 0; i < stops.size() && i <= 2; i++)
        hash_val += stops[i].second.rgba();

    QGLGradientColorTableHash::const_iterator it = cache.constFind(hash_val);

    if (it == cache.constEnd())
        return addCacheElement(hash_val, stops, opacity);
    else {
        do {
            const CacheInfo &cache_info = it.value();
            if (cache_info.stops == stops && cache_info.opacity == opacity) {
                return cache_info.texId;
            }
            ++it;
        } while (it != cache.constEnd() && it.key() == hash_val);
        // an exact match for these stops and opacity was not found, create new cache
        return addCacheElement(hash_val, stops, opacity);
    }
}


GLuint QGLGradientCache::addCacheElement(quint64 hash_val, const QGradientStops &stops, qreal opacity)
{
    if (cache.size() == maxCacheSize()) {
        int elem_to_remove = qrand() % maxCacheSize();
        quint64 key = cache.keys()[elem_to_remove];

        // need to call glDeleteTextures on each removed cache entry:
        QGLGradientColorTableHash::const_iterator it = cache.constFind(key);
        do {
            glDeleteTextures(1, &it.value().texId);
        } while (++it != cache.constEnd() && it.key() == key);
        cache.remove(key); // may remove more than 1, but OK
    }


    CacheInfo cache_entry(stops, opacity);
    uint buffer[1024];
    generateGradientColorTable(stops, buffer, paletteSize(), opacity);
    glGenTextures(1, &cache_entry.texId);
    glBindTexture(GL_TEXTURE_2D, cache_entry.texId);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, paletteSize(), 1,
                    0, GL_RGBA, GL_UNSIGNED_BYTE, buffer);
    return cache.insert(hash_val, cache_entry).value().texId;
}


// GL's expects pixels in RGBA, bin-endian (ABGR on x86).
// Qt stores in ARGB using whatever byte-order the mancine uses.
static inline uint qtToGlColor(uint c)
{
    uint o;
#if Q_BYTE_ORDER == Q_LITTLE_ENDIAN
    o = c & 0xFF00FF00; // alpha & green already in the right place
    o |= (c >> 16) & 0x000000FF; // red
    o |= (c << 16) & 0x00FF0000; // blue

#else //Q_BIG_ENDIAN
    o = c & 0x00FF00FF; // alpha & green already in the right place
    o |= (c << 16) & 0xFF000000; // red
    o |= (c >> 16) & 0x0000FF00; // blue
#error big endian not tested with QGLGraphicsContext
#endif // Q_BYTE_ORDER
    return o;
}


//TODO: Let GL generate the texture using an FBO
void QGLGradientCache::generateGradientColorTable(const QGradientStops& s, uint *colorTable, int size, qreal opacity) const
{
    int pos = 0;
    qreal fpos = 0.0;
    qreal incr = 1.0 / qreal(size);
    QVector<uint> colors(s.size());

    for (int i = 0; i < s.size(); ++i)
        colors[i] = s[i].second.rgba(); // Qt LIES! It returns ARGB

    uint alpha = qRound(opacity * 256);
    while (fpos < s.first().first) {
        colorTable[pos] = qtToGlColor(PREMUL(ARGB_COMBINE_ALPHA(colors[0], alpha)));
        pos++;
        fpos += incr;
    }

    for (int i = 0; i < s.size() - 1; ++i) {
        qreal delta = 1/(s[i+1].first - s[i].first);
        while (fpos < s[i+1].first && pos < size) {
            int dist = int(256 * ((fpos - s[i].first) * delta));
            int idist = 256 - dist;
            uint current_color = PREMUL(ARGB_COMBINE_ALPHA(colors[i], alpha));
            uint next_color = PREMUL(ARGB_COMBINE_ALPHA(colors[i+1], alpha));
            colorTable[pos] = qtToGlColor(INTERPOLATE_PIXEL_256(current_color, idist, next_color, dist));

            ++pos;
            fpos += incr;
        }
    }

    Q_ASSERT(s.size() > 0);

    uint last_color = qtToGlColor(PREMUL(ARGB_COMBINE_ALPHA(colors[s.size() - 1], alpha)));
    for (;pos < size; ++pos)
        colorTable[pos] = last_color;

    // Make sure the last color stop is represented at the end of the table
    colorTable[size-1] = last_color;
}
