/****************************************************************************
**
** Copyright (C) 2007-2008 Trolltech ASA. All rights reserved.
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License versions 2.0 or 3.0 as published by the Free Software
** Foundation and appearing in the files LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file.  Alternatively you may (at
** your option) use any later version of the GNU General Public
** License if such license has been publicly approved by Trolltech ASA
** (or its successors, if any) and the KDE Free Qt Foundation. In
** addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.2, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** Please review the following information to ensure GNU General
** Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/. If
** you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech, as the sole
** copyright holder for Qt Designer, grants users of the Qt/Eclipse
** Integration plug-in the right for the Qt/Eclipse Integration to
** link to functionality provided by Qt Designer and its related
** libraries.
**
** This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
** INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE. Trolltech reserves all rights not expressly
** granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <QMutex>

#include "private/qobject_p.h"
#include "qabstractmessagehandler.h"

QT_BEGIN_NAMESPACE

class QAbstractMessageHandlerPrivate : public QObjectPrivate
{
public:
    QMutex mutex;
};

/*!
  \class QAbstractMessageHandler
  \threadsafe
  \since 4.4
  \ingroup xml-tools

  \brief The QAbstractMessageHandler class provides a callback interface for receiving messages.

  QAbstractMessageHandler is an abstract base class that provides an interface for
  receiving messages. Typically one subclass QAbstractMessageHandler
  and implement the handleMessage() function and subsequently pass a pointer to the code that
  generates the messages.

  The user of QAbstractMessageHandler calls message(). message() forwards the arguments
  on to handleMessage() which the sub-class has implemented, and while doing so seralizes
  the calls. This means QAbstractMessageHandler is thread safe.

  QAbstractMessageHandler isn't tied to a particular use case. This means that the
  interpretation of the arguments passed to message(), depends on the context
  in which it is used, and must therefore be specified in that context.
 */

/*!
  Constructs a QAbstractMessageHandler.

  The argument \a parent is sent to the QObject's constructor.
 */
QAbstractMessageHandler::QAbstractMessageHandler(QObject *parent) : QObject(*new QAbstractMessageHandlerPrivate(), parent)
{
}

/*!
  Destructs this QAbstractMessageHandler.
 */
QAbstractMessageHandler::~QAbstractMessageHandler()
{
}

/*!
  Sends a message to this message handler.

  How \a type and \a description, its format and content, are interpreted
  must be defined by the context in which this message handler is used. \a identifier
  is a URI that identifies the message, and therefore is the key to how the other
  arguments should be interpreted.

  \a sourceLocation identifies where the message was generated.

  Since \a identifier is globally unique, messages from arbitrary sources can safely
  be identified.

  It is the caller's responsibility to guarantee that \a identifier is a valid QUrl,
  or a default constructed QUrl.

  This function unconditionally calls handleMessage().
 */
void QAbstractMessageHandler::message(QtMsgType type,
                                      const QString &description,
                                      const QUrl &identifier,
                                      const QSourceLocation &sourceLocation)
{
    Q_D(QAbstractMessageHandler);
    QMutexLocker(&d->mutex);
    handleMessage(type, description, identifier, sourceLocation);
}

/*!
  \fn void QAbstractMessageHandler::handleMessage(QtMsgType type,
                                                  const QString &description,
                                                  const QUrl &identifier = QUrl(),
                                                  const QSourceLocation &sourceLocation = QSourceLocation()) = 0

  This function is re-implemented by the sub-class. It's called by message() and receives its arguments,
  \a type, \a description, \a identifier and \a sourceLocation from message() unmodified.
 */

QT_END_NAMESPACE

