/****************************************************************************
**
** Copyright (C) 2007-2008 Trolltech ASA. All rights reserved.
**
** This file is part of the demonstration applications of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License versions 2.0 or 3.0 as published by the Free Software
** Foundation and appearing in the files LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file.  Alternatively you may (at
** your option) use any later version of the GNU General Public
** License if such license has been publicly approved by Trolltech ASA
** (or its successors, if any) and the KDE Free Qt Foundation. In
** addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.2, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** Please review the following information to ensure GNU General
** Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/. If
** you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech, as the sole
** copyright holder for Qt Designer, grants users of the Qt/Eclipse
** Integration plug-in the right for the Qt/Eclipse Integration to
** link to functionality provided by Qt Designer and its related
** libraries.
**
** This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
** INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE. Trolltech reserves all rights not expressly
** granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "browsertabproxy.h"

#include "browsermainwindow.h"
#include "webview.h"

#include <QtCore/QSignalMapper>

#include <QtGui/QAction>
#include <QtGui/QStatusBar>

#define INITIALIZE_WEB_ACTION(slotName, actionName, webAction) \
    m_webActionConnections.append(WebActionConnection(slotName, &m_mainWindow->actionName, webAction));

BrowserTabProxy::BrowserTabProxy(BrowserMainWindow* mainWindow)
    : QObject(mainWindow)
    , m_mainWindow(mainWindow)
    , m_mapper(0)
{
}

void BrowserTabProxy::init()
{
    if (m_mapper)
        return;

    m_mapper = new QSignalMapper(this);
    QObject::connect(m_mapper, SIGNAL(mapped(int)), this, SLOT(slotUpdateAction(int)));

    INITIALIZE_WEB_ACTION(SLOT(slotUndo()), m_undo, QWebPage::Undo)
    INITIALIZE_WEB_ACTION(SLOT(slotRedo()), m_redo, QWebPage::Redo)
    INITIALIZE_WEB_ACTION(SLOT(slotCut()), m_cut, QWebPage::Cut)
    INITIALIZE_WEB_ACTION(SLOT(slotCopy()), m_copy, QWebPage::Copy)
    INITIALIZE_WEB_ACTION(SLOT(slotPaste()), m_paste, QWebPage::Paste)
    INITIALIZE_WEB_ACTION(SLOT(slotStop()), m_stop, QWebPage::Stop)
    INITIALIZE_WEB_ACTION(SLOT(slotReload()), m_reload, QWebPage::Reload)
    INITIALIZE_WEB_ACTION(SLOT(slotRedo()), m_reload, QWebPage::Redo)
    INITIALIZE_WEB_ACTION(SLOT(slotBack()), m_historyBack, QWebPage::Back)
    INITIALIZE_WEB_ACTION(SLOT(slotForward()), m_historyForward, QWebPage::Forward);
}

/*
 * Three steps to victory
 *  1.) forget about the past
 *  2.) prepare for the future
 *  3.) update the GUI
 */
void BrowserTabProxy::setTab(WebView *newTab)
{
    init();
    disconnect(m_currentTab);
    connect(newTab);
    update(newTab);
    m_currentTab = newTab;
    if (m_currentTab)
        m_currentTab->setFocus();
}

void BrowserTabProxy::updateAction(int index, WebView* tab)
{
    init();
    QAction* source = tab->page()->action(m_webActionConnections[index].webAction);

    (*m_webActionConnections[index].action)->setChecked(source->isChecked());
    (*m_webActionConnections[index].action)->setEnabled(source->isEnabled());
    (*m_webActionConnections[index].action)->setVisible(source->isVisible());
}

void BrowserTabProxy::slotUpdateAction(int index)
{
    updateAction(index, m_currentTab);
}

void BrowserTabProxy::disconnect(WebView* oldTab)
{
    Q_ASSERT(oldTab == m_currentTab);
    if (!oldTab)
        return;

    m_currentTab->disconnect(m_mainWindow, SLOT(slotLoadStarted()));
    m_currentTab->disconnect(m_mainWindow, SLOT(slotLoadFinished()));
    m_currentTab->disconnect(m_mainWindow, SLOT(slotUpdateWindowTitle(const QString&)));
    m_currentTab->disconnect(m_mainWindow->statusBar(), SLOT(showMessage(const QString&)));
    m_currentTab->disconnect(m_mainWindow, SLOT(slotUrlChanged(const QUrl &)));
    m_currentTab->page()->disconnect(this, SIGNAL(linkHovered(const QString&)));

    for (int i = 0; i < m_webActionConnections.size(); ++i) {
        QAction* action = oldTab->page()->action(m_webActionConnections[i].webAction);
        action->disconnect(m_mainWindow);
        m_mapper->removeMappings(action);
    }
}

void BrowserTabProxy::connect(WebView* newTab)
{
    if (!newTab) {
        m_mainWindow->slotUpdateWindowTitle(QString());
        return;
    }

    for (int i = 0; i < m_webActionConnections.size(); ++i) {
        QAction* action = newTab->page()->action(m_webActionConnections[i].webAction);
        QObject::connect(action, SIGNAL(triggered(bool)), m_mainWindow, m_webActionConnections[i].slot);
        QObject::connect(action, SIGNAL(changed()), m_mapper, SLOT(map()));
        m_mapper->setMapping(action, i);
        updateAction(i, newTab);
    }

    QObject::connect(newTab, SIGNAL(statusBarMessage(const QString&)),
                     m_mainWindow->statusBar(), SLOT(showMessage(const QString&)));
    QObject::connect(newTab, SIGNAL(titleChanged(const QString&)),
                     m_mainWindow, SLOT(slotUpdateWindowTitle(const QString&)));
    QObject::connect(newTab, SIGNAL(loadStarted()),
                     m_mainWindow, SLOT(slotLoadStarted()));
    QObject::connect(newTab, SIGNAL(loadFinished()),
                     m_mainWindow, SLOT(slotLoadFinished()));
    QObject::connect(newTab, SIGNAL(urlChanged(const QUrl &)),
                     m_mainWindow, SLOT(slotUrlChanged(const QUrl &)));
    QObject::connect(newTab->page(), SIGNAL(linkHovered(const QString&, const QString&, const QString&)),
                     this, SIGNAL(linkHovered(const QString&)));
}

void BrowserTabProxy::update(WebView* newTab)
{
    if (!newTab)
        return;

    m_mainWindow->replaceCurrentUrl(newTab->url());
    m_mainWindow->statusBar()->showMessage(newTab->lastStausBarText());
    m_mainWindow->slotUpdateWindowTitle(newTab->title());
    newTab->loadIsFinished() ?
        m_mainWindow->slotLoadFinished() :
        m_mainWindow->slotLoadStarted();
}

BrowserTabProxy::WebActionConnection::WebActionConnection(const char* _slot, QAction** _action, QWebPage::WebAction _webAction)
    : slot(_slot)
    , action(_action)
    , webAction(_webAction)
{}

