
/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtTest module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may only be used for
** evaluation and testing purposes.  It may not be used for commercial
** development.  You may use this file in accordance with the terms and
** conditions contained in the either Technology Preview License
** Agreement or the Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "QtTest/private/qbenchmark_p.h"

#ifdef QTESTLIB_USE_VALGRIND

#include "QtTest/private/qbenchmarkvalgrind_p.h"
#include <QtCore/qstringlist.h>
#include <QtCore/qprocess.h>
#include <QtCore/qdir.h>
#include <QtCore/qset.h>
#include "3rdparty/callgrind_p.h"

// Returns true iff a sufficiently recent valgrind is available.
bool QBenchmarkValgrindUtils::haveValgrind()
{
#ifdef NVALGRIND
    return false;
#else
    QProcess process;
    QStringList args;
    args << QLatin1String("--version");
    process.start(QLatin1String("valgrind"), args);
    return process.waitForFinished(-1);
#endif
}

// Reruns this program through callgrind, loads results, and cleans up temporary files.
// Returns true upon success, otherwise false.
bool QBenchmarkValgrindUtils::rerunThroughCallgrind(const QStringList &origAppArgs)
{
    if (!QBenchmarkValgrindUtils::runCallgrindSubProcess(origAppArgs)) {
        qWarning("failed to run callgrind subprocess");
        return false;
    }
    if (!QBenchmarkCallgrindResults::instance().load()) {
        qWarning("failed to load callgrind results");
        return false;
    }
    QBenchmarkValgrindUtils::removeCallgrindFiles();
    return true;
}

// Reruns this program through callgrind, storing callgrind result files in the
// current directory.
// Returns true upon success, otherwise false.
bool QBenchmarkValgrindUtils::runCallgrindSubProcess(const QStringList &origAppArgs)
{
    const QString execFile(origAppArgs.at(0));
    QStringList args;
    args << QLatin1String("--tool=callgrind") << QLatin1String("--instr-atstart=no") << execFile << QLatin1String("-callgrindchild");

#if (defined Q_WS_QWS) && (defined QT_GUI_LIB)
    // While running the child process, we aren't processing events, and hence aren't
    // acting as the QWS server.  Therefore, if we are currently the QWS server, it's
    // necessary to tell the child to act as its own server instead of connecting to us.
    if (qApp->type() == QApplication::GuiServer) {
        args << QLatin1String("-qws");
    }
#endif

    // pass on original arguments that make sense (e.g. avoid wasting time producing output
    // that will be ignored anyway) ...
    for (int i = 1; i < origAppArgs.size(); ++i) {
        const QString arg(origAppArgs.at(i));
        if (arg == QLatin1String("-callgrind")) {
            continue;
        } else if (arg == QLatin1String("-o")) {
            ++i;
            continue;
        }
        args << arg; // ok to pass on
    }

    args << QLatin1String("-silent"); // minimize verbosity

    QProcess process;
    process.start(QLatin1String("valgrind"), args);
    if (!process.waitForFinished(-1))
        return false;

    return true;
}

// Removes all callgrind results files from current directory.
void QBenchmarkValgrindUtils::removeCallgrindFiles()
{
    QDir dir;
    QRegExp rx(QLatin1String("^callgrind\\.out\\.\\d+(\\.\\d+)?$"));
    foreach (QString fileName, dir.entryList())
        if (fileName.contains(rx))
            dir.remove(fileName);
}

QBenchmarkCallgrindResults & QBenchmarkCallgrindResults::instance()
{
    static QBenchmarkCallgrindResults instance_;
    return instance_;
}

// Loads results from all callgrind files in the current directory.
// Returns true upon success, otherwise false.
bool QBenchmarkCallgrindResults::load()
{
    QDir dir;

    QRegExp rxFile(QLatin1String("^callgrind\\.out\\.\\d+\\.\\d+$"));
    QRegExp rxTag(QLatin1String("^desc: Trigger: Client Request: (.+)$"));
    QRegExp rxValue(QLatin1String("^summary: ([0-9]+)"));
    QRegExp rxCheckpointGroup(QLatin1String("^(.+),\\d+$"));

    QMap<QString, int> tag2val;
    QSet<QString> checkpointGroups;

    foreach (QFileInfo fileInfo, dir.entryInfoList()) {
        if (!(fileInfo.isFile() && fileInfo.fileName().contains(rxFile)))
            continue;

        QFile file(fileInfo.filePath());
        if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
            qWarning(
                "collectResultsFromDisk(): failed to open file: %s",
                fileInfo.fileName().toAscii().data());
            return false;
        }

        QString tag;
        int val = -1;
        bool tagSeen = false;
        bool valSeen = false;
        while (!file.atEnd()) {
            const QString line(QLatin1String(file.readLine()));
            if (rxTag.indexIn(line) != -1) {
                Q_ASSERT(rxTag.numCaptures() == 1);
                tag = rxTag.cap(1).trimmed();
                tagSeen = true;
                if (valSeen)
                    break;
            } else if (rxValue.indexIn(line) != -1) {
                Q_ASSERT(rxValue.numCaptures() == 1);
                bool ok;
                val = rxValue.cap(1).toInt(&ok);
                Q_ASSERT(ok);
                valSeen = true;
                if (tagSeen)
                    break;
            }
        }

        if (valSeen) { // accumulate
            tag2val.insert(tag, val + (tag2val.contains(tag) ? tag2val.value(tag) : 0));
            Q_ASSERT(tag2val.contains(tag));
            rxCheckpointGroup.indexIn(tag);
            Q_ASSERT(rxCheckpointGroup.numCaptures() == 1);
            QString cpGroup = rxCheckpointGroup.cap(1);
            checkpointGroups.insert(cpGroup);
        }
    }

    // Compute accumulated values for multiple checkpoints (necessary because valgrind
    // zeroes callgrind event counters every time a new results file is dumped (which happens
    // every time checkpoint() is called)):
    foreach (QString cpGroup, checkpointGroups.values()) {
        int totalValue = 0;
        for (int cpIndex = 0;; ++cpIndex) {
            QString tag = QString(QLatin1String("%1,%2")).arg(cpGroup).arg(cpIndex);
            if (!tag2val.contains(tag))
                break;
            const int tagValue = tag2val.value(tag);
            totalValue += tagValue;
            tag2val.insert(tag, totalValue);
        }
    }

    foreach (QString tag, tag2val.keys())
        results.insert(tag, tag2val.value(tag));

    return true;
}

int QBenchmarkCallgrindResults::result(const QString &key) const
{
    return results.contains(key) ? results.value(key) : -1;
}

void QBenchmarkCallgrindMeasurer::start()
{
    if (QBenchmarkGlobalData::current->mode() == QBenchmarkGlobalData::CallgrindChildProcess)
        CALLGRIND_START_INSTRUMENTATION;
}

qint64 QBenchmarkCallgrindMeasurer::checkpoint()
{
    QBenchmarkGlobalData::current->context.checkpointIndex++;

    if (QBenchmarkGlobalData::current->mode() == QBenchmarkGlobalData::CallgrindChildProcess) {
        CALLGRIND_DUMP_STATS_AT(
            QBenchmarkGlobalData::current->context.toString().toAscii().data());
        return -1;
    }

    Q_ASSERT(
        QBenchmarkGlobalData::current->mode() == QBenchmarkGlobalData::CallgrindParentProcess);

    return QBenchmarkCallgrindResults::instance().result(
        QBenchmarkGlobalData::current->context.toString());
}

qint64 QBenchmarkCallgrindMeasurer::stop()
{	
    if (QBenchmarkGlobalData::current->mode() == QBenchmarkGlobalData::CallgrindChildProcess) {
        checkpoint();
        CALLGRIND_STOP_INSTRUMENTATION;
        return -1;
    }

    Q_ASSERT(
        QBenchmarkGlobalData::current->mode() == QBenchmarkGlobalData::CallgrindParentProcess);

    return checkpoint();
}

bool QBenchmarkCallgrindMeasurer::isMeasurementAccepted(qint64 measurement)
{
    Q_UNUSED(measurement);
    return true;
}

int QBenchmarkCallgrindMeasurer::adjustIterationCount(int)
{ 
    return 1; 
}

int QBenchmarkCallgrindMeasurer::adjustMedianCount(int)
{ 
    return 1; 
}

QString QBenchmarkCallgrindMeasurer::unitText()
{
    return QLatin1String("instr. loads");
}

QString QBenchmarkCallgrindMeasurer::metricText()
{
    return QLatin1String("callgrind");
}

#endif // QTESTLIB_USE_VALGRIND
