#
# Copyright (C) 2005, 2006, 2007, 2008 Nikolas Zimmermann <zimmermann@kde.org>
# Copyright (C) 2006 Anders Carlsson <andersca@mac.com>
# Copyright (C) 2006, 2007 Samuel Weinig <sam@webkit.org>
# Copyright (C) 2006 Alexey Proskuryakov <ap@webkit.org>
# Copyright (C) 2006, 2007, 2008 Apple Inc. All rights reserved.
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public License
# aint with this library; see the file COPYING.LIB.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA 02110-1301, USA.

package CodeGeneratorJS;

use File::stat;

my $module = "";
my $outputDir = "";

my @headerContentHeader = ();
my @headerContent = ();
my %headerIncludes = ();

my @implContentHeader = ();
my @implContent = ();
my %implIncludes = ();
my %implKJSIncludes = ();

# Default .h template
my $headerTemplate = << "EOF";
/*
    This file is part of the WebKit open source project.
    This file has been generated by generate-bindings.pl. DO NOT MODIFY!

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
EOF

# Default constructor
sub new
{
    my $object = shift;
    my $reference = { };

    $codeGenerator = shift;
    $outputDir = shift;

    bless($reference, $object);
    return $reference;
}

sub finish
{
    my $object = shift;

    # Commit changes!
    $object->WriteData();
}

sub leftShift($$) {
    my ($value, $distance) = @_;
    return (($value << $distance) & 0xFFFFFFFF);
}

# Params: 'domClass' struct
sub GenerateInterface
{
    my $object = shift;
    my $dataNode = shift;
    my $defines = shift;

    # Start actual generation
    $object->GenerateHeader($dataNode);
    $object->GenerateImplementation($dataNode);

    my $name = $dataNode->name;

    # Open files for writing
    my $headerFileName = "$outputDir/JS$name.h";
    my $implFileName = "$outputDir/JS$name.cpp";

    open($IMPL, ">$implFileName") || die "Couldn't open file $implFileName";
    open($HEADER, ">$headerFileName") || die "Couldn't open file $headerFileName";
}

# Params: 'idlDocument' struct
sub GenerateModule
{
    my $object = shift;
    my $dataNode = shift;

    $module = $dataNode->module;
}

sub GetParentClassName
{
    my $dataNode = shift;

    return $dataNode->extendedAttributes->{"LegacyParent"} if $dataNode->extendedAttributes->{"LegacyParent"};
    return "DOMObject" if @{$dataNode->parents} eq 0;
    return "JS" . $codeGenerator->StripModule($dataNode->parents(0));
}

sub GetVisibleClassName
{
    my $className = shift;

    return "DOMException" if $className eq "DOMCoreException";
    return $className;
}

sub AvoidInclusionOfType
{
    my $type = shift;

    # Special case: SVGRect.h / SVGPoint.h / SVGNumber.h / SVGMatrix.h do not exist.
    return 1 if $type eq "SVGRect" or $type eq "SVGPoint" or $type eq "SVGNumber" or $type eq "SVGMatrix";
    return 0;
}

sub UsesManualToJSImplementation
{
    my $type = shift;

    return 1 if $type eq "Node" or $type eq "Document" or $type eq "HTMLCollection" or $type eq "SVGPathSeg" or $type eq "StyleSheet" or $type eq "CSSRule" or $type eq "CSSValue" or $type eq "Event" or $type eq "ImageData" or $type eq "Element" or $type eq "Text" or $type eq "SVGElementInstance";
    return 0;
}

sub IndexGetterReturnsStrings
{
    my $type = shift;

    return 1 if $type eq "CSSStyleDeclaration" or $type eq "MediaList" or $type eq "CSSVariablesDeclaration";
    return 0;
}

sub CreateSVGContextInterfaceName
{
    my $type = shift;

    return $type if $codeGenerator->IsSVGAnimatedType($type);
    return "SVGPathSeg" if $type =~ /^SVGPathSeg/ and $type ne "SVGPathSegList";

    return "";
}

sub AddIncludesForType
{
    my $type = $codeGenerator->StripModule(shift);

    # When we're finished with the one-file-per-class
    # reorganization, we won't need these special cases.
    if ($codeGenerator->IsPrimitiveType($type) or AvoidInclusionOfType($type)
        or $type eq "DOMString" or $type eq "DOMObject" or $type eq "RGBColor") {
    } elsif ($type =~ /SVGPathSeg/) {
        $joinedName = $type;
        $joinedName =~ s/Abs|Rel//;
        $implIncludes{"${joinedName}.h"} = 1;
    } elsif ($type eq "XPathNSResolver") {
        $implIncludes{"JSXPathNSResolver.h"} = 1;
        $implIncludes{"JSCustomXPathNSResolver.h"} = 1;
    } else {
        # default, include the same named file
        $implIncludes{"${type}.h"} = 1;
    }

    # additional includes (things needed to compile the bindings but not the header)

    if ($type eq "CanvasRenderingContext2D") {
        $implIncludes{"CanvasGradient.h"} = 1;
        $implIncludes{"CanvasPattern.h"} = 1;
        $implIncludes{"CanvasStyle.h"} = 1;
    }

    if ($type eq "CanvasGradient" or $type eq "XPathNSResolver" or $type eq "MessagePort") {
        $implIncludes{"PlatformString.h"} = 1;
    }

    if ($type eq "Document") {
        $implIncludes{"NodeFilter.h"} = 1;
    }
}

sub AddIncludesForSVGAnimatedType
{
    my $type = shift;
    $type =~ s/SVGAnimated//;

    if ($type eq "Point" or $type eq "Rect") {
        $implIncludes{"Float$type.h"} = 1;
    } elsif ($type eq "String") {
        $implIncludes{"PlatformString.h"} = 1;
    }
}

sub AddClassForwardIfNeeded
{
    my $implClassName = shift;

    # SVGAnimatedLength/Number/etc.. are typedefs to SVGAnimtatedTemplate, so don't use class forwards for them!
    push(@headerContent, "class $implClassName;\n\n") unless $codeGenerator->IsSVGAnimatedType($implClassName);
}

sub IsSVGTypeNeedingContextParameter
{
    my $implClassName = shift;

    if ($implClassName =~ /SVG/ and not $implClassName =~ /Element/) {
        return 1 unless $implClassName =~ /SVGPaint/ or $implClassName =~ /SVGColor/ or $implClassName =~ /SVGDocument/;
    }

    return 0;
}

sub HashValueForClassAndName
{
    my $class = shift;
    my $name = shift;

    # SVG Filter enums live in WebCore namespace (platform/graphics/)
    if ($class =~ /^SVGFE*/ or $class =~ /^SVGComponentTransferFunctionElement$/) {
        return "WebCore::$name";
    }

    return "${class}::$name";
}

sub GenerateGetOwnPropertySlotBody
{
    my ($dataNode, $interfaceName, $className, $implClassName, $hasAttributes, $inlined) = @_;

    my $namespaceMaybe = ($inlined ? "JSC::" : "");

    my @getOwnPropertySlotImpl = ();

    if ($interfaceName eq "NamedNodeMap" or $interfaceName eq "HTMLCollection") {
        push(@getOwnPropertySlotImpl, "    ${namespaceMaybe}JSValue* proto = prototype();\n");
        push(@getOwnPropertySlotImpl, "    if (proto->isObject() && static_cast<${namespaceMaybe}JSObject*>(proto)->hasProperty(exec, propertyName))\n");
        push(@getOwnPropertySlotImpl, "        return false;\n\n");
    }

    my $hasNameGetterGeneration = sub {
        push(@getOwnPropertySlotImpl, "    if (canGetItemsForName(exec, static_cast<$implClassName*>(impl()), propertyName)) {\n");
        push(@getOwnPropertySlotImpl, "        slot.setCustom(this, nameGetter);\n");
        push(@getOwnPropertySlotImpl, "        return true;\n");
        push(@getOwnPropertySlotImpl, "    }\n");
        if ($inlined) {
            $headerIncludes{"AtomicString.h"} = 1;
        } else {
            $implIncludes{"AtomicString.h"} = 1;
        }
    };

    if ($dataNode->extendedAttributes->{"HasOverridingNameGetter"}) {
        &$hasNameGetterGeneration();
    }

    my $requiresManualLookup = $dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasNameGetter"};
    if ($requiresManualLookup) {
        push(@getOwnPropertySlotImpl, "    const ${namespaceMaybe}HashEntry* entry = ${className}Table.entry(exec, propertyName);\n");
        push(@getOwnPropertySlotImpl, "    if (entry) {\n");
        push(@getOwnPropertySlotImpl, "        slot.setCustom(this, entry->propertyGetter());\n");
        push(@getOwnPropertySlotImpl, "        return true;\n");
        push(@getOwnPropertySlotImpl, "    }\n");
    }

    if ($dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}) {
        push(@getOwnPropertySlotImpl, "    bool ok;\n");
        push(@getOwnPropertySlotImpl, "    unsigned index = propertyName.toUInt32(&ok, false);\n");
        push(@getOwnPropertySlotImpl, "    if (ok && index < static_cast<$implClassName*>(impl())->length()) {\n");
        if ($dataNode->extendedAttributes->{"HasCustomIndexGetter"}) {
            push(@getOwnPropertySlotImpl, "        slot.setValue(getByIndex(exec, index));\n");
        } else {
            push(@getOwnPropertySlotImpl, "        slot.setCustomIndex(this, index, indexGetter);\n");
        }
        push(@getOwnPropertySlotImpl, "        return true;\n");
        push(@getOwnPropertySlotImpl, "    }\n");
    }

    if ($dataNode->extendedAttributes->{"HasNameGetter"}) {
        &$hasNameGetterGeneration();
    }

    if ($dataNode->extendedAttributes->{"CustomGetOwnPropertySlot"}) {
        push(@getOwnPropertySlotImpl, "    if (customGetOwnPropertySlot(exec, propertyName, slot))\n");
        push(@getOwnPropertySlotImpl, "        return true;\n");
    }

    if ($hasAttributes) {
        if ($inlined) {
            push(@getOwnPropertySlotImpl, "    return ${namespaceMaybe}getStaticValueSlot<$className, Base>(exec, s_info.staticPropHashTable, this, propertyName, slot);\n");
        } else {
           push(@getOwnPropertySlotImpl, "    return ${namespaceMaybe}getStaticValueSlot<$className, Base>(exec, &${className}Table, this, propertyName, slot);\n");
        }
    } else {
        push(@getOwnPropertySlotImpl, "    return Base::getOwnPropertySlot(exec, propertyName, slot);\n");
    }

    return @getOwnPropertySlotImpl;
}

sub GenerateHeader
{
    my $object = shift;
    my $dataNode = shift;

    my $interfaceName = $dataNode->name;
    my $className = "JS$interfaceName";
    my $implClassName = $interfaceName;

    # We only support multiple parents with SVG (for now).
    if (@{$dataNode->parents} > 1) {
        die "A class can't have more than one parent" unless $interfaceName =~ /SVG/;
        $codeGenerator->AddMethodsConstantsAndAttributesFromParentClasses($dataNode);
    }

    my $hasLegacyParent = $dataNode->extendedAttributes->{"LegacyParent"};
    my $hasRealParent = @{$dataNode->parents} > 0;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($dataNode);
    my $conditional = $dataNode->extendedAttributes->{"Conditional"};

    # - Add default header template
    @headerContentHeader = split("\r", $headerTemplate);

    # - Add header protection
    push(@headerContentHeader, "\n#ifndef $className" . "_h");
    push(@headerContentHeader, "\n#define $className" . "_h\n\n");

    my $conditionalString;
    if ($conditional) {
        $conditionalString = "ENABLE(" . join(") && ENABLE(", split(/&/, $conditional)) . ")";
        push(@headerContentHeader, "\n#if ${conditionalString}\n\n");
    }

    if ($hasParent) {
        push(@headerContentHeader, "#include \"$parentClassName.h\"\n");
    } else {
        push(@headerContentHeader, "#include \"JSDOMBinding.h\"\n");
        push(@headerContentHeader, "#include <kjs/JSGlobalObject.h>\n");
        push(@headerContentHeader, "#include <kjs/ObjectPrototype.h>\n");
    }
    if ($interfaceName eq "Node") {
        push(@headerContentHeader, "#include \"EventTargetNode.h\"\n");
    } elsif ($interfaceName eq "SVGElementInstance") {
        push(@headerContentHeader, "#include \"EventTargetSVGElementInstance.h\"\n");
    }

    if ($dataNode->extendedAttributes->{"CustomCall"}) {
        push(@headerContentHeader, "#include <kjs/CallData.h>\n");
    }

    if ($dataNode->extendedAttributes->{"InlineGetOwnPropertySlot"}) {
        push(@headerContentHeader, "#include <kjs/lookup.h>\n");
        push(@headerContentHeader, "#include <wtf/AlwaysInline.h>\n");
    }

    if ($hasParent && $dataNode->extendedAttributes->{"GenerateNativeConverter"}) {
        push(@headerContentHeader, "#include \"${implClassName}.h\"");
    }

    # Get correct pass/store types respecting PODType flag
    my $podType = $dataNode->extendedAttributes->{"PODType"};
    my $implType = $podType ? "JSSVGPODTypeWrapper<$podType> " : $implClassName;
    push(@headerContentHeader, "#include \"$podType.h\"\n") if $podType and $podType ne "float";

    push(@headerContentHeader, "#include \"JSSVGPODTypeWrapper.h\"\n") if $podType;

    my $numConstants = @{$dataNode->constants};
    my $numAttributes = @{$dataNode->attributes};
    my $numFunctions = @{$dataNode->functions};

    push(@headerContent, "\nnamespace WebCore {\n\n");

    # Implementation class forward declaration
    AddClassForwardIfNeeded($implClassName) unless $podType;
    AddClassForwardIfNeeded("JSDOMWindowShell") if $interfaceName eq "DOMWindow";

    # Class declaration
    push(@headerContent, "class $className : public $parentClassName {\n");
    push(@headerContent, "    typedef $parentClassName Base;\n");
    push(@headerContent, "public:\n");

    # Constructor
    if ($interfaceName eq "DOMWindow") {
        push(@headerContent, "    $className(PassRefPtr<JSC::StructureID>, PassRefPtr<$implType>, JSDOMWindowShell*);\n");
    } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
        push(@headerContent, "    $className(PassRefPtr<JSC::StructureID>, PassRefPtr<$implType>, SVGElement* context);\n");
    } else {
        push(@headerContent, "    $className(PassRefPtr<JSC::StructureID>, PassRefPtr<$implType>);\n");
    }

    # Destructor
    push(@headerContent, "    virtual ~$className();\n") if (!$hasParent or $interfaceName eq "Document");

    # Prototype
    push(@headerContent, "    static JSC::JSObject* createPrototype(JSC::ExecState*);\n") if $interfaceName ne "DOMWindow";

    $implIncludes{"${className}Custom.h"} = 1 if $dataNode->extendedAttributes->{"CustomHeader"} || $dataNode->extendedAttributes->{"CustomPutFunction"};

    my $hasGetter = $numAttributes > 0 
                 || $dataNode->extendedAttributes->{"GenerateConstructor"} 
                 || $dataNode->extendedAttributes->{"HasIndexGetter"}
                 || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}
                 || $dataNode->extendedAttributes->{"CustomGetOwnPropertySlot"}
                 || $dataNode->extendedAttributes->{"HasNameGetter"}
                 || $dataNode->extendedAttributes->{"HasOverridingNameGetter"};

    # Getters
    if ($hasGetter) {
        push(@headerContent, "    virtual bool getOwnPropertySlot(JSC::ExecState*, const JSC::Identifier& propertyName, JSC::PropertySlot&);\n");
        push(@headerContent, "    virtual bool getOwnPropertySlot(JSC::ExecState*, unsigned propertyName, JSC::PropertySlot&);\n") if ($dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}) && !$dataNode->extendedAttributes->{"HasOverridingNameGetter"};
        push(@headerContent, "    bool customGetOwnPropertySlot(JSC::ExecState*, const JSC::Identifier&, JSC::PropertySlot&);\n") if $dataNode->extendedAttributes->{"CustomGetOwnPropertySlot"};
    }

    # Check if we have any writable properties
    my $hasReadWriteProperties = 0;
    foreach (@{$dataNode->attributes}) {
        if ($_->type !~ /^readonly\ attribute$/) {
            $hasReadWriteProperties = 1;
        }
    }

    my $hasSetter = $hasReadWriteProperties
                 || $dataNode->extendedAttributes->{"CustomPutFunction"}
                 || $dataNode->extendedAttributes->{"HasCustomIndexSetter"};

    # Getters
    if ($hasSetter) {
        push(@headerContent, "    virtual void put(JSC::ExecState*, const JSC::Identifier& propertyName, JSC::JSValue*, JSC::PutPropertySlot&);\n");
        push(@headerContent, "    virtual void put(JSC::ExecState*, unsigned propertyName, JSC::JSValue*);\n") if $dataNode->extendedAttributes->{"HasCustomIndexSetter"};
        push(@headerContent, "    bool customPut(JSC::ExecState*, const JSC::Identifier&, JSC::JSValue*, JSC::PutPropertySlot&);\n") if $dataNode->extendedAttributes->{"CustomPutFunction"};
    }

    # Class info
    push(@headerContent, "    virtual const JSC::ClassInfo* classInfo() const { return &s_info; }\n");
    push(@headerContent, "    static const JSC::ClassInfo s_info;\n\n");

    # Structure ID
    if ($interfaceName eq "DOMWindow") {
        push(@headerContent, "    static PassRefPtr<JSC::StructureID> createStructureID(JSC::JSValue* proto)\n" .
             "    {\n" .
             "        return JSC::StructureID::create(proto, JSC::TypeInfo(JSC::ObjectType, JSC::ImplementsHasInstance | JSC::NeedsThisConversion));\n" .
             "    }\n\n");
    }
    # Custom mark function
    push(@headerContent, "    virtual void mark();\n\n") if $dataNode->extendedAttributes->{"CustomMarkFunction"};

    # Custom pushEventHandlerScope function
    push(@headerContent, "    virtual void pushEventHandlerScope(JSC::ExecState*, JSC::ScopeChain&) const;\n\n") if $dataNode->extendedAttributes->{"CustomPushEventHandlerScope"};

    # Custom call functions
    push(@headerContent, "    virtual JSC::CallType getCallData(JSC::CallData&);\n\n") if $dataNode->extendedAttributes->{"CustomCall"};

    # Custom deleteProperty function
    push(@headerContent, "    virtual bool deleteProperty(JSC::ExecState*, const JSC::Identifier&);\n") if $dataNode->extendedAttributes->{"CustomDeleteProperty"};

    # Custom getPropertyNames function
    push(@headerContent, "    virtual void getPropertyNames(JSC::ExecState*, JSC::PropertyNameArray&);\n") if ($dataNode->extendedAttributes->{"CustomGetPropertyNames"} || $dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"});
    push(@headerContent, "    bool customGetPropertyNames(JSC::ExecState*, JSC::PropertyNameArray&);\n") if $dataNode->extendedAttributes->{"CustomGetPropertyNames"};

    # Custom getPropertyAttributes function
    push(@headerContent, "    virtual bool getPropertyAttributes(JSC::ExecState*, const JSC::Identifier&, unsigned& attributes) const;\n") if $dataNode->extendedAttributes->{"CustomGetPropertyAttributes"};

    # Custom defineGetter function
    push(@headerContent, "    virtual void defineGetter(JSC::ExecState*, const JSC::Identifier& propertyName, JSC::JSObject* getterFunction);\n") if $dataNode->extendedAttributes->{"CustomDefineGetter"};

    # Custom defineSetter function
    push(@headerContent, "    virtual void defineSetter(JSC::ExecState*, const JSC::Identifier& propertyName, JSC::JSObject* setterFunction);\n") if $dataNode->extendedAttributes->{"CustomDefineSetter"};

    # Custom lookupGetter function
    push(@headerContent, "    virtual JSC::JSValue* lookupGetter(JSC::ExecState*, const JSC::Identifier& propertyName);\n") if $dataNode->extendedAttributes->{"CustomLookupGetter"};

    # Custom lookupSetter function
    push(@headerContent, "    virtual JSC::JSValue* lookupSetter(JSC::ExecState*, const JSC::Identifier& propertyName);\n") if $dataNode->extendedAttributes->{"CustomLookupSetter"};

    # Constructor object getter
    push(@headerContent, "    static JSC::JSValue* getConstructor(JSC::ExecState*);\n") if $dataNode->extendedAttributes->{"GenerateConstructor"};

    my $numCustomFunctions = 0;
    my $numCustomAttributes = 0;

    # Attribute and function enums
    if ($numAttributes > 0) {
        foreach (@{$dataNode->attributes}) {
            my $attribute = $_;
            $numCustomAttributes++ if $attribute->signature->extendedAttributes->{"Custom"};
            $numCustomAttributes++ if $attribute->signature->extendedAttributes->{"CustomGetter"};
            $numCustomAttributes++ if $attribute->signature->extendedAttributes->{"CustomSetter"};
        }
    }

    if ($numCustomAttributes > 0) {
        push(@headerContent, "\n    // Custom attributes\n");

        foreach my $attribute (@{$dataNode->attributes}) {
            if ($attribute->signature->extendedAttributes->{"Custom"}) {
                push(@headerContent, "    JSC::JSValue* " . $codeGenerator->WK_lcfirst($attribute->signature->name) . "(JSC::ExecState*) const;\n");
                if ($attribute->type !~ /^readonly/) {
                    push(@headerContent, "    void set" . $codeGenerator->WK_ucfirst($attribute->signature->name) . "(JSC::ExecState*, JSC::JSValue*);\n");
                }
            } elsif ($attribute->signature->extendedAttributes->{"CustomGetter"}) {
                push(@headerContent, "    JSC::JSValue* " . $codeGenerator->WK_lcfirst($attribute->signature->name) . "(JSC::ExecState*) const;\n");
            } elsif ($attribute->signature->extendedAttributes->{"CustomSetter"}) {
                if ($attribute->type !~ /^readonly/) {
                    push(@headerContent, "    void set" . $codeGenerator->WK_ucfirst($attribute->signature->name) . "(JSC::ExecState*, JSC::JSValue*);\n");
                }
            }
        }
    }

    foreach my $function (@{$dataNode->functions}) {
        $numCustomFunctions++ if $function->signature->extendedAttributes->{"Custom"};
    }

    if ($numCustomFunctions > 0) {
        push(@headerContent, "\n    // Custom functions\n");
        foreach my $function (@{$dataNode->functions}) {
            if ($function->signature->extendedAttributes->{"Custom"}) {
                my $functionImplementationName = $function->signature->extendedAttributes->{"ImplementationFunction"} || $codeGenerator->WK_lcfirst($function->signature->name);
                push(@headerContent, "    JSC::JSValue* " . $functionImplementationName . "(JSC::ExecState*, const JSC::ArgList&);\n");
            }
        }
    }

    if (!$hasParent) {
        if ($podType) {
            push(@headerContent, "    JSSVGPODTypeWrapper<$podType>* impl() const { return m_impl.get(); }\n");
            push(@headerContent, "    SVGElement* context() const { return m_context.get(); }\n\n");
            push(@headerContent, "private:\n");
            push(@headerContent, "    RefPtr<SVGElement> m_context;\n");
            push(@headerContent, "    RefPtr<JSSVGPODTypeWrapper<$podType> > m_impl;\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@headerContent, "    $implClassName* impl() const { return m_impl.get(); }\n");
            push(@headerContent, "    SVGElement* context() const { return m_context.get(); }\n\n");
            push(@headerContent, "private:\n");
            push(@headerContent, "    RefPtr<SVGElement> m_context;\n");
            push(@headerContent, "    RefPtr<$implClassName > m_impl;\n");
        } else {
            push(@headerContent, "    $implClassName* impl() const { return m_impl.get(); }\n\n");
            push(@headerContent, "private:\n");
            push(@headerContent, "    RefPtr<$implClassName> m_impl;\n");
        }
    } elsif ($dataNode->extendedAttributes->{"GenerateNativeConverter"}) {
        push(@headerContent, "    $implClassName* impl() const\n");
        push(@headerContent, "    {\n");
        push(@headerContent, "        return static_cast<$implClassName*>(Base::impl());\n");
        push(@headerContent, "    }\n");
    }

    # Index getter
    if ($dataNode->extendedAttributes->{"HasIndexGetter"}) {
        push(@headerContent, "    static JSC::JSValue* indexGetter(JSC::ExecState*, const JSC::Identifier&, const JSC::PropertySlot&);\n");
    }
    if ($dataNode->extendedAttributes->{"HasCustomIndexGetter"}) {
        push(@headerContent, "    JSC::JSValue* getByIndex(JSC::ExecState*, unsigned index);\n");
    }
    
    # Index setter
    if ($dataNode->extendedAttributes->{"HasCustomIndexSetter"}) {
        push(@headerContent, "    void indexSetter(JSC::ExecState*, unsigned index, JSC::JSValue*);\n");
    }
    # Name getter
    if ($dataNode->extendedAttributes->{"HasNameGetter"} || $dataNode->extendedAttributes->{"HasOverridingNameGetter"}) {
        push(@headerContent, "private:\n");
        push(@headerContent, "    static bool canGetItemsForName(JSC::ExecState*, $implClassName*, const JSC::Identifier&);\n");
        push(@headerContent, "    static JSC::JSValue* nameGetter(JSC::ExecState*, const JSC::Identifier&, const JSC::PropertySlot&);\n");
    }

    push(@headerContent, "};\n\n");

    if ($dataNode->extendedAttributes->{"InlineGetOwnPropertySlot"}) {
        push(@headerContent, "ALWAYS_INLINE bool ${className}::getOwnPropertySlot(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)\n");
        push(@headerContent, "{\n");
        push(@headerContent, GenerateGetOwnPropertySlotBody($dataNode, $interfaceName, $className, $implClassName, $numAttributes > 0, 1));
        push(@headerContent, "}\n\n");
    }

    if (!$hasParent || $dataNode->extendedAttributes->{"GenerateToJS"}) {
        if ($podType) {
            push(@headerContent, "JSC::JSValue* toJS(JSC::ExecState*, JSSVGPODTypeWrapper<$podType>*, SVGElement* context);\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@headerContent, "JSC::JSValue* toJS(JSC::ExecState*, $implType*, SVGElement* context);\n");
        } else {
            push(@headerContent, "JSC::JSValue* toJS(JSC::ExecState*, $implType*);\n");
        }

        # Resolve ambiguity with EventTarget that otherwise exists.
        if ($interfaceName eq "Node") {
            push(@headerContent, "inline JSC::JSValue* toJS(JSC::ExecState* exec, EventTargetNode* node) { return toJS(exec, static_cast<Node*>(node)); }\n");
        } elsif ($interfaceName eq "SVGElementInstance") {
            push(@headerContent, "inline JSC::JSValue* toJS(JSC::ExecState* exec, EventTargetSVGElementInstance* instance) { return toJS(exec, static_cast<SVGElementInstance*>(instance)); }\n");
        }
    }
    if (!$hasParent || $dataNode->extendedAttributes->{"GenerateNativeConverter"}) {
        if ($podType) {
            push(@headerContent, "$podType to${interfaceName}(JSC::JSValue*);\n");
        } elsif ($interfaceName eq "NodeFilter") {
            push(@headerContent, "PassRefPtr<NodeFilter> toNodeFilter(JSC::JSValue*);\n");
        } else {
            push(@headerContent, "$implClassName* to${interfaceName}(JSC::JSValue*);\n");
        }
        if ($interfaceName eq "Node" or $interfaceName eq "Element" or $interfaceName eq "Text") {
            push(@headerContent, "JSC::JSValue* toJSNewlyCreated(JSC::ExecState*, $interfaceName*);\n");
        }
    }
    push(@headerContent, "\n");

    # Add prototype declaration.
    push(@headerContent, "class ${className}Prototype : public JSC::JSObject {\n");
    push(@headerContent, "public:\n");
    if ($interfaceName eq "DOMWindow") {
        push(@headerContent, "    void* operator new(size_t);\n");
    } else {
        push(@headerContent, "    static JSC::JSObject* self(JSC::ExecState*);\n");
    }
    push(@headerContent, "    virtual const JSC::ClassInfo* classInfo() const { return &s_info; }\n");
    push(@headerContent, "    static const JSC::ClassInfo s_info;\n");
    if ($numFunctions > 0 || $numConstants > 0) {
        push(@headerContent, "    virtual bool getOwnPropertySlot(JSC::ExecState*, const JSC::Identifier&, JSC::PropertySlot&);\n");
    }
    push(@headerContent, "    ${className}Prototype(PassRefPtr<JSC::StructureID> structure) : JSC::JSObject(structure) { }\n");

    push(@headerContent, "};\n\n");

    if ($numFunctions > 0) {
        push(@headerContent,"// Functions\n\n");
        foreach my $function (@{$dataNode->functions}) {
            my $functionName = $codeGenerator->WK_lcfirst($className) . "PrototypeFunction" . $codeGenerator->WK_ucfirst($function->signature->name);
            push(@headerContent, "JSC::JSValue* ${functionName}(JSC::ExecState*, JSC::JSObject*, JSC::JSValue*, const JSC::ArgList&);\n");
        }
    }

    if ($numAttributes > 0 || $dataNode->extendedAttributes->{"GenerateConstructor"}) {
        push(@headerContent,"// Attibutes\n\n");
        foreach my $attribute (@{$dataNode->attributes}) {
            my $getter = "js" . $interfaceName . $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
            push(@headerContent, "JSC::JSValue* ${getter}(JSC::ExecState*, const JSC::Identifier&, const JSC::PropertySlot&);\n");
            unless ($attribute->type =~ /readonly/) {
                my $setter = "setJS" . $interfaceName . $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
                push(@headerContent, "void ${setter}(JSC::ExecState*, JSC::JSObject*, JSC::JSValue*);\n");
            }
        }
        
        if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
            my $getter = "js" . $interfaceName . "Constructor";
            push(@headerContent, "JSC::JSValue* ${getter}(JSC::ExecState*, const JSC::Identifier&, const JSC::PropertySlot&);\n");
        }
    }

    if ($numConstants > 0) {
        push(@headerContent,"// Constants\n\n");
        foreach my $constant (@{$dataNode->constants}) {
            my $getter = "js" . $interfaceName . $codeGenerator->WK_ucfirst($constant->name);
            push(@headerContent, "JSC::JSValue* ${getter}(JSC::ExecState*, const JSC::Identifier&, const JSC::PropertySlot&);\n");
        }
    }

    push(@headerContent, "\n} // namespace WebCore\n\n");
    push(@headerContent, "#endif // ${conditionalString}\n\n") if $conditional;
    push(@headerContent, "#endif\n");
}

sub GenerateImplementation
{
    my ($object, $dataNode) = @_;

    my $interfaceName = $dataNode->name;
    my $className = "JS$interfaceName";
    my $implClassName = $interfaceName;

    my $hasLegacyParent = $dataNode->extendedAttributes->{"LegacyParent"};
    my $hasRealParent = @{$dataNode->parents} > 0;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($dataNode);
    my $conditional = $dataNode->extendedAttributes->{"Conditional"};
    my $visibleClassName = GetVisibleClassName($interfaceName);

    # - Add default header template
    @implContentHeader = split("\r", $headerTemplate);
    push(@implContentHeader, "\n#include \"config.h\"\n\n");
    my $conditionalString;
    if ($conditional) {
        $conditionalString = "ENABLE(" . join(") && ENABLE(", split(/&/, $conditional)) . ")";
        push(@implContentHeader, "\n#if ${conditionalString}\n\n");
    }

    if ($className =~ /^JSSVG/) {
        push(@implContentHeader, "#include \"SVGElement.h\"\n");

        if ($className =~ /^JSSVGAnimated/) {
            AddIncludesForSVGAnimatedType($interfaceName);
        }
    }

    push(@implContentHeader, "#include \"$className.h\"\n\n");
    push(@implContentHeader, "#include <wtf/GetPtr.h>\n\n");

    push(@implContentHeader, "#include <kjs/PropertyNameArray.h>\n") if $dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"};

    AddIncludesForType($interfaceName);

    @implContent = ();

    push(@implContent, "\nusing namespace JSC;\n\n");
    push(@implContent, "namespace WebCore {\n\n");

    push(@implContent, "ASSERT_CLASS_FITS_IN_CELL($className)\n\n");

    # - Add all attributes in a hashtable definition
    my $numAttributes = @{$dataNode->attributes};
    $numAttributes++ if $dataNode->extendedAttributes->{"GenerateConstructor"};

    if ($numAttributes > 0) {
        my $hashName = $className . "Table";

        my @hashKeys = ();
        my @hashSpecials = ();
        my @hashValue1 = ();
        my @hashValue2 = ();
        my %conditionals = ();

        my @entries = ();

        foreach my $attribute (@{$dataNode->attributes}) {
            my $name = $attribute->signature->name;
            push(@hashKeys, $name);

            my @specials = ();
            push(@specials, "DontDelete") unless $attribute->signature->extendedAttributes->{"Deletable"};
            push(@specials, "DontEnum") if $attribute->signature->extendedAttributes->{"DontEnum"};
            push(@specials, "ReadOnly") if $attribute->type =~ /readonly/;
            my $special = (@specials > 0) ? join("|", @specials) : "0";
            push(@hashSpecials, $special);

            my $getter = "js" . $interfaceName . $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
            push(@hashValue1, $getter);
    
            if ($attribute->type =~ /readonly/) {
                push(@hashValue2, "0");
            } else {
                my $setter = "setJS" . $interfaceName . $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
                push(@hashValue2, $setter);
            }

            my $conditional = $attribute->signature->extendedAttributes->{"Conditional"};
            if ($conditional) {
                $conditionals{$name} = $conditional;
            }
        }

        if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
            push(@hashKeys, "constructor");
            my $getter = "js" . $interfaceName . "Constructor";
            push(@hashValue1, $getter);
            push(@hashValue2, "0");
            push(@hashSpecials, "DontEnum|ReadOnly"); # FIXME: Setting the constructor should be possible.
        }

        $object->GenerateHashTable($hashName,
                                   \@hashKeys, \@hashSpecials,
                                   \@hashValue1, \@hashValue2,
                                   \%conditionals);
    }

    my $numConstants = @{$dataNode->constants};
    my $numFunctions = @{$dataNode->functions};

    # - Add all constants
    if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
        $hashName = $className . "ConstructorTable";

        @hashKeys = ();
        @hashValue1 = ();
        @hashValue2 = ();
        @hashSpecials = ();

        # FIXME: we should not need a function for every constant.
        foreach my $constant (@{$dataNode->constants}) {
            push(@hashKeys, $constant->name);
            my $getter = "js" . $interfaceName . $codeGenerator->WK_ucfirst($constant->name);
            push(@hashValue1, $getter);
            push(@hashValue2, "0");
            push(@hashSpecials, "DontDelete|ReadOnly");
        }

        $object->GenerateHashTable($hashName,
                                   \@hashKeys, \@hashSpecials,
                                   \@hashValue1, \@hashValue2);

        my $protoClassName;
        $protoClassName = "${className}Prototype";

        push(@implContent, constructorFor($className, $protoClassName, $interfaceName, $visibleClassName, $dataNode->extendedAttributes->{"CanBeConstructed"}));
    }

    # - Add functions and constants to a hashtable definition
    $hashName = $className . "PrototypeTable";

    @hashKeys = ();
    @hashValue1 = ();
    @hashValue2 = ();
    @hashSpecials = ();

    # FIXME: we should not need a function for every constant.
    foreach my $constant (@{$dataNode->constants}) {
        push(@hashKeys, $constant->name);
        my $getter = "js" . $interfaceName . $codeGenerator->WK_ucfirst($constant->name);
        push(@hashValue1, $getter);
        push(@hashValue2, "0");
        push(@hashSpecials, "DontDelete|ReadOnly");
    }

    foreach my $function (@{$dataNode->functions}) {
        my $name = $function->signature->name;
        push(@hashKeys, $name);

        my $value = $codeGenerator->WK_lcfirst($className) . "PrototypeFunction" . $codeGenerator->WK_ucfirst($name);
        push(@hashValue1, $value);

        my $numParameters = @{$function->parameters};
        push(@hashValue2, $numParameters);

        my @specials = ();
        push(@specials, "DontDelete") unless $function->signature->extendedAttributes->{"Deletable"};
        push(@specials, "DontEnum") if $function->signature->extendedAttributes->{"DontEnum"};
        push(@specials, "Function");        
        my $special = (@specials > 0) ? join("|", @specials) : "0";
        push(@hashSpecials, $special);
    }

    $object->GenerateHashTable($hashName,
                               \@hashKeys, \@hashSpecials,
                               \@hashValue1, \@hashValue2);

    push(@implContent, "const ClassInfo ${className}Prototype::s_info = { \"${visibleClassName}Prototype\", 0, &${className}PrototypeTable, 0 };\n\n");
    if ($interfaceName ne "DOMWindow") {
        push(@implContent, "JSObject* ${className}Prototype::self(ExecState* exec)\n");
        push(@implContent, "{\n");
        push(@implContent, "    return getDOMPrototype<${className}>(exec);\n");
        push(@implContent, "}\n\n");
    }
    if ($interfaceName eq "DOMWindow") {
        push(@implContent, "void* ${className}Prototype::operator new(size_t size)\n");
        push(@implContent, "{\n");
        push(@implContent, "    return JSDOMWindow::commonJSGlobalData()->heap.allocate(size);\n");
        push(@implContent, "}\n\n");
    }
    if ($numConstants > 0 || $numFunctions > 0) {
        push(@implContent, "bool ${className}Prototype::getOwnPropertySlot(ExecState* exec, const Identifier& propertyName, PropertySlot& slot)\n");
        push(@implContent, "{\n");
        if ($numConstants eq 0) {
            push(@implContent, "    return getStaticFunctionSlot<JSObject>(exec, &${className}PrototypeTable, this, propertyName, slot);\n");
        } elsif ($numFunctions eq 0) {
            push(@implContent, "    return getStaticValueSlot<${className}Prototype, JSObject>(exec, &${className}PrototypeTable, this, propertyName, slot);\n");
        } else {
            push(@implContent, "    return getStaticPropertySlot<${className}Prototype, JSObject>(exec, &${className}PrototypeTable, this, propertyName, slot);\n");
        }
        push(@implContent, "}\n\n");
    }

    # - Initialize static ClassInfo object
    push(@implContent, "const ClassInfo $className" . "::s_info = { \"${visibleClassName}\", ");
    if ($hasParent) {
        push(@implContent, "&" . $parentClassName . "::s_info, ");
    } else {
        push(@implContent, "0, ");
    }

    if ($numAttributes > 0) {
        push(@implContent, "&${className}Table ");
    } else {
        push(@implContent, "0 ");
    }
    push(@implContent, ", 0 ");
    push(@implContent, "};\n\n");

    # Get correct pass/store types respecting PODType flag
    my $podType = $dataNode->extendedAttributes->{"PODType"};
    my $implType = $podType ? "JSSVGPODTypeWrapper<$podType> " : $implClassName;

    my $needsSVGContext = IsSVGTypeNeedingContextParameter($implClassName);
    my $parentNeedsSVGContext = ($needsSVGContext and $parentClassName =~ /SVG/);

    # Constructor
    if ($interfaceName eq "DOMWindow") {
        AddIncludesForType("JSDOMWindowShell");
        push(@implContent, "${className}::$className(PassRefPtr<StructureID> structure, PassRefPtr<$implType> impl, JSDOMWindowShell* shell)\n");
        push(@implContent, "    : $parentClassName(structure, impl, shell)\n");
    } else {
        push(@implContent, "${className}::$className(PassRefPtr<StructureID> structure, PassRefPtr<$implType> impl" . ($needsSVGContext ? ", SVGElement* context" : "") . ")\n");
        if ($hasParent) {
            push(@implContent, "    : $parentClassName(structure, impl" . ($parentNeedsSVGContext ? ", context" : "") . ")\n");
        } else {
            push(@implContent, "    : $parentClassName(structure)\n");
            push(@implContent, "    , m_context(context)\n") if $needsSVGContext;
            push(@implContent, "    , m_impl(impl)\n");  
        }
    }
    push(@implContent, "{\n");
    push(@implContent, "}\n\n");

    # Destructor
    if (!$hasParent) {
        push(@implContent, "${className}::~$className()\n");
        push(@implContent, "{\n");

        if ($interfaceName eq "Node") {
            push(@implContent, "    forgetDOMNode(m_impl->document(), m_impl.get());\n");
        } else {
            if ($podType) {
                my $animatedType = $implClassName;
                $animatedType =~ s/SVG/SVGAnimated/;

                # Special case for JSSVGNumber
                if ($codeGenerator->IsSVGAnimatedType($animatedType) and $podType ne "float") {
                    push(@implContent, "    JSSVGDynamicPODTypeWrapperCache<$podType, $animatedType>::forgetWrapper(m_impl.get());\n");
                }
            }
            push(@implContent, "    forgetDOMObject(*Heap::heap(this)->globalData(), m_impl.get());\n");
        }

        push(@implContent, "\n}\n\n");
    }

    # Document needs a special destructor because it's a special case for caching. It needs
    # its own special handling rather than relying on the caching that Node normally does.
    if ($interfaceName eq "Document") {
        push(@implContent, "${className}::~$className()\n");
        push(@implContent, "{\n    forgetDOMObject(*Heap::heap(this)->globalData(), static_cast<${implClassName}*>(impl()));\n}\n\n");
    }

    if ($interfaceName ne "DOMWindow") {
        push(@implContent, "JSObject* ${className}::createPrototype(ExecState* exec)\n");
        push(@implContent, "{\n");
        if ($hasParent && $parentClassName ne "JSC::DOMNodeFilter") {
            push(@implContent, "    return new (exec) ${className}Prototype(${className}Prototype::createStructureID(${parentClassName}Prototype::self(exec)));\n");
        } else {
            push(@implContent, "    return new (exec) ${className}Prototype(${className}Prototype::createStructureID(exec->lexicalGlobalObject()->objectPrototype()));\n");
        }
        push(@implContent, "}\n\n");
    }

    my $hasGetter = $numAttributes > 0 
                 || $dataNode->extendedAttributes->{"GenerateConstructor"} 
                 || $dataNode->extendedAttributes->{"HasIndexGetter"}
                 || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}
                 || $dataNode->extendedAttributes->{"CustomGetOwnPropertySlot"}
                 || $dataNode->extendedAttributes->{"HasNameGetter"}
                 || $dataNode->extendedAttributes->{"HasOverridingNameGetter"};

    # Attributes
    if ($hasGetter) {
        if (!$dataNode->extendedAttributes->{"InlineGetOwnPropertySlot"}) {
            push(@implContent, "bool ${className}::getOwnPropertySlot(ExecState* exec, const Identifier& propertyName, PropertySlot& slot)\n");
            push(@implContent, "{\n");
            push(@implContent, GenerateGetOwnPropertySlotBody($dataNode, $interfaceName, $className, $implClassName, $numAttributes > 0, 0));
            push(@implContent, "}\n\n");
        }

        if (($dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}) 
                && !$dataNode->extendedAttributes->{"HasOverridingNameGetter"}) {
            push(@implContent, "bool ${className}::getOwnPropertySlot(ExecState* exec, unsigned propertyName, PropertySlot& slot)\n");
            push(@implContent, "{\n");
            push(@implContent, "    if (propertyName < static_cast<$implClassName*>(impl())->length()) {\n");
            if ($dataNode->extendedAttributes->{"HasCustomIndexGetter"}) {
                push(@implContent, "        slot.setValue(getByIndex(exec, propertyName));\n");
            } else {
                push(@implContent, "        slot.setCustomIndex(this, propertyName, indexGetter);\n");
            }
            push(@implContent, "        return true;\n");
            push(@implContent, "    }\n");
            push(@implContent, "    return getOwnPropertySlot(exec, Identifier::from(exec, propertyName), slot);\n");
            push(@implContent, "}\n\n");
        }
        
        if ($numAttributes > 0) {
            foreach my $attribute (@{$dataNode->attributes}) {
                my $name = $attribute->signature->name;
                my $type = $codeGenerator->StripModule($attribute->signature->type);
                my $getFunctionName = "js" . $interfaceName .  $codeGenerator->WK_ucfirst($attribute->signature->name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
                my $implGetterFunctionName = $codeGenerator->WK_lcfirst($name);

                my $conditional = $attribute->signature->extendedAttributes->{"Conditional"};
                if ($conditional) {
                    $conditionalString = "ENABLE(" . join(") && ENABLE(", split(/&/, $conditional)) . ")";
                    push(@implContent, "#if ${conditionalString}\n");
                }

                push(@implContent, "JSValue* ${getFunctionName}(ExecState* exec, const Identifier&, const PropertySlot& slot)\n");
                push(@implContent, "{\n");

                my $implClassNameForValueConversion = "";
                if (!$podType and ($codeGenerator->IsSVGAnimatedType($implClassName) or $attribute->type !~ /^readonly/)) {
                    $implClassNameForValueConversion = $implClassName;
                }

                if ($dataNode->extendedAttributes->{"CheckDomainSecurity"} && 
                        !$attribute->signature->extendedAttributes->{"DoNotCheckDomainSecurity"} &&
                        !$attribute->signature->extendedAttributes->{"DoNotCheckDomainSecurityOnGet"}) {
                    push(@implContent, "    if (!static_cast<$className*>(slot.slotBase())->allowsAccessFrom(exec))\n");
                    push(@implContent, "        return jsUndefined();\n");
                }

                if ($attribute->signature->extendedAttributes->{"Custom"} || $attribute->signature->extendedAttributes->{"CustomGetter"}) {
                    push(@implContent, "    return static_cast<$className*>(slot.slotBase())->$implGetterFunctionName(exec);\n");
                } elsif ($attribute->signature->extendedAttributes->{"CheckNodeSecurity"}) {
                    $implIncludes{"JSDOMBinding.h"} = 1;
                    push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(slot.slotBase())->impl());\n");
                    push(@implContent, "    return checkNodeSecurity(exec, imp->$implGetterFunctionName()) ? " . NativeToJSValue($attribute->signature, 0, $implClassName, $implClassNameForValueConversion, "imp->$implGetterFunctionName()", "static_cast<$className*>(slot.slotBase())") . " : jsUndefined();\n");
                } elsif ($attribute->signature->extendedAttributes->{"CheckFrameSecurity"}) {
                    $implIncludes{"Document.h"} = 1;
                    $implIncludes{"JSDOMBinding.h"} = 1;
                    push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(slot.slotBase())->impl());\n");
                    push(@implContent, "    return checkNodeSecurity(exec, imp->contentDocument()) ? " . NativeToJSValue($attribute->signature,  0, $implClassName, $implClassNameForValueConversion, "imp->$implGetterFunctionName()", "static_cast<$className*>(slot.slotBase())") . " : jsUndefined();\n");
                } elsif ($type eq "EventListener") {
                    $implIncludes{"JSEventListener.h"} = 1;
                    $implIncludes{"EventListener.h"} = 1;
                    my $listenerType;
                    if ($attribute->signature->extendedAttributes->{"ProtectedListener"}) {
                        $listenerType = "JSEventListener";
                    } else {
                        $listenerType = "JSUnprotectedEventListener";
                    }
                    push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(slot.slotBase())->impl());\n");
                    push(@implContent, "    if (${listenerType}* listener = static_cast<${listenerType}*>(imp->$implGetterFunctionName())) {\n");
                    push(@implContent, "        if (JSObject* listenerObj = listener->listenerObj())\n");
                    push(@implContent, "            return listenerObj;\n");
                    push(@implContent, "    }\n");
                    push(@implContent, "    return jsNull();\n");
                } elsif ($attribute->signature->type =~ /Constructor$/) {
                    my $constructorType = $codeGenerator->StripModule($attribute->signature->type);
                    $constructorType =~ s/Constructor$//;
                    push(@implContent, "    return JS" . $constructorType . "::getConstructor(exec);\n");
                } elsif (!@{$attribute->getterExceptions}) {
                    if ($podType) {
                        push(@implContent, "    $podType imp(*static_cast<$className*>(slot.slotBase())->impl());\n");
                        if ($podType eq "float") { # Special case for JSSVGNumber
                            push(@implContent, "    return " . NativeToJSValue($attribute->signature, 0, $implClassName, "", "imp", "static_cast<$className*>(slot.slotBase())") . ";\n");
                        } else {
                            push(@implContent, "    return " . NativeToJSValue($attribute->signature, 0, $implClassName, "", "imp.$implGetterFunctionName()", "static_cast<$className*>(slot.slotBase())") . ";\n");
                        }
                    } else {
                        push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(slot.slotBase())->impl());\n");
                        my $jsType = NativeToJSValue($attribute->signature, 0, $implClassName, $implClassNameForValueConversion, "imp->$implGetterFunctionName()", "static_cast<$className*>(slot.slotBase())");
                        if ($codeGenerator->IsSVGAnimatedType($type)) {
                            push(@implContent, "    RefPtr<$type> obj = $jsType;\n");
                            push(@implContent, "    return toJS(exec, obj.get(), imp);\n");
                        } else {
                            push(@implContent, "    return $jsType;\n");
                        }
                    }
                } else {
                    push(@implContent, "    ExceptionCode ec = 0;\n");

                    if ($podType) {
                        push(@implContent, "    $podType imp(*static_cast<$className*>(slot.slotBase())->impl());\n");
                        push(@implContent, "    JSC::JSValue* result = " . NativeToJSValue($attribute->signature, 0, $implClassName, "", "imp.$implGetterFunctionName(ec)", "static_cast<$className*>(slot.slotBase())") . ";\n");
                    } else {
                        push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(slot.slotBase())->impl());\n");
                        push(@implContent, "    JSC::JSValue* result = " . NativeToJSValue($attribute->signature, 0, $implClassName, $implClassNameForValueConversion, "imp->$implGetterFunctionName(ec)", "static_cast<$className*>(slot.slotBase())") . ";\n");
                    }

                    push(@implContent, "    setDOMException(exec, ec);\n");
                    push(@implContent, "    return result;\n");
                }

                push(@implContent, "}\n");

                if ($conditional) {
                    push(@implContent, "#endif\n");
                }

                push(@implContent, "\n");
            }

            if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
                my $constructorFunctionName = "js" . $interfaceName . "Constructor";

                push(@implContent, "JSValue* ${constructorFunctionName}(ExecState* exec, const Identifier&, const PropertySlot& slot)\n");
                push(@implContent, "{\n");
                push(@implContent, "    return static_cast<$className*>(slot.slotBase())->getConstructor(exec);\n");
                push(@implContent, "}\n");
            }
        }

        # Check if we have any writable attributes
        my $hasReadWriteProperties = 0;
        foreach my $attribute (@{$dataNode->attributes}) {
            $hasReadWriteProperties = 1 if $attribute->type !~ /^readonly/;
        }

        my $hasSetter = $hasReadWriteProperties
                     || $dataNode->extendedAttributes->{"CustomPutFunction"}
                     || $dataNode->extendedAttributes->{"HasCustomIndexSetter"};

        if ($hasSetter) {
            push(@implContent, "void ${className}::put(ExecState* exec, const Identifier& propertyName, JSValue* value, PutPropertySlot& slot)\n");
            push(@implContent, "{\n");
            if ($dataNode->extendedAttributes->{"HasCustomIndexSetter"}) {
                push(@implContent, "    bool ok;\n");
                push(@implContent, "    unsigned index = propertyName.toUInt32(&ok, false);\n");
                push(@implContent, "    if (ok) {\n");
                push(@implContent, "        indexSetter(exec, index, value);\n");
                push(@implContent, "        return;\n");
                push(@implContent, "    }\n");
            }
            if ($dataNode->extendedAttributes->{"CustomPutFunction"}) {
                push(@implContent, "    if (customPut(exec, propertyName, value, slot))\n");
                push(@implContent, "        return;\n");
            }

            if ($hasReadWriteProperties) {
                push(@implContent, "    lookupPut<$className, Base>(exec, propertyName, value, &${className}Table, this, slot);\n");
            } else {
                push(@implContent, "    Base::put(exec, propertyName, value, slot);\n");
            }
            push(@implContent, "}\n\n");

            if ($dataNode->extendedAttributes->{"HasCustomIndexSetter"}) {
                push(@implContent, "void ${className}::put(ExecState* exec, unsigned propertyName, JSValue* value)\n");
                push(@implContent, "{\n");
                push(@implContent, "    indexSetter(exec, propertyName, value);\n");
                push(@implContent, "    return;\n");
                push(@implContent, "}\n\n");
            }

            if ($hasReadWriteProperties) {
                foreach my $attribute (@{$dataNode->attributes}) {
                    if ($attribute->type !~ /^readonly/) {
                        my $name = $attribute->signature->name;
                        my $type = $codeGenerator->StripModule($attribute->signature->type);
                        my $putFunctionName = "setJS" . $interfaceName .  $codeGenerator->WK_ucfirst($name) . ($attribute->signature->type =~ /Constructor$/ ? "Constructor" : "");
                        my $implSetterFunctionName = $codeGenerator->WK_ucfirst($name);

                        push(@implContent, "void ${putFunctionName}(ExecState* exec, JSObject* thisObject, JSValue* value)\n");
                        push(@implContent, "{\n");

                        if ($dataNode->extendedAttributes->{"CheckDomainSecurity"} && !$attribute->signature->extendedAttributes->{"DoNotCheckDomainSecurity"}) {
                            if ($interfaceName eq "DOMWindow") {
                                push(@implContent, "    if (!static_cast<$className*>(thisObject)->allowsAccessFrom(exec))\n");
                            } else {
                                push(@implContent, "    if (!allowsAccessFromFrame(exec, static_cast<$className*>(thisObject)->impl()->frame()))\n");
                            }
                            push(@implContent, "        return;\n");
                        }

                        if ($attribute->signature->extendedAttributes->{"Custom"} || $attribute->signature->extendedAttributes->{"CustomSetter"}) {
                            push(@implContent, "    static_cast<$className*>(thisObject)->set$implSetterFunctionName(exec, value);\n");
                        } elsif ($type eq "EventListener") {
                            $implIncludes{"JSEventListener.h"} = 1;
                            push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(thisObject)->impl());\n");
                            my $listenerType;
                            if ($attribute->signature->extendedAttributes->{"ProtectedListener"}) {
                                $listenerType = "JSEventListener";
                            } else {
                                $listenerType = "JSUnprotectedEventListener";
                            }
                            if ($interfaceName eq "DOMWindow") {
                                push(@implContent, "    JSDOMWindow* window = static_cast<JSDOMWindow*>(thisObject);\n");
                            } else {
                                $implIncludes{"Frame.h"} = 1;
                                $implIncludes{"JSDOMWindow.h"} = 1;
                                push(@implContent, "    Frame* frame = imp->associatedFrame();\n");
                                push(@implContent, "    if (!frame)\n");
                                push(@implContent, "        return;\n");
                                push(@implContent, "    JSDOMWindow* window = toJSDOMWindow(frame);\n");
                            }
                            push(@implContent, "    imp->set$implSetterFunctionName(window->findOrCreate${listenerType}(exec, value, true));\n");
                        } elsif ($attribute->signature->type =~ /Constructor$/) {
                            my $constructorType = $attribute->signature->type;
                            $constructorType =~ s/Constructor$//;
                            $implIncludes{"JS" . $constructorType . ".h"} = 1;
                            push(@implContent, "    // Shadowing a built-in constructor\n");
                            push(@implContent, "    static_cast<$className*>(thisObject)->putDirect(Identifier(exec, \"$name\"), value);\n");
                        } elsif ($attribute->signature->extendedAttributes->{"Replaceable"}) {
                            push(@implContent, "    static_cast<$className*>(thisObject)->putDirect(Identifier(exec, \"$name\"), value);\n");
                        } else {
                            if ($podType) {
                                push(@implContent, "    $podType imp(*static_cast<$className*>(thisObject)->impl());\n");
                                if ($podType eq "float") { # Special case for JSSVGNumber
                                    push(@implContent, "    imp = " . JSValueToNative($attribute->signature, "value") . ";\n");
                                } else {
                                    push(@implContent, "    imp.set$implSetterFunctionName(" . JSValueToNative($attribute->signature, "value") . ");\n");
                                }
                                push(@implContent, "        static_cast<$className*>(thisObject)->impl()->commitChange(imp, static_cast<$className*>(thisObject)->context());\n");
                            } else {
                                push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(static_cast<$className*>(thisObject)->impl());\n");
                                push(@implContent, "    ExceptionCode ec = 0;\n") if @{$attribute->setterExceptions};
                                push(@implContent, "    imp->set$implSetterFunctionName(" . JSValueToNative($attribute->signature, "value"));
                                push(@implContent, ", ec") if @{$attribute->setterExceptions};
                                push(@implContent, ");\n");
                                push(@implContent, "    setDOMException(exec, ec);\n") if @{$attribute->setterExceptions};

                                if (IsSVGTypeNeedingContextParameter($implClassName)) {
                                    push(@implContent, "    if (static_cast<$className*>(thisObject)->context())\n");
                                    push(@implContent, "        static_cast<$className*>(thisObject)->context()->svgAttributeChanged(static_cast<$className*>(thisObject)->impl()->associatedAttributeName());\n");
                                }
                            }
                        }
                        
                        push(@implContent, "}\n\n");
                    }
                }
            }
        }
    }

    if ($dataNode->extendedAttributes->{"CustomGetPropertyNames"} || $dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}) {
        push(@implContent, "void ${className}::getPropertyNames(ExecState* exec, PropertyNameArray& propertyNames)\n");
        push(@implContent, "{\n");
        if ($dataNode->extendedAttributes->{"CustomGetPropertyNames"}) {
            push(@implContent, "    if (customGetPropertyNames(exec, propertyNames))\n");
            push(@implContent, "        return;\n");
        }
        if ($dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasCustomIndexGetter"}) {
            push(@implContent, "    for (unsigned i = 0; i < static_cast<${implClassName}*>(impl())->length(); ++i)\n");
            push(@implContent, "        propertyNames.add(Identifier::from(exec, i));\n");
        }
        push(@implContent, "     Base::getPropertyNames(exec, propertyNames);\n");
        push(@implContent, "}\n\n");
    }

    if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
        push(@implContent, "JSValue* ${className}::getConstructor(ExecState* exec)\n{\n");
        push(@implContent, "    return getDOMConstructor<${className}Constructor>(exec);\n");
        push(@implContent, "}\n\n");
    }

    # Functions
    if ($numFunctions > 0) {
        foreach my $function (@{$dataNode->functions}) {
            AddIncludesForType($function->signature->type);

            my $functionName = $codeGenerator->WK_lcfirst($className) . "PrototypeFunction" . $codeGenerator->WK_ucfirst($function->signature->name);
            my $functionImplementationName = $function->signature->extendedAttributes->{"ImplementationFunction"} || $codeGenerator->WK_lcfirst($function->signature->name);

            push(@implContent, "JSValue* ${functionName}(ExecState* exec, JSObject*, JSValue* thisValue, const ArgList& args)\n");
            push(@implContent, "{\n");

            $implKJSInclude{"Error.h"} = 1;

            if ($interfaceName eq "DOMWindow") {
                push(@implContent, "    $className* castedThisObj = toJSDOMWindow(thisValue);\n");
                push(@implContent, "    if (!castedThisObj)\n");
                push(@implContent, "        return throwError(exec, TypeError);\n");
            } else {
                push(@implContent, "    if (!thisValue->isObject(&${className}::s_info))\n");
                push(@implContent, "        return throwError(exec, TypeError);\n");
                push(@implContent, "    $className* castedThisObj = static_cast<$className*>(thisValue);\n");
            }

            if ($dataNode->extendedAttributes->{"CheckDomainSecurity"} && 
                !$function->signature->extendedAttributes->{"DoNotCheckDomainSecurity"}) {
                push(@implContent, "    if (!castedThisObj->allowsAccessFrom(exec))\n");
                push(@implContent, "        return jsUndefined();\n");
            }

            if ($function->signature->extendedAttributes->{"Custom"}) {
                push(@implContent, "    return castedThisObj->" . $functionImplementationName . "(exec, args);\n");
            } else {
                if ($podType) {
                    push(@implContent, "    JSSVGPODTypeWrapper<$podType>* wrapper = castedThisObj->impl();\n");
                    push(@implContent, "    $podType imp(*wrapper);\n");
                } else {
                    push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(castedThisObj->impl());\n");
                }

                my $numParameters = @{$function->parameters};

                if ($function->signature->extendedAttributes->{"RequiresAllArguments"}) {
                        push(@implContent, "    if (args.size() < $numParameters)\n");
                        push(@implContent, "        return jsUndefined();\n");
                }

                if (@{$function->raisesExceptions}) {
                    push(@implContent, "    ExceptionCode ec = 0;\n");
                }

                if ($function->signature->extendedAttributes->{"SVGCheckSecurityDocument"}) {
                    push(@implContent, "    if (!checkNodeSecurity(exec, imp->getSVGDocument(" . (@{$function->raisesExceptions} ? "ec" : "") .")))\n");
                    push(@implContent, "        return jsUndefined();\n");
                    $implIncludes{"JSDOMBinding.h"} = 1;
                }

                my $paramIndex = 0;
                my $functionString = "imp" . ($podType ? "." : "->") . $functionImplementationName . "(";

                my $hasOptionalArguments = 0;

                foreach my $parameter (@{$function->parameters}) {
                    if (!$hasOptionalArguments && $parameter->extendedAttributes->{"Optional"}) {
                        push(@implContent, "\n    int argsCount = args.size();\n");
                        $hasOptionalArguments = 1;
                    }

                    if ($hasOptionalArguments) {
                        push(@implContent, "    if (argsCount < " . ($paramIndex + 1) . ") {\n");
                        GenerateImplementationFunctionCall($function, $functionString, $paramIndex, "    " x 2, $podType, $implClassName);
                        push(@implContent, "    }\n\n");
                    }

                    my $name = $parameter->name;
                    
                    if ($parameter->type eq "XPathNSResolver") {
                        push(@implContent, "    RefPtr<XPathNSResolver> customResolver;\n");
                        push(@implContent, "    XPathNSResolver* resolver = toXPathNSResolver(args.at(exec, $paramIndex));\n");
                        push(@implContent, "    if (!resolver) {\n");
                        push(@implContent, "        customResolver = JSCustomXPathNSResolver::create(exec, args.at(exec, $paramIndex));\n");
                        push(@implContent, "        if (exec->hadException())\n");
                        push(@implContent, "            return jsUndefined();\n");
                        push(@implContent, "        resolver = customResolver.get();\n");
                        push(@implContent, "    }\n");
                    } else {
                        push(@implContent, "    " . GetNativeTypeFromSignature($parameter) . " $name = " . JSValueToNative($parameter, "args.at(exec, $paramIndex)") . ";\n");

                        # If a parameter is "an index" and it's negative it should throw an INDEX_SIZE_ERR exception.
                        # But this needs to be done in the bindings, because the type is unsigned and the fact that it
                        # was negative will be lost by the time we're inside the DOM.
                        if ($parameter->extendedAttributes->{"IsIndex"}) {
                            $implIncludes{"ExceptionCode.h"} = 1;
                            push(@implContent, "    if ($name < 0) {\n");
                            push(@implContent, "        setDOMException(exec, INDEX_SIZE_ERR);\n");
                            push(@implContent, "        return jsUndefined();\n");
                            push(@implContent, "    }\n");
                        }
                    }

                    $functionString .= ", " if $paramIndex;

                    if ($parameter->type eq "NodeFilter") {
                        $functionString .= "$name.get()";
                    } else {
                        $functionString .= $name;
                    }

                    $paramIndex++;
                }

                push(@implContent, "\n");
                GenerateImplementationFunctionCall($function, $functionString, $paramIndex, "    ", $podType, $implClassName);
            }
            push(@implContent, "}\n\n");
        }
    }

    if ($numConstants > 0) {
        push(@implContent, "// Constant getters\n\n");

        foreach my $constant (@{$dataNode->constants}) {
            my $getter = "js" . $interfaceName . $codeGenerator->WK_ucfirst($constant->name);

            # FIXME: this casts into int to match our previous behavior which turned 0xFFFFFFFF in -1 for NodeFilter.SHOW_ALL
            push(@implContent, "JSValue* ${getter}(ExecState* exec, const Identifier&, const PropertySlot&)\n");
            push(@implContent, "{\n");
            push(@implContent, "    return jsNumber(exec, static_cast<int>(" . $constant->value . "));\n");
            push(@implContent, "}\n\n");
        }
    }

    if ($dataNode->extendedAttributes->{"HasIndexGetter"}) {
        push(@implContent, "\nJSValue* ${className}::indexGetter(ExecState* exec, const Identifier& propertyName, const PropertySlot& slot)\n");
        push(@implContent, "{\n");
        push(@implContent, "    ${className}* thisObj = static_cast<$className*>(slot.slotBase());\n");
        if (IndexGetterReturnsStrings($implClassName)) {
            $implIncludes{"KURL.h"} = 1;
            push(@implContent, "    return jsStringOrNull(exec, thisObj->impl()->item(slot.index()));\n");
        } else {
            push(@implContent, "    return toJS(exec, static_cast<$implClassName*>(thisObj->impl())->item(slot.index()));\n");
        }
        push(@implContent, "}\n");
        if ($interfaceName eq "HTMLCollection") {
            $implIncludes{"JSNode.h"} = 1;
            $implIncludes{"Node.h"} = 1;
        }
    }

    if ((!$hasParent or $dataNode->extendedAttributes->{"GenerateToJS"}) and !UsesManualToJSImplementation($implClassName)) {
        if ($podType) {
            push(@implContent, "JSC::JSValue* toJS(JSC::ExecState* exec, JSSVGPODTypeWrapper<$podType>* object, SVGElement* context)\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
             push(@implContent, "JSC::JSValue* toJS(JSC::ExecState* exec, $implType* object, SVGElement* context)\n");
        } else {
            push(@implContent, "JSC::JSValue* toJS(JSC::ExecState* exec, $implType* object)\n");
        }

        push(@implContent, "{\n");
        if ($podType) {
            push(@implContent, "    return getDOMObjectWrapper<$className, JSSVGPODTypeWrapper<$podType> >(exec, object, context);\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@implContent, "    return getDOMObjectWrapper<$className>(exec, object, context);\n");
        } else {
            push(@implContent, "    return getDOMObjectWrapper<$className>(exec, object);\n");
        }
        push(@implContent, "}\n");
    }

    if ((!$hasParent or $dataNode->extendedAttributes->{"GenerateNativeConverter"}) and !$dataNode->extendedAttributes->{"CustomNativeConverter"}) {
        if ($podType) {
            push(@implContent, "$podType to${interfaceName}(JSC::JSValue* val)\n");
        } else {
            push(@implContent, "$implClassName* to${interfaceName}(JSC::JSValue* val)\n");
        }

        push(@implContent, "{\n");

        push(@implContent, "    return val->isObject(&${className}::s_info) ? " . ($podType ? "($podType) *" : "") . "static_cast<$className*>(val)->impl() : ");
        if ($podType and $podType ne "float") {
            push(@implContent, "$podType();\n}\n");
        } else {
            push(@implContent, "0;\n}\n");
        }
    }

    push(@implContent, "\n}\n");

    push(@implContent, "\n#endif // ${conditionalString}\n") if $conditional;
}

sub GenerateImplementationFunctionCall()
{
    my $function = shift;
    my $functionString = shift;
    my $paramIndex = shift;
    my $indent = shift;
    my $podType = shift;
    my $implClassName = shift;

    if (@{$function->raisesExceptions}) {
        $functionString .= ", " if $paramIndex;
        $functionString .= "ec";
    }
    $functionString .= ")";

    if ($function->signature->type eq "void") {
        push(@implContent, $indent . "$functionString;\n");
        push(@implContent, $indent . "setDOMException(exec, ec);\n") if @{$function->raisesExceptions};

        if ($podType) {
            push(@implContent, $indent . "wrapper->commitChange(imp, castedThisObj->context());\n");
        }

        push(@implContent, $indent . "return jsUndefined();\n");
    } else {
        push(@implContent, "\n" . $indent . "JSC::JSValue* result = " . NativeToJSValue($function->signature, 1, $implClassName, "", $functionString, "castedThisObj") . ";\n");
        push(@implContent, $indent . "setDOMException(exec, ec);\n") if @{$function->raisesExceptions};

        if ($podType) {
            push(@implContent, $indent . "wrapper->commitChange(imp, castedThisObj->context());\n");
        }

        push(@implContent, $indent . "return result;\n");
    }
}

sub GetNativeTypeFromSignature
{
    my $signature = shift;
    my $type = $codeGenerator->StripModule($signature->type);

    if ($type eq "unsigned long" and $signature->extendedAttributes->{"IsIndex"}) {
        # Special-case index arguments because we need to check that they aren't < 0.
        return "int";
    }

    return GetNativeType($type);
}

my %nativeType = (
    "CompareHow" => "Range::CompareHow",
    "DOMString" => "const UString&",
    "EventTarget" => "EventTargetNode*",
    "NodeFilter" => "RefPtr<NodeFilter>",
    "SVGLength" => "SVGLength",
    "SVGMatrix" => "AffineTransform",
    "SVGNumber" => "float",
    "SVGPaintType" => "SVGPaint::SVGPaintType",
    "SVGPoint" => "FloatPoint",
    "SVGRect" => "FloatRect",
    "SVGTransform" => "SVGTransform",
    "boolean" => "bool",
    "double" => "double",
    "float" => "float",
    "long" => "int",
    "unsigned long" => "unsigned",
    "unsigned short" => "unsigned short",
);

sub GetNativeType
{
    my $type = shift;

    return $nativeType{$type} if exists $nativeType{$type};

    # For all other types, the native type is a pointer with same type name as the IDL type.
    return "${type}*";
}

sub JSValueToNative
{
    my $signature = shift;
    my $value = shift;

    my $type = $codeGenerator->StripModule($signature->type);

    return "$value->toBoolean(exec)" if $type eq "boolean";
    return "$value->toNumber(exec)" if $type eq "double";
    return "$value->toFloat(exec)" if $type eq "float" or $type eq "SVGNumber";
    return "$value->toInt32(exec)" if $type eq "unsigned long" or $type eq "long" or $type eq "unsigned short";

    return "static_cast<Range::CompareHow>($value->toInt32(exec))" if $type eq "CompareHow";
    return "static_cast<SVGPaint::SVGPaintType>($value->toInt32(exec))" if $type eq "SVGPaintType";

    if ($type eq "DOMString") {
        return "valueToStringWithNullCheck(exec, $value)" if $signature->extendedAttributes->{"ConvertNullToNullString"};
        return "valueToStringWithUndefinedOrNullCheck(exec, $value)" if $signature->extendedAttributes->{"ConvertUndefinedOrNullToNullString"};
        return "$value->toString(exec)";
    }

    if ($type eq "EventTarget") {
        $implIncludes{"JSEventTargetNode.h"} = 1;
        return "toEventTargetNode($value)";
    }

    $implIncludes{"FloatPoint.h"} = 1 if $type eq "SVGPoint";
    $implIncludes{"FloatRect.h"} = 1 if $type eq "SVGRect";
    $implIncludes{"HTMLOptionElement.h"} = 1 if $type eq "HTMLOptionElement";
    $implIncludes{"JSCustomVoidCallback.h"} = 1 if $type eq "VoidCallback";
    $implIncludes{"Event.h"} = 1 if $type eq "Event";

    # Default, assume autogenerated type conversion routines
    $implIncludes{"JS$type.h"} = 1;
    return "to$type($value)";
}

sub NativeToJSValue
{
    my $signature = shift;
    my $inFunctionCall = shift;
    my $implClassName = shift;
    my $implClassNameForValueConversion = shift;
    my $value = shift;
    my $thisValue = shift;

    my $type = $codeGenerator->StripModule($signature->type);

    return "jsBoolean($value)" if $type eq "boolean";
    
    if ($codeGenerator->IsPrimitiveType($type) or $type eq "SVGPaintType" or $type eq "DOMTimeStamp") {
        $implKJSInclude{"JSNumberCell.h"} = 1;
        return "jsNumber(exec, $value)";
    }

    if ($codeGenerator->IsStringType($type)) {
        $implIncludes{"KURL.h"} = 1;
        my $conv = $signature->extendedAttributes->{"ConvertNullStringTo"};
        if (defined $conv) {
            return "jsStringOrNull(exec, $value)" if $conv eq "Null";
            return "jsStringOrUndefined(exec, $value)" if $conv eq "Undefined";
            return "jsStringOrFalse(exec, $value)" if $conv eq "False";

            die "Unknown value for ConvertNullStringTo extended attribute";
        }
        $implKJSInclude{"JSString.h"} = 1;
        return "jsString(exec, $value)";
    }

    if ($type eq "RGBColor") {
        $implIncludes{"JS$type.h"} = 1;
        return "getJSRGBColor(exec, $value)";
    }

    if ($codeGenerator->IsPodType($type)) {
        $implIncludes{"JS$type.h"} = 1;

        my $nativeType = GetNativeType($type);

        my $getter = $value;
        $getter =~ s/imp->//;
        $getter =~ s/\(\)//;

        my $setter = "set" . $codeGenerator->WK_ucfirst($getter);

        # Function calls will never return 'modifyable' POD types (ie. SVGRect getBBox()) - no need to keep track changes to the returned SVGRect
        if ($inFunctionCall eq 0
            and not $codeGenerator->IsSVGAnimatedType($implClassName)
            and $codeGenerator->IsPodTypeWithWriteableProperties($type)
            and not defined $signature->extendedAttributes->{"Immutable"}) {
            if ($codeGenerator->IsPodType($implClassName)) {
                return "toJS(exec, JSSVGStaticPODTypeWrapperWithPODTypeParent<$nativeType, $implClassName>::create($value, $thisValue->impl()).get(), $thisValue->context())";
            } else {
                return "toJS(exec, JSSVGStaticPODTypeWrapperWithParent<$nativeType, $implClassName>::create(imp, &${implClassName}::$getter, &${implClassName}::$setter).get(), imp)";
            }
        }

        if ($implClassNameForValueConversion eq "") {
            if (IsSVGTypeNeedingContextParameter($implClassName)) {
                return "toJS(exec, JSSVGStaticPODTypeWrapper<$nativeType>::create($value).get(), castedThisObj->context())" if $inFunctionCall eq 1;

                # Special case: SVGZoomEvent - it doesn't have a context, but it's no problem, as there are no readwrite props
                return "toJS(exec, JSSVGStaticPODTypeWrapper<$nativeType>::create($value).get(), 0)" if $implClassName eq "SVGZoomEvent";
                return "toJS(exec, JSSVGStaticPODTypeWrapper<$nativeType>::create($value).get(), $thisValue->context())";
            } else {
                return "toJS(exec, JSSVGStaticPODTypeWrapper<$nativeType>::create($value).get(), imp)";
            }
        } else { # These classes, always have a m_context pointer!
            return "toJS(exec, JSSVGDynamicPODTypeWrapperCache<$nativeType, $implClassNameForValueConversion>::lookupOrCreateWrapper(imp, &${implClassNameForValueConversion}::$getter, &${implClassNameForValueConversion}::$setter).get(), $thisValue->context())";
        }
    }

    if ($codeGenerator->IsSVGAnimatedType($type)) {
        $value =~ s/\(\)//;
        $value .= "Animated()";
    }

    if ($type eq "CSSStyleDeclaration") {
        $implIncludes{"CSSMutableStyleDeclaration.h"} = 1;
    }

    if ($type eq "NamedNodeMap") {
        $implIncludes{"NamedAttrMap.h"} = 1;
    }

    if ($type eq "NodeList") {
        $implIncludes{"NameNodeList.h"} = 1;
    }

    if ($type eq "SVGElementInstance") {
        $implIncludes{"EventTargetSVGElementInstance.h"} = 1;
        $implIncludes{"JSEventTargetSVGElementInstance.h"} = 1;
    } elsif ($type eq "DOMObject") {
        $implIncludes{"JSCanvasRenderingContext2D.h"} = 1;
    } elsif ($type =~ /SVGPathSeg/) {
        $implIncludes{"JS$type.h"} = 1;
        $joinedName = $type;
        $joinedName =~ s/Abs|Rel//;
        $implIncludes{"$joinedName.h"} = 1;
    } else {
        # Default, include header with same name.
        $implIncludes{"JS$type.h"} = 1;
        $implIncludes{"$type.h"} = 1;
    }

    return $value if $codeGenerator->IsSVGAnimatedType($type);

    if (IsSVGTypeNeedingContextParameter($type)) {
        if (IsSVGTypeNeedingContextParameter($implClassName)) {
            return "toJS(exec, WTF::getPtr($value), $thisValue->context())";
        } else {
            return "toJS(exec, WTF::getPtr($value), imp)";
        }
    }

    if ($signature->extendedAttributes->{"ReturnsNew"}) {        
        return "toJSNewlyCreated(exec, WTF::getPtr($value))";
    }

    return "toJS(exec, WTF::getPtr($value))";
}

sub ceilingToPowerOf2
{
    my ($size) = @_;

    my $powerOf2 = 1;
    while ($size > $powerOf2) {
        $powerOf2 <<= 1;
    }

    return $powerOf2;
}

# Internal Helper
sub GenerateHashTable
{
    my $object = shift;

    my $name = shift;
    my $keys = shift;
    my $specials = shift;
    my $value1 = shift;
    my $value2 = shift;
    my $conditionals = shift;

    my @hashes = ();
    foreach my $key (@{$keys}) {
        push @hashes, $object->GenerateHashValue($key);
    }

    # Collect hashtable information
    my $size;
tableSizeLoop:
    for ($size = ceilingToPowerOf2(scalar @{$keys}); ; $size += $size) {
        my @table = ();
        my $i = 0;
        foreach my $hash (@hashes) {
            my $h = $hash % $size;
            next tableSizeLoop if defined $table[$h];
            $table[$h] = $i++;
        }
        last;
    }

    # Start outputing the hashtables
    my $nameEntries = "${name}Values";
    $nameEntries =~ s/:/_/g;

    if (($name =~ /Prototype/) or ($name =~ /Constructor/)) {
        my $type = $name;
        my $implClass;

        if ($name =~ /Prototype/) {
            $type =~ s/Prototype.*//;
            $implClass = $type; $implClass =~ s/Wrapper$//;
            push(@implContent, "/* Hash table for prototype */\n");
        } else {
            $type =~ s/Constructor.*//;
            $implClass = $type; $implClass =~ s/Constructor$//;
            push(@implContent, "/* Hash table for constructor */\n");
        }
    } else {
        push(@implContent, "/* Hash table */\n");
    }

    # Dump the hash table
    my $count = scalar @{$keys} + 1;
    push(@implContent, "\nstatic const HashTableValue $nameEntries\[$count\] =\n\{\n");
    my $i = 0;
    foreach my $key (@{$keys}) {
        my $conditional;

        if ($conditionals) {
            $conditional = $conditionals->{$key};
        }
        if ($conditional) {
            my $conditionalString = "ENABLE(" . join(") && ENABLE(", split(/&/, $conditional)) . ")";
            push(@implContent, "#if ${conditionalString}\n");
        }
        push(@implContent, "    { \"$key\", @$specials[$i], (intptr_t)@$value1[$i], (intptr_t)@$value2[$i] },\n");
        if ($conditional) {
            push(@implContent, "#endif\n");
        }
        ++$i;
    }
    push(@implContent, "    { 0, 0, 0, 0 }\n");
    push(@implContent, "};\n\n");
    my $sizeMask = $size - 1;
    push(@implContent, "static const HashTable $name = { $sizeMask, $nameEntries, 0 };\n\n");
}

# Internal helper
sub GenerateHashValue
{
    my $object = shift;

    @chars = split(/ */, $_[0]);

    # This hash is designed to work on 16-bit chunks at a time. But since the normal case
    # (above) is to hash UTF-16 characters, we just treat the 8-bit chars as if they
    # were 16-bit chunks, which should give matching results

    my $EXP2_32 = 4294967296;

    my $hash = 0x9e3779b9;
    my $l    = scalar @chars; #I wish this was in Ruby --- Maks
    my $rem  = $l & 1;
    $l = $l >> 1;

    my $s = 0;

    # Main loop
    for (; $l > 0; $l--) {
        $hash   += ord($chars[$s]);
        my $tmp = leftShift(ord($chars[$s+1]), 11) ^ $hash;
        $hash   = (leftShift($hash, 16)% $EXP2_32) ^ $tmp;
        $s += 2;
        $hash += $hash >> 11;
        $hash %= $EXP2_32;
    }

    # Handle end case
    if ($rem != 0) {
        $hash += ord($chars[$s]);
        $hash ^= (leftShift($hash, 11)% $EXP2_32);
        $hash += $hash >> 17;
    }

    # Force "avalanching" of final 127 bits
    $hash ^= leftShift($hash, 3);
    $hash += ($hash >> 5);
    $hash = ($hash% $EXP2_32);
    $hash ^= (leftShift($hash, 2)% $EXP2_32);
    $hash += ($hash >> 15);
    $hash = $hash% $EXP2_32;
    $hash ^= (leftShift($hash, 10)% $EXP2_32);

    # this avoids ever returning a hash code of 0, since that is used to
    # signal "hash not computed yet", using a value that is likely to be
    # effectively the same as 0 when the low bits are masked
    $hash = 0x80000000 if ($hash == 0);

    return $hash;
}

# Internal helper
sub WriteData
{
    if (defined($IMPL)) {
        # Write content to file.
        print $IMPL @implContentHeader;

        foreach my $implInclude (sort keys(%implIncludes)) {
            my $checkType = $implInclude;
            $checkType =~ s/\.h//;

            print $IMPL "#include \"$implInclude\"\n" unless $codeGenerator->IsSVGAnimatedType($checkType);
        }
        
        print $IMPL "\n";

        foreach my $implKJSInclude (sort keys(%implKJSInclude)) {
            print $IMPL "#include <kjs/$implKJSInclude>\n";
        }

        print $IMPL @implContent;
        close($IMPL);
        undef($IMPL);

        @implContentHeader = ();
        @implContent = ();
        %implIncludes = ();
        %implKJSIncludes = ();
    }

    if (defined($HEADER)) {
        # Write content to file.
        print $HEADER @headerContentHeader;

        foreach my $headerInclude (sort keys(%headerIncludes)) {
            print $HEADER "#include \"$headerInclude\"\n";
        }

        print $HEADER @headerContent;
        close($HEADER);
        undef($HEADER);

        @headerContentHeader = ();
        @headerContent = ();
        %headerIncludes = ();
    }
}

sub constructorFor
{
    my $className = shift;
    my $protoClassName = shift;
    my $interfaceName = shift;
    my $visibleClassName = shift;
    my $canConstruct = shift;

my $implContent = << "EOF";
class ${className}Constructor : public DOMObject {
public:
    ${className}Constructor(ExecState* exec)
        : DOMObject(${className}Constructor::createStructureID(exec->lexicalGlobalObject()->objectPrototype()))
    {
        putDirect(exec->propertyNames().prototype, ${protoClassName}::self(exec), None);
    }
    virtual bool getOwnPropertySlot(ExecState*, const Identifier&, PropertySlot&);
    virtual const ClassInfo* classInfo() const { return &s_info; }
    static const ClassInfo s_info;

    static PassRefPtr<StructureID> createStructureID(JSValue* proto) 
    { 
        return StructureID::create(proto, TypeInfo(ObjectType, ImplementsHasInstance)); 
    }
EOF

    if ($canConstruct) {
$implContent .= << "EOF";
    static JSObject* construct(ExecState* exec, JSObject*, const ArgList&)
    {
        return static_cast<JSObject*>(toJS(exec, ${interfaceName}::create()));
    }
    virtual ConstructType getConstructData(ConstructData& constructData)
    {
        constructData.native.function = construct;
        return ConstructTypeHost;
    }
EOF
    }

$implContent .= << "EOF";
};

const ClassInfo ${className}Constructor::s_info = { "${visibleClassName}Constructor", 0, &${className}ConstructorTable, 0 };

bool ${className}Constructor::getOwnPropertySlot(ExecState* exec, const Identifier& propertyName, PropertySlot& slot)
{
    return getStaticValueSlot<${className}Constructor, DOMObject>(exec, &${className}ConstructorTable, this, propertyName, slot);
}

EOF

    $implKJSInclude{"JSNumberCell.h"} = 1;

    return $implContent;
}

1;
