/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the example classes of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the either Technology Preview License Agreement or the
** Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include <QtGui>

#include "filterexample.h"

static const QSize resultSize(150, 150);

QPoint pixmapPos(const QPixmap &pixmap)
{
    return QPoint((resultSize.width() - pixmap.width()) / 2,
                  (resultSize.height() - pixmap.height()) / 2);
}

//![0]
FilterWidget::FilterWidget(const QPixmap &background, const QPixmap &icon, const Ui::ValueControls &control)
{
    backgroundPixmap = background;
    sourcePixmap = icon;
    controls =  control;
    type = QPixmapFilter::ConvolutionFilter;
}
//![0]

void FilterWidget::setType(QPixmapFilter::FilterType filterType)
{
    type = filterType;
}

void FilterWidget::setType(int filterType)
{
    setType((QPixmapFilter::FilterType)filterType);
    update();
}

//![1]
void FilterWidget::paintEvent(QPaintEvent* /*event*/)
{
    QPainter painter(this);
    painter.setClipRect(QRect(0, 0, resultSize.width(), resultSize.height()));
    painter.drawPixmap(0, 0, backgroundPixmap);
    QPixmapFilter *currentFilter = setupFilter(type);
    currentFilter->draw(&painter, pixmapPos(sourcePixmap), sourcePixmap);
    delete currentFilter;
    painter.end();
}
//![1]

//![2]
QPixmapFilter* FilterWidget::setupFilter(QPixmapFilter::FilterType type)
{
    controls.ConvolutionControlWidget->hide();
    controls.ColorizeControlWidget->hide();
    controls.DropShadowControlWidget->hide();

    QPixmapFilter *filter = 0;
//![2]
//![3]
    qreal kernel[] = {
        controls.kernel_1x1->text().toFloat(),
        controls.kernel_2x1->text().toFloat(),
        controls.kernel_3x1->text().toFloat(),
        controls.kernel_1x2->text().toFloat(),
        controls.kernel_2x2->text().toFloat(),
        controls.kernel_3x2->text().toFloat(),
        controls.kernel_1x3->text().toFloat(),
        controls.kernel_2x3->text().toFloat(),
        controls.kernel_3x3->text().toFloat()
    };
//![3]

//![4]
    switch(type) {

    case QPixmapFilter::ConvolutionFilter:
        controls.ConvolutionControlWidget->show();
        filter = new QPixmapConvolutionFilter();
        reinterpret_cast<QPixmapConvolutionFilter*>(filter)->setConvolutionKernel(kernel, 3,3);
        break;
//![4]
//![5]
    case QPixmapFilter::ColorizeFilter: {
        controls.ColorizeControlWidget->show();
        filter = new QPixmapColorizeFilter();
        QColor color = QColor(controls.colorizeRedSlider->value(),
                       controls.colorizeGreenSlider->value(),
                       controls.colorizeBlueSlider->value());
        reinterpret_cast<QPixmapColorizeFilter*>(filter)->setColor(color);
        break;
    }
//![5]

//![6]
    case QPixmapFilter::DropShadowFilter: {
        controls.DropShadowControlWidget->show();
        filter = new QPixmapDropShadowFilter();
        QColor color = QColor(controls.dropShadowRedSlider->value(),
                       controls.dropShadowGreenSlider->value(),
                       controls.dropShadowBlueSlider->value(),
                       controls.dropShadowAlphaSlider->value());
        reinterpret_cast<QPixmapDropShadowFilter*>(filter)->setColor(color);
        reinterpret_cast<QPixmapDropShadowFilter*>(filter)->setBlurRadius((qreal)(controls.dropShadowRadiusSlider->value())/20.0);
        reinterpret_cast<QPixmapDropShadowFilter*>(filter)->setOffset(
                QPointF((qreal)(controls.dropShadowXSlider->value()-500)/10.0,
                        (qreal)(controls.dropShadowYSlider->value()-500)/10.0
                        ));
        break;
    }
//![6]
    default:
        break;
    }

//![7]
    return filter;
}
//![7]

FilterExample::FilterExample()
{
    controlWidget = new QWidget();
    controls.setupUi(controlWidget);

    sourceLabel = new QLabel;
    sourceLabel->setMinimumWidth(resultSize.width());

    filterComboBox = new QComboBox;
    filterComboBox->setFocusPolicy(Qt::NoFocus);

    filterComboBox->addItem(tr("Convolution Filter"), QPixmapFilter::ConvolutionFilter);
    filterComboBox->addItem(tr("Colorize Filter"), QPixmapFilter::ColorizeFilter);
    filterComboBox->addItem(tr("Drop Shadow Filter"), QPixmapFilter::DropShadowFilter);

    backgroundPixmap = QPixmap(":/images/checker.png");
    sourcePixmap = QPixmap(":/images/qt-logo.png");

    QImage fixedImage(resultSize, QImage::Format_ARGB32_Premultiplied);
    QPainter painter(&fixedImage);
    painter.drawPixmap(pixmapPos(backgroundPixmap), backgroundPixmap);
    painter.drawPixmap(pixmapPos(sourcePixmap), sourcePixmap);
    painter.end();
    sourceLabel->setPixmap(QPixmap::fromImage(fixedImage));

    resultWidget = new FilterWidget(backgroundPixmap, sourcePixmap, controls);
    resultWidget->setMinimumWidth(resultSize.width());
    resultWidget->setMinimumHeight(resultSize.height());

    connect(filterComboBox, SIGNAL(activated(int)),
            resultWidget, SLOT(setType(int)));

    connect(controls.colorizeRedSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.colorizeGreenSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.colorizeBlueSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowRedSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowGreenSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowBlueSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowAlphaSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowXSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowYSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.dropShadowRadiusSlider, SIGNAL(valueChanged(int)), resultWidget, SLOT(update()));
    connect(controls.kernel_1x1, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_2x1, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_3x1, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_1x2, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_2x2, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_3x2, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_1x3, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_2x3, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));
    connect(controls.kernel_3x3, SIGNAL(textChanged(QString)), resultWidget, SLOT(update()));

    QGridLayout *mainLayout = new QGridLayout;
    mainLayout->addWidget(sourceLabel, 0, 0, 3, 1);
    mainLayout->addWidget(filterComboBox, 1, 1);
    mainLayout->addWidget(resultWidget, 0, 2, 3, 1);
    mainLayout->addWidget(controlWidget, 3, 0, 1, 3);
    mainLayout->setSizeConstraint(QLayout::SetFixedSize);
    setLayout(mainLayout);

    setWindowTitle(tr("Pixmap Filters"));
}
